/**
 ****************************************************************************************
 *
 * @file rwnx_baws.h
 *
 * Copyright (C) RivieraWaves 2012-2019
 *
 ****************************************************************************************
 */
#ifndef _RWNX_BAWS_H_
#define _RWNX_BAWS_H_

#include <net/mac80211.h>

/**
 * Enum listing the possible states of a BAW element
 */
enum {
    BAW_FREE,
    BAW_PENDING,
    BAW_CONFIRMED,
};

#ifdef CONFIG_RWNX_DBG_OOBAWS
struct rwnx_baw_dbg_states {
    struct rwnx_baw_dbg_state {
        u32 tx_flags;
        u16 fsn;
        u16 sn;
        u8 buf_size;
        u8 ba_idx;
    } states[16];
    int idx;
};
#endif

/**
 * struct rwnx_baw_status - Transmission status of an AMPDU
 *
 * @total_cnt: Number of MDPU in the AMPDU
 * @ok_cnt: Number of MPDU acknowledged
 * @rate: Rate used for the transmission
 */
struct rwnx_baw_ampdu_status {
    u8 total_cnt;
    u8 ok_cnt;
    struct ieee80211_tx_rate rate;
};

/**
 * struct rwnx_baw - Per RA/TID Data for AMPDU TX
 *
 * @ssn: Starting Seq Num
 * @fsn: Starting Seq Num of BlockAck window
 * @buf_size: reorder buffer size at receive
 * @states: State of each element of the BA window BAW_{FREE,PENDING,CONFIRMED}
 * @fsn_idx: Index in the state table corresponding to the fsn value
 * @agg_on: Flag indicating if aggregation is enabled
 * @ba_idx: Index of the BlockAck agreement
 * @rc_status: Transmission status on current AMPDU
 * @last_states: For debug
 */
struct rwnx_baw {
    u16 ssn;
#ifdef CONFIG_RWNX_AGG_TX
    u16 fsn;
    u8 buf_size;
    u8 states[IEEE80211_MAX_AMPDU_BUF];
    u8 fsn_idx;
    bool agg_on;
    u8 ba_idx;
    struct rwnx_baw_ampdu_status rc_status;
#ifdef CONFIG_RWNX_DBG_OOBAWS
    struct rwnx_baw_dbg_states last_states[2];
#endif
#endif
};

extern const int rwnx_tid2hwq[IEEE80211_NUM_TIDS];

#ifdef CONFIG_RWNX_AGG_TX
/**
 * rwnx_move_baw - Move window start to the first non-confirmed MDPU
 *
 * @baw: Pointer on Block Ack window status
 * @return "Size" of the move (i.e. number of MPDU that was confirmed at the
 * start of the BA window)
 */
static inline int rwnx_move_baw(struct rwnx_baw *baw)
{
    int credit = 0;

    while (baw->states[baw->fsn_idx] == BAW_CONFIRMED) {
        baw->states[baw->fsn_idx] = BAW_FREE;
        baw->fsn = (baw->fsn + 1) & ((1 << 12) - 1);
        baw->fsn_idx = (baw->fsn_idx + 1) % baw->buf_size;
        credit++;
    }

    return credit;
}

/**
 * rwnx_is_first_in_baw - Test if a seq number is the first pending MDPU in the BA window
 *
 * @baw: Pointer on Block Ack window status
 * @sn: Seq number
 */
static inline bool rwnx_is_first_in_baw(struct rwnx_baw *baw, u16 sn)
{
    return sn == baw->fsn;
}

/**
 * rwnx_is_last_in_baw - Test if a seq number is the last MPDU in the BA window
 *
 * @baw: Pointer on Block Ack window status
 * @sn: Seq number
 */
static inline bool rwnx_is_last_in_baw(struct rwnx_baw *baw, u16 sn)
{
    u16 last_sn = (baw->fsn + baw->buf_size - 1) & ((1 << 12) - 1);

    return sn == last_sn;
}

/**
 * rwnx_is_in_baw - Test if a seq number is in the window
 *
 * @baw: Pointer on Block Ack window status
 * @sn: Seq number
 */
static inline bool rwnx_is_in_baw(struct rwnx_baw *baw, u16 sn)
{
    u16 baw_offset;
    baw_offset = ((u16)(sn - baw->fsn)) & ((1 << 12) - 1);

    return (baw_offset < baw->buf_size);
}

/**
 * rwnx_set_baw_state - Update state on a MDPU in the BA window
 *
 * @baw: Pointer on Block Ack window status
 * @sn: Seq number
 * @state: New state
 */
#ifndef CONFIG_RWNX_DBG_OOBAWS
static inline int rwnx_set_baw_state(struct rwnx_baw *baw, u16 sn, u8 state,
                                     struct ieee80211_tx_info *__unused)
{
    u16 baw_offset;
    int index;

    baw_offset = ((u16)(sn - baw->fsn)) & ((1 << 12) - 1);
    if (WARN_ON_ONCE(baw_offset >= baw->buf_size))
        return -1;

    index = (baw_offset + baw->fsn_idx) % baw->buf_size;

    baw->states[index] = state;

    return 0;
}
#else
// TODO: UNINLINE ME elsewhere
static inline int rwnx_set_baw_state(struct rwnx_baw *baw, u16 sn, u8 state,
                                     struct ieee80211_tx_info *txi)
{
    u16 baw_offset;
    int index, ret;

    struct rwnx_baw_dbg_states *last_states;
    struct rwnx_baw_dbg_state *last_state;

    last_states = &baw->last_states[state == BAW_CONFIRMED];
    last_states->idx = (last_states->idx + 1) % ARRAY_SIZE(last_states->states);
    last_state = &last_states->states[last_states->idx];

    baw_offset = ((u16)(sn - baw->fsn)) & ((1 << 12) - 1);

    if (WARN_ON_ONCE(baw_offset >= baw->buf_size)) {
        int i;
        printk(KERN_CRIT "baw[%s]: agg_on: %d\n",
               state == BAW_CONFIRMED ? "CFM":"PUSH", baw->agg_on);
        printk(KERN_CRIT " fsn      sn   sz    idx      txflags\n");
        for (i = 0; i < ARRAY_SIZE(last_states->states); i++) {
            struct rwnx_baw_dbg_state *dbg_state =
                &last_states->states[(i + last_states->idx) % ARRAY_SIZE(last_states->states)];
            printk(KERN_CRIT "%04d    %04d   %02d    %03d   0x%08x\n",
                    dbg_state->fsn, dbg_state->sn, dbg_state->buf_size,
                    dbg_state->ba_idx, dbg_state->tx_flags);
        }
        printk(KERN_CRIT "%04d    %04d   %02d    %03d   0x%08x\n",
                baw->fsn, sn, baw->buf_size, baw->ba_idx, txi->flags);
        ret = -1;
    } else {
        index = (baw_offset + baw->fsn_idx) % baw->buf_size;
        baw->states[index] = state;
        ret = 0;
    }

    last_state->fsn = baw->fsn;
    last_state->sn = sn;
    last_state->buf_size = baw->buf_size;
    last_state->ba_idx = baw->ba_idx;
    last_state->tx_flags = txi->flags;

    return ret;
}
#endif

/**
 * rwnx_reset_baw_state - Reset status of the BA window
 *
 * @baw: Pointer on Block Ack window status
 */
static inline void rwnx_reset_baw_state(struct rwnx_baw *baw)
{
    int i;
    for (i = 0; i < IEEE80211_MAX_AMPDU_BUF; i++) {
#ifdef CONFIG_RWNX_DBG_OOBAWS
        int j;
        for (j = 0; j < ARRAY_SIZE(baw->last_states); j++)
            baw->last_states[0].idx = baw->last_states[1].idx = 0;
#endif
        baw->states[i] = BAW_FREE;
    }
}
#endif /* CONFIG_RWNX_AGG_TX */
#endif /* _RWNX_BAWS_H_ */
