/**
 ******************************************************************************
 *
 * @file rwnx_defs.h
 *
 * @brief Main driver structure declarations for softmac driver
 *
 * Copyright (C) RivieraWaves 2012-2019
 *
 ******************************************************************************
 */

#ifndef _RWNX_DEFS_H_
#define _RWNX_DEFS_H_

#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/dmapool.h>
#include <net/mac80211.h>

#include "rwnx_mod_params.h"
#include "rwnx_txq.h"
#include "rwnx_debugfs.h"
#include "rwnx_radar.h"
#include "rwnx_utils.h"
#include "rwnx_mu_group.h"
#include "rwnx_platform.h"
#include "hal_desc.h"
#include "rwnx_cmds.h"
#include "ipc_shared.h"

#define WPI_HDR_LEN    18
#define WPI_PN_LEN     16
#define WPI_PN_OFST     2
#define WPI_MIC_LEN    16
#define WPI_KEY_LEN    32
#define WPI_SUBKEY_LEN 16 // WPI key is actually two 16bytes key

/*
 * Structure used to save information relative to the channel contexts.
 * Will be used as the 'drv_priv' field of the "struct ieee80211_chanctx_conf"
 * structure.
 */
struct rwnx_chanctx {
    struct list_head list;  /* for rwnx_hw.chan_ctxts */
    struct list_head vifs;  /* List of VIFs linked to this channel context */
    struct ieee80211_chanctx_conf *ctx;
    u8 index;               /* Identifier of the channel context */
    bool active;            /* this is the current active channel */
};

/*
 * Structure used to save information relative to the TDLS peer.
 */
struct rwnx_tdls {
    bool active;                /* Indicate if TDLS link is active */
    bool chsw_en;               /* Indicate if channel switch is enabled */
    bool ps_on;                 /* Indicate if the power save is enabled on the
                                   TDLS STA */
};

/*
 * Structure used to save information relative to the managed interfaces.
 * Will be used as the 'drv_priv' field of the "struct ieee80211_vif" structure.
 * This is also linked within the rwnx_hw vifs list.
 *
 */
struct rwnx_vif {
    struct list_head list;      /* for rwnx_hw.vifs */
    struct list_head list_chan_ctxt;  /* for rwnx_chanctx.vifs */
    struct list_head stations;
    struct ieee80211_vif *vif;
    struct rwnx_chanctx *chanctx;
    u8 vif_index;               /* Identifier of the station */
    u8 bcn_on;                  /* Beaconing is enabled */
    s8 txpower;                 /* actual power set for the vif (dBm)*/
    u8 txpower_idx;             /* Idx to be used in policy table */
    bool roc_tdls;              /* Indicate if the ROC has been called by a
                                   TDLS station */
    struct rwnx_txq txqs[NX_NB_TXQ_PER_VIF]; /* no station associated with frame */
};

#define RWNX_VIF_TYPE(rwnx_vif) (rwnx_vif->vif->type)

static inline struct ieee80211_vif *rwnx_to_ieee80211_vif(struct rwnx_vif *rwnx_vif)
{
    if (!rwnx_vif)
        return NULL;
    return container_of((void *)rwnx_vif, struct ieee80211_vif, drv_priv);
}

/**
 * struct rwnx_sta_ps - PS information for a given STA
 *
 * @on: Indicate if STA is in PS mode
 * @uapsd: Indicate if last call to .release_buffered_frames was for uapsd
 *   service period (true) or legacy PS (false).
 * @more_data: Indicate if mac80211 has more data ready when calling
 *   .release_buffered_frames
 * @sp_cnt: Number of buffer "scheduled" for push in last call to
 *   .release_buffered_frames
 */
struct rwnx_sta_ps {
    bool on;
    bool uapsd;
    bool more_data;
    int sp_cnt;
};

/*
 * Structure used to save information relative to the managed stations.
 * Will be used as the 'drv_priv' field of the "struct ieee80211_sta" structure.
 */
struct rwnx_sta {
    struct list_head list;                    /* for rwnx_vif.stations */
    u8 sta_idx;                               /* Identifier of the station */
    u8 hw_sta_idx;                            /* Identifier of the station */
    u8 vif_idx;                               /* Identifier of the VIF the
                                                 station belongs to */
    u8 stbc_nss;                              /* Maximum number of spatial streams
                                                 that can be used with STBC */
    u16 gid;                                  /* group ID */
    u16 paid;                                 /* partial AID */
#ifndef CONFIG_MAC80211_TXQ
    struct rwnx_txq txqs[NX_NB_TXQ_PER_STA];
#endif
#ifdef CONFIG_RWNX_BFMER
    struct rwnx_bfmer_report *bfm_report;     /* Beamforming report to be used for
                                                 VHT TX Beamforming */
#ifdef CONFIG_RWNX_MUMIMO_TX
    struct rwnx_sta_group_info group_info; /* MU grouping information for the STA */
#endif /* CONFIG_RWNX_MUMIMO_TX */
#endif /* CONFIG_RWNX_BFMER */

    struct rwnx_tdls tdls;                    /* TDLS station information */
    struct rwnx_sta_ps ps;                    /* STA Power Save info (AP only) */
};

static inline struct ieee80211_sta *rwnx_to_ieee80211_sta(struct rwnx_sta *rwnx_sta)
{
    if (!rwnx_sta)
        return NULL;
    return container_of((void *)rwnx_sta, struct ieee80211_sta, drv_priv);
}

static inline const u8 *rwnx_sta_addr(struct rwnx_sta *rwnx_sta) {
    struct ieee80211_sta *sta = container_of((void *)rwnx_sta,
                                             struct ieee80211_sta, drv_priv);
    return sta->addr;
}

struct rwnx_stats_aggrx_intra {
    int cnt;
};

struct rwnx_intra_ampdu_stats {
    /* TX AGG stats */
    int done;
    int failed;
};


#ifdef CONFIG_RWNX_SPLIT_TX_BUF
struct rwnx_amsdu_stats {
    int done;
    int failed;
};
#endif

struct rwnx_stats {
    /* RX AGG stats */
    struct rwnx_stats_aggrx_intra rx_in_ampdu[IEEE80211_MAX_AMPDU_BUF];

    /* TX AGG stats */
    int agg_done;
    int agg_retries;
    int agg_retries_last;
    int agg_died;
    int ampdu_all_ko;
    struct rwnx_intra_ampdu_stats in_ampdu[IEEE80211_MAX_AMPDU_BUF];

#ifdef CONFIG_RWNX_SPLIT_TX_BUF
    struct rwnx_amsdu_stats amsdus[NX_TX_PAYLOAD_MAX];
#endif

    /* TX queues stats */
    int queues_stops;
    int cfm_balance[NX_TXQ_CNT];

    unsigned long last_rx, last_tx; /* jiffies */
};

struct rwnx_sec_phy_chan {
    u16 prim20_freq;
    u16 center_freq1;
    u16 center_freq2;
    enum nl80211_band band;
    u8 type;
};

struct rwnx_wpi_key {
    struct hlist_node list;
    struct ieee80211_key_conf *conf;
    u8 pn[16];
};

/* Structure containing channel survey information received from MAC */
struct rwnx_survey_info {
    // Filled
    u32 filled;
    // Amount of time in ms the radio spent on the channel
    u32 chan_time_ms;
    // Amount of time the primary channel was sensed busy
    u32 chan_time_busy_ms;
    // Noise in dbm
    s8 noise_dbm;
};

/* Structure containing TDLS information */
struct rwnx_tdls_info {
    // Number of TDLS stations connected
    u8 n_sta;
    // TDLS channel switch is enabled on one of the TDLS links
    bool chsw_en;
    // PS mode to be set when TDLS link is disabled
    u8 next_ps_mode;
};

/**
 * rwnx_phy_info - Phy information
 *
 * @cnt: Number of phy interface
 * @stbc_nss: Number of spatial stream available for STBC
 * @ctrlinfo_1: Init value for phyctrlinfo_1 field of policy table
 * @ctrlinfo_2: Init value for phyctrlinfo_2 field of policy table
 * @cfg: Configuration send to firmware
 * @sec_chan: Channel configuration of the second phy interface (if phy_cnt > 1)
 * @limit_bw: Set to true to limit BW on requested channel. Only set to use
 * VHT with old radio that don't support 80MHz (deprecated)
 */
struct rwnx_phy_info {
    u8 cnt;
    u8 stbc_nss;
    union rwnx_pol_phy_ctrl_info_1 ctrlinfo_1;
    union rwnx_pol_phy_ctrl_info_2 ctrlinfo_2;
    struct phy_cfg_tag cfg;
    struct rwnx_sec_phy_chan sec_chan;
    bool limit_bw;
};

struct rwnx_hw {
    struct rwnx_mod_params *mod_params;
    struct device *dev;
    struct ieee80211_hw *hw;
    struct list_head vifs;
    struct list_head chan_ctxts;
    enum nl80211_band cur_band;
    u16 cur_freq;

    struct rwnx_cmd_mgr cmd_mgr;

    unsigned long drv_flags;
    struct rwnx_plat *plat;

    spinlock_t tx_lock;
    spinlock_t cb_lock;
#ifdef CONFIG_RWNX_AMSDUS_TX
    spinlock_t amsdu_lock;
#endif
    struct mutex mutex;                         /* per-device perimeter lock */

    struct tasklet_struct task;
    struct mm_version_cfm version_cfm;          /* Lower layers versions - obtained via MM_VERSION_REQ */

    u32 tcp_pacing_shift;

    /* IPC */
    struct ipc_host_env_tag *ipc_env;
    struct rwnx_ipc_elem_pool e2amsgs_pool;
    struct rwnx_ipc_elem_pool dbgmsgs_pool;
    struct rwnx_ipc_elem_pool e2aradars_pool;
    struct rwnx_ipc_elem_var pattern_elem;
    struct rwnx_ipc_dbgdump_elem dbgdump_elem;
    struct rwnx_ipc_skb_elem *e2aunsuprxvec_elems;
    struct rwnx_ipc_skb_elem *rxbuf_elems;
    struct rwnx_ipc_elem_var scan_ie;

    struct kmem_cache      *sw_txhdr_cache;

    struct rwnx_debugfs     debugfs;
    struct rwnx_stats       stats;

    struct rwnx_radar       radar;

    struct rwnx_phy_info phy;

    struct hlist_head wpi_keys;

    bool scanning;                              /* Indicate if SW scan is running */
    u8 scan_txpower_idx;                        /* Power idx to used during SW scan */

    struct rwnx_hwq hwq[NX_TXQ_CNT];
    struct rwnx_survey_info survey[SCAN_CHANNEL_MAX];

#ifdef CONFIG_RWNX_MUMIMO_TX
    struct rwnx_mu_info mu;
#endif

    struct rwnx_tdls_info tdls_info;            /* TDLS information */
};

static inline void *rwnx_get_shared_trace_buf(struct rwnx_hw *rwnx_hw)
{
    return (void *)&(rwnx_hw->debugfs.fw_trace.buf);
}

#endif /* _RWNX_DEFS_H_ */
