/**
 ******************************************************************************
 *
 * @file rwnx_main.c
 *
 * @brief Entry point of the RWNX driver
 *
 * Copyright (C) RivieraWaves 2012-2019
 *
 ******************************************************************************
 */

#include <linux/module.h>
#include <linux/pci.h>
#include <linux/dma-mapping.h>
#include <net/mac80211.h>

#include "rwnx_defs.h"
#include "rwnx_dini.h"
#include "rwnx_msg_tx.h"
#include "rwnx_tx.h"
#include "rwnx_baws.h"
#include "rwnx_testmode.h"
#include "reg_access.h"
#include "hal_desc.h"
#include "rwnx_debugfs.h"
#include "rwnx_cfgfile.h"
#include "rwnx_irqs.h"
#include "rwnx_radar.h"
#ifdef CONFIG_RWNX_BFMER
#include "rwnx_bfmer.h"
#endif //(CONFIG_RWNX_BFMER)
#include "rwnx_version.h"
#include "rwnx_events.h"

#define RW_DRV_DESCRIPTION  "RivieraWaves 11nac driver for Linux"
#define RW_DRV_COPYRIGHT    "Copyright(c) 2016 RivieraWaves"
#define RW_DRV_AUTHOR       "RivieraWaves S.A.S"

#define RWNX_PRINT_CFM_ERR(req) \
        printk(KERN_CRIT "%s: Status Error(%d)\n", #req, (&req##_cfm)->status)

#define RWNX_HT_CAPABILITIES                                    \
{                                                               \
    .ht_supported   = true,                                     \
    .cap            = 0,                                        \
    .ampdu_factor   = IEEE80211_HT_MAX_AMPDU_64K,               \
    .ampdu_density  = IEEE80211_HT_MPDU_DENSITY_8,              \
    .mcs        = {                                             \
        .rx_mask = { 0xff, 0, 0, 0, 0, 0, 0, 0, 0, 0, },        \
        .rx_highest = cpu_to_le16(65),                          \
        .tx_params = IEEE80211_HT_MCS_TX_DEFINED,               \
    },                                                          \
}

#define RWNX_VHT_CAPABILITIES                                   \
{                                                               \
    .vht_supported = false,                                     \
    .cap       =                                                \
        7 << IEEE80211_VHT_CAP_MAX_A_MPDU_LENGTH_EXPONENT_SHIFT,\
    .vht_mcs       = {                                          \
        .rx_mcs_map = cpu_to_le16(                              \
                      IEEE80211_VHT_MCS_SUPPORT_0_9    << 0  |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 2  |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 4  |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 6  |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 8  |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 10 |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 12 |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 14),  \
        .tx_mcs_map = cpu_to_le16(                              \
                      IEEE80211_VHT_MCS_SUPPORT_0_9    << 0  |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 2  |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 4  |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 6  |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 8  |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 10 |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 12 |  \
                      IEEE80211_VHT_MCS_NOT_SUPPORTED  << 14),  \
    }                                                           \
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 20, 0)
#define RWNX_HE_CAPABILITIES                                    \
{                                                               \
    .has_he = false,                                            \
    .he_cap_elem = {                                            \
        .mac_cap_info[0] = 0,                                   \
        .mac_cap_info[1] = 0,                                   \
        .mac_cap_info[2] = 0,                                   \
        .mac_cap_info[3] = 0,                                   \
        .mac_cap_info[4] = 0,                                   \
        .mac_cap_info[5] = 0,                                   \
        .phy_cap_info[0] = 0,                                   \
        .phy_cap_info[1] = 0,                                   \
        .phy_cap_info[2] = 0,                                   \
        .phy_cap_info[3] = 0,                                   \
        .phy_cap_info[4] = 0,                                   \
        .phy_cap_info[5] = 0,                                   \
        .phy_cap_info[6] = 0,                                   \
        .phy_cap_info[7] = 0,                                   \
        .phy_cap_info[8] = 0,                                   \
        .phy_cap_info[9] = 0,                                   \
        .phy_cap_info[10] = 0,                                  \
    },                                                          \
    .he_mcs_nss_supp = {                                        \
        .rx_mcs_80 = cpu_to_le16(0xfffa),                       \
        .tx_mcs_80 = cpu_to_le16(0xfffa),                       \
        .rx_mcs_160 = cpu_to_le16(0xffff),                      \
        .tx_mcs_160 = cpu_to_le16(0xffff),                      \
        .rx_mcs_80p80 = cpu_to_le16(0xffff),                    \
        .tx_mcs_80p80 = cpu_to_le16(0xffff),                    \
    },                                                          \
    .ppe_thres = {0x08, 0x1c, 0x07},                            \
}
#endif

#define RATE(_bitrate, _hw_rate, _flags) {      \
    .bitrate    = (_bitrate),                   \
    .flags      = (_flags),                     \
    .hw_value   = (_hw_rate),                   \
}

#define CHAN(_freq) {                           \
    .center_freq    = (_freq),                  \
    .max_power  = 30, /* FIXME */               \
}

#ifdef CONFIG_NL80211_TESTMODE
/* The TLVs used in the gnl message policy between the kernel module and
 * user space application. iwl_testmode_gnl_msg_policy is to be carried
 * through the NL80211_CMD_TESTMODE channel regulated by nl80211.
 * See iwl-testmode.h
 */
static const struct nla_policy rwnx_testmode_gnl_msg_policy[RWNX_TM_ATTR_MAX] = {
    [RWNX_TM_ATTR_COMMAND]     = {.type = NLA_U32,},
    [RWNX_TM_ATTR_REG_OFFSET]  = {.type = NLA_U32,},
    [RWNX_TM_ATTR_REG_VALUE32] = {.type = NLA_U32,},
    [RWNX_TM_ATTR_REG_FILTER]  = {.type = NLA_U32,},

};
#endif /* CONFIG_NL80211_TESTMODE */

static struct ieee80211_rate rwnx_ratetable[] = {
    RATE(10,  0x00, 0),
    RATE(20,  0x01, IEEE80211_RATE_SHORT_PREAMBLE),
    RATE(55,  0x02, IEEE80211_RATE_SHORT_PREAMBLE),
    RATE(110, 0x03, IEEE80211_RATE_SHORT_PREAMBLE),
    RATE(60,  0x04, 0),
    RATE(90,  0x05, 0),
    RATE(120, 0x06, 0),
    RATE(180, 0x07, 0),
    RATE(240, 0x08, 0),
    RATE(360, 0x09, 0),
    RATE(480, 0x0A, 0),
    RATE(540, 0x0B, 0),
};

/* The channels indexes here are not used anymore */
static struct ieee80211_channel rwnx_2ghz_channels[] = {
    CHAN(2412),
    CHAN(2417),
    CHAN(2422),
    CHAN(2427),
    CHAN(2432),
    CHAN(2437),
    CHAN(2442),
    CHAN(2447),
    CHAN(2452),
    CHAN(2457),
    CHAN(2462),
    CHAN(2467),
    CHAN(2472),
    CHAN(2484),
    // Extra channels defined only to be used for PHY measures.
    // Enabled only if custregd and custchan parameters are set
    CHAN(2390),
    CHAN(2400),
    CHAN(2410),
    CHAN(2420),
    CHAN(2430),
    CHAN(2440),
    CHAN(2450),
    CHAN(2460),
    CHAN(2470),
    CHAN(2480),
    CHAN(2490),
    CHAN(2500),
    CHAN(2510),
};

static struct ieee80211_channel rwnx_5ghz_channels[] = {
    CHAN(5180),             // 36 -   20MHz
    CHAN(5200),             // 40 -   20MHz
    CHAN(5220),             // 44 -   20MHz
    CHAN(5240),             // 48 -   20MHz
    CHAN(5260),             // 52 -   20MHz
    CHAN(5280),             // 56 -   20MHz
    CHAN(5300),             // 60 -   20MHz
    CHAN(5320),             // 64 -   20MHz
    CHAN(5500),             // 100 -  20MHz
    CHAN(5520),             // 104 -  20MHz
    CHAN(5540),             // 108 -  20MHz
    CHAN(5560),             // 112 -  20MHz
    CHAN(5580),             // 116 -  20MHz
    CHAN(5600),             // 120 -  20MHz
    CHAN(5620),             // 124 -  20MHz
    CHAN(5640),             // 128 -  20MHz
    CHAN(5660),             // 132 -  20MHz
    CHAN(5680),             // 136 -  20MHz
    CHAN(5700),             // 140 -  20MHz
    CHAN(5720),             // 144 -  20MHz
    CHAN(5745),             // 149 -  20MHz
    CHAN(5765),             // 153 -  20MHz
    CHAN(5785),             // 157 -  20MHz
    CHAN(5805),             // 161 -  20MHz
    CHAN(5825),             // 165 -  20MHz
    // Extra channels defined only to be used for PHY measures.
    // Enabled only if custregd and custchan parameters are set
    CHAN(5190),
    CHAN(5210),
    CHAN(5230),
    CHAN(5250),
    CHAN(5270),
    CHAN(5290),
    CHAN(5310),
    CHAN(5330),
    CHAN(5340),
    CHAN(5350),
    CHAN(5360),
    CHAN(5370),
    CHAN(5380),
    CHAN(5390),
    CHAN(5400),
    CHAN(5410),
    CHAN(5420),
    CHAN(5430),
    CHAN(5440),
    CHAN(5450),
    CHAN(5460),
    CHAN(5470),
    CHAN(5480),
    CHAN(5490),
    CHAN(5510),
    CHAN(5530),
    CHAN(5550),
    CHAN(5570),
    CHAN(5590),
    CHAN(5610),
    CHAN(5630),
    CHAN(5650),
    CHAN(5670),
    CHAN(5690),
    CHAN(5710),
    CHAN(5730),
    CHAN(5750),
    CHAN(5760),
    CHAN(5770),
    CHAN(5780),
    CHAN(5790),
    CHAN(5800),
    CHAN(5810),
    CHAN(5820),
    CHAN(5830),
    CHAN(5840),
    CHAN(5850),
    CHAN(5860),
    CHAN(5870),
    CHAN(5880),
    CHAN(5890),
    CHAN(5900),
    CHAN(5910),
    CHAN(5920),
    CHAN(5930),
    CHAN(5940),
    CHAN(5950),
    CHAN(5960),
    CHAN(5970),
};

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 20, 0)
static struct ieee80211_sband_iftype_data rwnx_he_capa = {
    .types_mask = BIT(NL80211_IFTYPE_STATION),
    .he_cap = RWNX_HE_CAPABILITIES,
};
#endif

static struct ieee80211_supported_band rwnx_band_2GHz = {
    .channels   = rwnx_2ghz_channels,
    .n_channels = ARRAY_SIZE(rwnx_2ghz_channels) - 13, // -13 to exclude extra channels
    .bitrates   = rwnx_ratetable,
    .n_bitrates = ARRAY_SIZE(rwnx_ratetable),
    .ht_cap     = RWNX_HT_CAPABILITIES,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 20, 0)
    .iftype_data = &rwnx_he_capa,
    .n_iftype_data = 1,
#endif
};

static struct ieee80211_supported_band rwnx_band_5GHz = {
    .channels   = rwnx_5ghz_channels,
    .n_channels = ARRAY_SIZE(rwnx_5ghz_channels) - 59, // -59 to exclude extra channels
    .bitrates   = &rwnx_ratetable[4],
    .n_bitrates = ARRAY_SIZE(rwnx_ratetable) - 4,
    .ht_cap     = RWNX_HT_CAPABILITIES,
    .vht_cap    = RWNX_VHT_CAPABILITIES,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 20, 0)
    .iftype_data = &rwnx_he_capa,
    .n_iftype_data = 1,
#endif
};

static struct ieee80211_iface_limit rwnx_limits[] = {
    { .max = NX_VIRT_DEV_MAX, .types = BIT(NL80211_IFTYPE_AP)         |
                                       BIT(NL80211_IFTYPE_STATION)    |
                                       BIT(NL80211_IFTYPE_P2P_CLIENT) |
                                       BIT(NL80211_IFTYPE_P2P_GO)}
};

static struct ieee80211_iface_limit rwnx_limits_dfs[] = {
    { .max = NX_VIRT_DEV_MAX, .types = BIT(NL80211_IFTYPE_AP)}
};

static struct ieee80211_iface_combination rwnx_combinations[] = {
    {
        .limits                 = rwnx_limits,
        .n_limits               = ARRAY_SIZE(rwnx_limits),
        .num_different_channels = 1,
        .max_interfaces         = NX_VIRT_DEV_MAX,
    },
    /* Keep this combination as the last one */
    {
        .limits                 = rwnx_limits_dfs,
        .n_limits               = ARRAY_SIZE(rwnx_limits_dfs),
        .num_different_channels = 1,
        .max_interfaces         = NX_VIRT_DEV_MAX,
        .radar_detect_widths = (BIT(NL80211_CHAN_WIDTH_20_NOHT) |
                                BIT(NL80211_CHAN_WIDTH_20) |
                                BIT(NL80211_CHAN_WIDTH_40) |
                                BIT(NL80211_CHAN_WIDTH_80)),
    }
};

static const int rwnx_ac2hwq[IEEE80211_NUM_ACS] = {
    [NL80211_TXQ_Q_VO] = RWNX_HWQ_VO,
    [NL80211_TXQ_Q_VI] = RWNX_HWQ_VI,
    [NL80211_TXQ_Q_BE] = RWNX_HWQ_BE,
    [NL80211_TXQ_Q_BK] = RWNX_HWQ_BK
};

const int rwnx_tid2hwq[IEEE80211_NUM_TIDS] = {
    RWNX_HWQ_BE,
    RWNX_HWQ_BK,
    RWNX_HWQ_BK,
    RWNX_HWQ_BE,
    RWNX_HWQ_VI,
    RWNX_HWQ_VI,
    RWNX_HWQ_VO,
    RWNX_HWQ_VO,
    /* Use same mapping as mac80211 for other TID */
    RWNX_HWQ_BE,
    RWNX_HWQ_BK,
    RWNX_HWQ_BK,
    RWNX_HWQ_BE,
    RWNX_HWQ_VI,
    RWNX_HWQ_VI,
    RWNX_HWQ_VO,
    RWNX_HWQ_VO,
};

static struct ieee80211_cipher_scheme rwnx_cs[] ={
    {
        .cipher = WLAN_CIPHER_SUITE_SMS4,
        .iftype = (BIT(NL80211_IFTYPE_STATION) |
                   BIT(NL80211_IFTYPE_MESH_POINT) |
                   BIT(NL80211_IFTYPE_AP)),
        .hdr_len = WPI_HDR_LEN,
        .pn_len = WPI_PN_LEN,
        .pn_off = WPI_PN_OFST,
        .key_idx_off = 0,
        .key_idx_mask = 0x1,
        .key_idx_shift = 0,
        .mic_len = IEEE80211_CCMP_MIC_LEN
        /* This value is used by mac82011 to strip the MIC. In our case MIC is
           stripped by the HW and readded in rwnx_rxdataind(). As there is way
           to distinguish CCMP from WPI in this function, always use CCMP size
           even if WPI MIC length is actually 16 bytes.
        */
    }
};

/**
 * @start: Called before the first netdevice attached to the hardware
 *  is enabled. This should turn on the hardware and must turn on
 *  frame reception (for possibly enabled monitor interfaces.)
 *  Returns negative error codes, these may be seen in userspace,
 *  or zero.
 *  When the device is started it should not have a MAC address
 *  to avoid acknowledging frames before a non-monitor device
 *  is added.
 *  Must be implemented and can sleep.
 *  It does not return until the LMAC is up and running.
 */
static int rwnx_ops_start(struct ieee80211_hw *hw)
{
    int error = 0;
    struct rwnx_hw *rwnx_hw;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;

    /* Exits if device is already started */
    if (WARN_ON(test_bit(RWNX_DEV_STARTED, &rwnx_hw->drv_flags)))
        return -EBUSY;

    /* Start LMAC FW
     * ATM this starts us in non-idle mode */
    if ((error = rwnx_send_start(rwnx_hw)))
        return error;

    if (rwnx_send_set_idle(rwnx_hw, 1))
        return -EIO;

    /* Device is now started */
    set_bit(RWNX_DEV_STARTED, &rwnx_hw->drv_flags);

    ieee80211_wake_queues(hw);

    return error;
}

/**
 * @stop: Called after last netdevice attached to the hardware
 *	is disabled. This should turn off the hardware (at least
 *	it must turn off frame reception.)
 *	May be called right after add_interface if that rejects
 *	an interface. If you added any work onto the mac80211 workqueue
 *	you should ensure to cancel it on this callback.
 *	Must be implemented and can sleep.
 */
static void rwnx_ops_stop(struct ieee80211_hw *hw)
{
    struct rwnx_hw *rwnx_hw;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;

    /* FIXME: ATM when AP we dont get any IEEE80211_CONF_CHANGE_IDLE event */
    rwnx_send_set_idle(rwnx_hw, 1);

    /* relies on REQ_IDLE */
    ieee80211_stop_queues(hw);

    /* This also lets both ipc sides remain in sync before resetting */
    rwnx_ipc_tx_drain(rwnx_hw);

    rwnx_send_reset(rwnx_hw);

    clear_bit(RWNX_DEV_STARTED, &rwnx_hw->drv_flags);
}

/**
 * @add_interface: Called when a netdevice attached to the hardware is
 *	enabled. Because it is not called for monitor mode devices, @start
 *	and @stop must be implemented.
 *	The driver should perform any initialization it needs before
 *	the device can be enabled. The initial configuration for the
 *	interface is given in the conf parameter.
 *	The callback may refuse to add an interface by returning a
 *	negative error code (which will be seen in userspace.)
 *	Must be implemented and can sleep.
 */
static int rwnx_ops_add_interface(struct ieee80211_hw *hw, struct ieee80211_vif *vif)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_vif *rwnx_vif;
    struct mm_add_if_cfm add_if_cfm;
    u8 txq_status;
    int ac;
    int error = 0;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_vif = (struct rwnx_vif *)vif->drv_priv;

    INIT_LIST_HEAD(&rwnx_vif->list_chan_ctxt);
    INIT_LIST_HEAD(&rwnx_vif->stations);

    /* Forward the information to the LMAC */
    if ((error = rwnx_send_add_if(rwnx_hw, vif->addr, vif->type, vif->p2p, &add_if_cfm)))
        return error;

    if (add_if_cfm.status != 0) {
        RWNX_PRINT_CFM_ERR(add_if);
        return -EIO;
    }

    rwnx_vif->vif = vif;
    /* Save the index retrieved from LMAC */
    rwnx_vif->vif_index = add_if_cfm.inst_nbr;

    if (rwnx_mod_params.hwscan) {
        /* Status will be updated when channel context is assigned */
        txq_status = RWNX_TXQ_STOP_CHAN;
    } else {
        txq_status = 0;
    }
    rwnx_txq_vif_init(rwnx_hw, rwnx_vif, txq_status);

    for (ac = 0; ac < IEEE80211_NUM_ACS; ac++)
        vif->hw_queue[ac] = rwnx_ac2hwq[ac];

    if ((vif->type == NL80211_IFTYPE_AP) ||
        (vif->type == NL80211_IFTYPE_MESH_POINT)) {
        vif->cab_queue = RWNX_HWQ_BCN;
    } else {
        vif->cab_queue = IEEE80211_INVAL_HW_QUEUE;
    }

    /* Enable connection quality monitoring */
    vif->driver_flags |= IEEE80211_VIF_SUPPORTS_CQM_RSSI;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 0, 0)
    if (rwnx_hw->mod_params->uapsd_timeout) {
        vif->driver_flags |= IEEE80211_VIF_SUPPORTS_UAPSD;
    }
#endif

    spin_lock_bh(&rwnx_hw->cb_lock);
    list_add_tail(&rwnx_vif->list, &rwnx_hw->vifs);
    spin_unlock_bh(&rwnx_hw->cb_lock);

    return 0;
}

/**
 * @remove_interface: Notifies a driver that an interface is going down.
 *	The @stop callback is called after this if it is the last interface
 *	and no monitor interfaces are present.
 *	When all interfaces are removed, the MAC address in the hardware
 *	must be cleared so the device no longer acknowledges packets,
 *	the mac_addr member of the conf structure is, however, set to the
 *	MAC address of the device going away.
 *	Hence, this callback must be implemented. It can sleep.
 */
static void rwnx_ops_remove_interface(struct ieee80211_hw *hw,
                                      struct ieee80211_vif *vif)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_vif *rwnx_vif;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_vif = (struct rwnx_vif *)vif->drv_priv;

    /* rwnx_vif->vif might be NULL c.f. {ieee80211,rwnx}_restart_hw */
    spin_lock_bh(&rwnx_hw->cb_lock);
    if (rwnx_vif->vif) {
        rwnx_vif->vif = NULL;
        list_del(&rwnx_vif->list);

        if (rwnx_vif->chanctx) {
            list_del(&rwnx_vif->list_chan_ctxt);
        }
    }
    spin_unlock_bh(&rwnx_hw->cb_lock);

    rwnx_txq_vif_deinit(rwnx_hw, rwnx_vif);

    rwnx_send_remove_if(rwnx_hw, rwnx_vif->vif_index);
}

/**
 * @conf_tx: Configure TX queue parameters (EDCF (aifs, cw_min, cw_max),
 *  bursting) for a hardware TX queue.
 *  Returns a negative error code on failure.
 *  The callback can sleep.
 */
static int rwnx_ops_conf_tx(struct ieee80211_hw *hw, struct ieee80211_vif *vif,
                            u16 ac_queue, const struct ieee80211_tx_queue_params *params)
{
    struct rwnx_hw *rwnx_hw;
    u8 hw_queue, aifs, cwmin, cwmax;
    u32 param;
    struct rwnx_vif *rwnx_vif;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_vif = (struct rwnx_vif *)vif->drv_priv;

    hw_queue = rwnx_ac2hwq[ac_queue];

    aifs  = params->aifs;
    cwmin = fls(params->cw_min);
    cwmax = fls(params->cw_max);

    /* Store queue information in general structure */
    param  = (u32) (aifs << 0);
    param |= (u32) (cwmin << 4);
    param |= (u32) (cwmax << 8);
    param |= (u32) (params->txop) << 12;

    /* Sending MM_SET_EDCA_REQ will overwrite all ACs properties registers, so we must
     * be careful when sending it, but since all the table was initialized with the
     * Reset values in the probe function it is OK. */
    return rwnx_send_set_edca(rwnx_hw, hw_queue, param, params->uapsd,
                              rwnx_vif->vif_index);
}

/**
 * @sta_add: Notifies low level driver about addition of an associated station,
 *	AP, IBSS/WDS/mesh peer etc. This callback can sleep.
 */
static int rwnx_ops_sta_add(struct ieee80211_hw *hw, struct ieee80211_vif *vif,
                            struct ieee80211_sta *sta)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_sta *rwnx_sta;
    struct rwnx_vif *rwnx_vif;

    struct mm_sta_add_cfm sta_add_cfm;
    int error = 0;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_sta = (struct rwnx_sta *)sta->drv_priv;
    rwnx_vif = (struct rwnx_vif *)vif->drv_priv;

    if ((error = rwnx_send_sta_add(rwnx_hw, sta, rwnx_vif->vif_index, &sta_add_cfm)))
        return error;

    if (sta_add_cfm.status != 0) {
        RWNX_PRINT_CFM_ERR(sta_add);
        return -EIO;
    }

#ifdef CONFIG_RWNX_BFMER
    if (rwnx_hw->mod_params->bfmer) {
        // Enable Beamformer if supported
        rwnx_send_bfmer_enable(rwnx_hw, sta);
    }
    rwnx_mu_group_sta_init(rwnx_sta,
                           (const struct ieee80211_vht_cap *)&sta->vht_cap);
#endif //(CONFIG_RWNX_BFMER)

    /* Save the index retrieved from LMAC */
    rwnx_sta->sta_idx = sta_add_cfm.sta_idx;
    rwnx_sta->hw_sta_idx = sta_add_cfm.hw_sta_idx;
    rwnx_sta->vif_idx = rwnx_vif->vif_index;
    rwnx_sta->ps.on = false;

    if (sta->tdls) {
        printk("Add TDLS STA %i (%pM)\n", rwnx_sta->sta_idx, sta->addr);
        /* Update number of TDLS stations connected */
        rwnx_hw->tdls_info.n_sta++;
        /* Disable PS when TDLS peer is added */
        rwnx_send_tdls_ps(rwnx_hw, MM_PS_MODE_OFF);
        /* Set TDLS active */
        rwnx_sta->tdls.active = true;
    }

    /* Check if we can use STBC with this station */
    if (sta->ht_cap.ht_supported) {
        u8 stbc_nss;
        if (sta->vht_cap.vht_supported) {
            stbc_nss = (sta->vht_cap.cap & IEEE80211_VHT_CAP_RXSTBC_MASK) >> 8;
            stbc_nss = (stbc_nss <= 4) ? stbc_nss : 0;
        } else {
            stbc_nss = (sta->ht_cap.cap & IEEE80211_HT_CAP_RX_STBC) >>
                                             IEEE80211_HT_CAP_RX_STBC_SHIFT;
        }
        rwnx_sta->stbc_nss = min_t(u8, stbc_nss, rwnx_hw->phy.stbc_nss);
    } else {
        rwnx_sta->stbc_nss = 0;
    }

    /* Set the 'unicast' VHT partial AID and GID */
    if (sta->vht_cap.vht_supported) {
        u16 paid, gid;
        const u8 *mac = NULL;

        /* TODO DLS/TDLS */
        switch (vif->type) {
        /* to AP */
        case NL80211_IFTYPE_STATION:
            if (sta->tdls) {
                mac = (void *)vif->bss_conf.bssid;
                if (mac) {
                    paid = (sta->aid & 0x1ff) + (((mac[5] & 0x0f) ^ ((mac[5] & 0xf0) >> 4)) << 5);
                    paid &= 0x1ff;
                    gid = 63;
                } else {
                    paid = 0;
                    gid = 0;
                }
                break;
            }
            /* fall through */
            /* to MESH STA */
        case NL80211_IFTYPE_MESH_POINT:
            mac = (void *)sta->addr;
            paid = (mac[4] >> 7) | (mac[5] << 1);
            gid = 0;
            break;
        /* from AP to associated STA or
         * direct path from DLS/TDLS STA to DLS/TDLS STA */
        case NL80211_IFTYPE_AP:
            mac = (void *)vif->addr;
            paid = (sta->aid & 0x1ff) + (((mac[5] & 0x0f) ^ ((mac[5] & 0xf0) >> 4)) << 5);
            paid &= 0x1ff;
            gid = 63;
            break;
            /* WARN ? */
        default:
            paid = 0;
            gid = 63;
            break;
        }
        rwnx_sta->paid = paid;
        rwnx_sta->gid = gid;
        RWNX_DBG("%s: VHT sta %pM to %pM - partial AID %d - GID %d\n", __func__,
                 sta->addr, mac, paid, gid);
    }

    spin_lock_bh(&rwnx_hw->cb_lock);
    rwnx_txq_sta_init(rwnx_hw, rwnx_sta, rwnx_txq_vif_get_status(rwnx_vif));
    list_add_tail(&rwnx_sta->list, &rwnx_vif->stations);
    spin_unlock_bh(&rwnx_hw->cb_lock);

    return 0;
}

/**
 * @sta_remove: Notifies low level driver about removal of an associated
 *	station, AP, IBSS/WDS/mesh peer etc. This callback can sleep.
 */
static int rwnx_ops_sta_remove(struct ieee80211_hw *hw, struct ieee80211_vif *vif,
                               struct ieee80211_sta *sta)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_sta *rwnx_sta;
    int error = 0;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_sta = (struct rwnx_sta *)sta->drv_priv;
    spin_lock_bh(&rwnx_hw->cb_lock);
    list_del(&rwnx_sta->list);
    spin_unlock_bh(&rwnx_hw->cb_lock);

#ifdef CONFIG_RWNX_BFMER
    /* If beamforming was used for this station, clean all allocated structures
     * and unmap DMA addresses
     */
    rwnx_bfmer_report_del(rwnx_hw, rwnx_sta);
    rwnx_mu_group_sta_del(rwnx_hw, rwnx_sta);
#endif //(CONFIG_RWNX_BFMER)

    rwnx_txq_sta_deinit(rwnx_hw, rwnx_sta);

    if ((error = rwnx_send_sta_del(rwnx_hw, rwnx_sta->sta_idx)))
        return error;

    if ((sta->tdls) && (rwnx_hw->tdls_info.n_sta)) {
        /* Update number of TDLS stations connected */
        rwnx_hw->tdls_info.n_sta--;
        /* Check if other TDLS STAs are connected */
        if (rwnx_hw->tdls_info.n_sta == 0) {
            RWNX_DBG("%s: no TDLS STAs connected: disable TDLS link and enable PS\n",
                    __func__);
            /* Enable PS when TDLS peer is deleted */
            rwnx_send_tdls_ps(rwnx_hw, rwnx_hw->tdls_info.next_ps_mode);
        }
        /* Set TDLS not active */
        rwnx_sta->tdls.active = false;
        printk("Del TDLS STA %i (%pM)\n", rwnx_sta->sta_idx, sta->addr);
    }

    return 0;
}

/**
 * @sta_notify: Notifies low level driver about power state transition of an
 *	associated station, AP,  IBSS/WDS/mesh peer etc. For a VIF operating
 *	in AP mode, this callback will not be called when the flag
 *	%IEEE80211_HW_AP_LINK_PS is set. Must be atomic.
 */
static void rwnx_ops_sta_notify(struct ieee80211_hw *hw, struct ieee80211_vif *vif,
                                enum sta_notify_cmd cmd, struct ieee80211_sta *sta)
{
    if ((vif->type == NL80211_IFTYPE_AP) ||
        (vif->type == NL80211_IFTYPE_AP_VLAN)) {
        struct rwnx_sta *rwnx_sta = (struct rwnx_sta *)sta->drv_priv;
        struct rwnx_txq *txq;
        u8 tid;

        if (cmd == STA_NOTIFY_SLEEP) {
            rwnx_sta->ps.on = true;
            /* Note: we don't stop txq with RWNX_TXQ_STOP_STA_PS because mac80211
               take care of this and it would require rwnx_tx_ops to test for
               IEEE80211_TX_CTL_NO_PS_BUFFER flag */
            foreach_sta_txq(rwnx_sta, txq, tid, NULL) {
                ieee80211_sta_set_buffered(sta, tid,
                                           !skb_queue_empty(&txq->sk_list));
            }

        } else if (cmd == STA_NOTIFY_AWAKE) {
            rwnx_sta->ps.on = false;
            foreach_sta_txq(rwnx_sta, txq, tid, NULL) {
                txq->push_limit = 0;
            }
        }
    }
}

/**
 * @flush: Flush all pending frames from the hardware queue, making sure
 *	that the hardware queues are empty. The @queues parameter is a bitmap
 *	of queues to flush, which is useful if different virtual interfaces
 *	use different hardware queues; it may also indicate all queues.
 *	If the parameter @drop is set to %true, pending frames may be dropped.
 *	Note that vif can be NULL.
 *	The callback can sleep.
 * TODO: handle params
 */
static void rwnx_ops_flush(struct ieee80211_hw *hw, struct ieee80211_vif *vif,
                           u32 queues, bool drop)
{
    struct rwnx_hw *rwnx_hw = hw->priv;
    int flush_duration;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    if (test_bit(RWNX_DEV_RESTARTING, &rwnx_hw->drv_flags)) {
        printk(KERN_CRIT "%s: bypassing (RWNX_DEV_RESTARTING set)\n", __func__);
        return;
    } else if (!rwnx_hw->ipc_env) {
        printk(KERN_CRIT "%s: bypassing (restart must have failed)\n", __func__);
        return;
    }

    /* Wait for a maximum time of 200ms until all pending frames are flushed */
    for (flush_duration = 0; flush_duration < 200; flush_duration++) {
        if (!rwnx_ipc_tx_pending(rwnx_hw))
            return;
        usleep_range(1000, 2000);
    }

    /* TODO Add flush of all remaining packets */
}

/**
 * @config: Handler for configuration requests. IEEE 802.11 code calls this
 *  function to change hardware configuration, e.g., channel.
 *  This function should never fail but returns a negative error code
 *  if it does. The callback can sleep
 */
static int rwnx_ops_config(struct ieee80211_hw *hw, u32 changed)
{
    struct rwnx_hw *rwnx_hw;
    int error = 0;
    u8 ps_mode;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;

    if (!rwnx_hw->mod_params->hwscan) {
        if (changed & IEEE80211_CONF_CHANGE_CHANNEL) {
            struct mm_set_channel_cfm cfm;

            /* Flush the pending data to ensure that we will finish the pending
             * transmissions before changing the channel */
            rwnx_ops_flush(hw, NULL, -1, false);

            if (rwnx_send_set_channel(rwnx_hw, 0, &cfm))
                return -EIO;

            if (rwnx_hw->scanning)
                rwnx_hw->scan_txpower_idx = cfm.radio_idx;

            rwnx_hw->cur_band = hw->conf.chandef.chan->band;
            rwnx_hw->cur_freq = hw->conf.chandef.center_freq1;
            if (hw->conf.chandef.chan->flags & IEEE80211_CHAN_RADAR) {
                rwnx_radar_detection_enable(&rwnx_hw->radar,
                                            RWNX_RADAR_DETECT_REPORT,
                                            RWNX_RADAR_RIU);
            } else {
                rwnx_radar_detection_enable(&rwnx_hw->radar,
                                            RWNX_RADAR_DETECT_DISABLE,
                                            RWNX_RADAR_RIU);
            }
        }
    }

    if (rwnx_hw->mod_params->ps_on) {
        if (changed & IEEE80211_CONF_CHANGE_PS) {
            if ((hw->conf.flags & IEEE80211_CONF_PS) == IEEE80211_CONF_PS) {
                if (rwnx_hw->mod_params->dpsm) {
                    /* Switch to Dynamic Power Save */
                    ps_mode = MM_PS_MODE_ON_DYN;
                } else {
                    /* Switch to Power Save */
                    ps_mode = MM_PS_MODE_ON;
                }
            } else {
                /* Exit Power Save */
                ps_mode = MM_PS_MODE_OFF;
            }
            rwnx_hw->tdls_info.next_ps_mode = ps_mode;
            if (rwnx_hw->tdls_info.n_sta == 0) {
                error = rwnx_send_set_ps_mode(rwnx_hw, ps_mode);
            }
        }
    }

    if (changed & IEEE80211_CONF_CHANGE_RETRY_LIMITS) {

    }

    if (changed & IEEE80211_CONF_CHANGE_SMPS) {

    }

    if (changed & IEEE80211_CONF_CHANGE_LISTEN_INTERVAL) {

    }

    if (changed & IEEE80211_CONF_CHANGE_MONITOR) {

    }

    if (changed & IEEE80211_CONF_CHANGE_IDLE) {
        if (rwnx_send_set_idle(rwnx_hw,
                               !!(hw->conf.flags & IEEE80211_CONF_IDLE)))
            return -EIO;
    }

    return error;
}

/**
 * @bss_info_changed: Handler for configuration requests related to BSS
 *  parameters that may vary during BSS's lifespan, and may affect low
 *  level driver (e.g. assoc/disassoc status, erp parameters).
 *  This function should not be used if no BSS has been set, unless
 *  for association indication. The @changed parameter indicates which
 *  of the bss parameters has changed when a call is made. The callback
 *  can sleep.
 */
static void rwnx_ops_bss_info_changed(struct ieee80211_hw *hw,
                                      struct ieee80211_vif *vif,
                                      struct ieee80211_bss_conf *info,
                                      u32 changed)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_vif *rwnx_vif;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_vif = (struct rwnx_vif *)vif->drv_priv;

    if (changed & BSS_CHANGED_ASSOC) {
        if (rwnx_send_set_vif_state(rwnx_hw, info->assoc, info->aid,
                                    rwnx_vif->vif_index))
            return;
        if (rwnx_hw->mod_params->ps_on) {
            if (rwnx_send_set_ps_options(rwnx_hw, rwnx_mod_params.listen_bcmc,
                                         rwnx_mod_params.listen_itv,
                                         rwnx_vif->vif_index))
                return;
        }
        /* TODO: else... */
    }

    if (changed & BSS_CHANGED_BSSID) {
        if (rwnx_send_set_bssid(rwnx_hw, info->bssid, rwnx_vif->vif_index))
            return;
    }

    if (changed & BSS_CHANGED_BEACON_INT) {
        if (rwnx_send_set_beacon_int(rwnx_hw, info->beacon_int, rwnx_vif->vif_index))
            return;
        if ((vif->type == NL80211_IFTYPE_AP) || (vif->type == NL80211_IFTYPE_MESH_POINT)) {
            if (rwnx_send_dtim_req(rwnx_hw, info->dtim_period))
                return;
        }
    }

    if (changed & BSS_CHANGED_BEACON_ENABLED) {
        if (rwnx_send_set_vif_state(rwnx_hw, info->enable_beacon, 0,
                                    rwnx_vif->vif_index))
            return;
        rwnx_vif->bcn_on = info->enable_beacon;
    }

    if (rwnx_hw->mod_params->autobcn) {
        if (changed & BSS_CHANGED_BEACON) {
             struct sk_buff *skb;
             dma_addr_t dma_addr;
            struct ieee80211_mutable_offsets offs;
            skb = ieee80211_beacon_get_template(rwnx_hw->hw, vif, &offs);

            dma_addr = dma_map_single(rwnx_hw->dev, skb->data, skb->len,
                                      DMA_TO_DEVICE);
            if (WARN_ON(dma_mapping_error(rwnx_hw->dev, dma_addr)))
                return;
            if (rwnx_send_bcn_change(rwnx_hw, rwnx_vif->vif_index, dma_addr,
                                     skb->len, offs.tim_offset, offs.tim_length,
                                     offs.csa_counter_offs))
                return;

            dma_unmap_single(rwnx_hw->dev, dma_addr, skb->len, DMA_TO_DEVICE);
            dev_kfree_skb_any(skb);
        }
    }

    if ((changed & BSS_CHANGED_BASIC_RATES) && info->chandef.chan) {
        int shift =
             hw->wiphy->bands[info->chandef.chan->band]->bitrates[0].hw_value;
        if (rwnx_send_set_br(rwnx_hw, info->basic_rates << shift,
                             rwnx_vif->vif_index, info->chandef.chan->band))
            return;
        /* TODO: check if rwnx_send_set_mode() should be called */
    }

    if (changed & BSS_CHANGED_ERP_SLOT) {
        /* We must be in 11g mode here */
        /* TODO: we can add a check on the mode */
        if (rwnx_send_set_slottime(rwnx_hw, info->use_short_slot))
            return;
    }

    if (changed & BSS_CHANGED_TXPOWER) {
        struct mm_set_power_cfm cfm;

        if (rwnx_send_set_power(rwnx_hw, rwnx_vif->vif_index, info->txpower, &cfm))
            return;

        rwnx_vif->txpower = cfm.power;
        rwnx_vif->txpower_idx = cfm.radio_idx;
    }

    /* TODO check against tx():IEEE80211_TX_RC_USE_SHORT_PREAMBLE */
    if (changed & BSS_CHANGED_ERP_PREAMBLE) {

    }

    /* TODO check against tx():IEEE80211_TX_RC_USE_CTS_PROTECT */
    if (changed & BSS_CHANGED_ERP_CTS_PROT) {

    }

    if (changed & BSS_CHANGED_HT) {

    }

    if (changed & BSS_CHANGED_P2P_PS) {

    }

    if (changed & BSS_CHANGED_CQM) {
        if (rwnx_send_cfg_rssi_req(rwnx_hw, rwnx_vif->vif_index, info->cqm_rssi_thold, info->cqm_rssi_hyst))
            return;
    }
}

static u64 rwnx_prepare_multicast(struct ieee80211_hw *hw,
                                  struct netdev_hw_addr_list *mc_list)
{
    return netdev_hw_addr_list_count(mc_list);
}

/**
 * @configure_filter: Configure the device's RX filter.
 *  See the section "Frame filtering" for more information.
 *  This callback must be implemented and can sleep.
 */
static void rwnx_ops_configure_filter(struct ieee80211_hw *hw,
                                      u32 changed_flags,
                                      u32 *total_flags, u64 multicast)
{
    struct rwnx_hw *rwnx_hw;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;

    RWNX_DBG("    total_flags = 0x%08x\n", *total_flags);

    /* Reset our filter flags since our start/stop ops reset
     * the programmed settings */
    if (!test_bit(RWNX_DEV_STARTED, &rwnx_hw->drv_flags)) {
        *total_flags = 0;
        return;
    }

    if (multicast)
        *total_flags |= FIF_ALLMULTI;
    else
        *total_flags &= ~FIF_ALLMULTI;

    /* TODO optimize with changed_flags vs multicast */
    rwnx_send_set_filter(rwnx_hw, *total_flags);

    /* TODO update total_flags with truly set flags */
    *total_flags &= ~(1 << 31);
}

/**
 * Allocate and init structure for WPI key
 */
static int rwnx_alloc_wpi_key(struct rwnx_hw *rwnx_hw, bool is_ap_if,
                              struct ieee80211_key_conf *conf)
{
    struct rwnx_wpi_key *key;
    int i;
    key = kzalloc(sizeof(struct rwnx_wpi_key), GFP_KERNEL);

    if (key == NULL)
        return -ENOMEM;

    key->conf = conf;
    for (i = 0; i < WPI_PN_LEN ; i += 2) {
        key->pn[i] = 0x36;
        key->pn[i + 1] = 0x5c;
    }

    if (conf->flags & IEEE80211_KEY_FLAG_PAIRWISE && is_ap_if)
        key->pn[0]++;

    hlist_add_head(&key->list, &rwnx_hw->wpi_keys);

    return 0;
}


/**
 * Free structure for WPI key
 */
static int rwnx_free_wpi_key(struct rwnx_hw *rwnx_hw,
                             struct ieee80211_key_conf *conf)
{
     struct rwnx_wpi_key *key;

    hlist_for_each_entry(key, &rwnx_hw->wpi_keys, list) {
        if (key->conf == conf)
            break;
    }

    if (!key)
        return -EINVAL;

    hlist_del(&key->list);
    kfree(key);

    return 0;
}

/**
 * @set_key: See the section "Hardware crypto acceleration"
 *  This callback is only called between add_interface and
 *  remove_interface calls, i.e. while the given virtual interface
 *  is enabled.
 *  Returns a negative error code if the key can't be added.
 *  The callback can sleep.
 */
static int rwnx_ops_set_key(struct ieee80211_hw *hw, enum set_key_cmd cmd,
                            struct ieee80211_vif *vif,
                            struct ieee80211_sta *sta, struct ieee80211_key_conf *key)
{
    struct rwnx_hw *rwnx_hw;
    int i, error = 0;
    struct mm_key_add_cfm key_add_cfm;
    u8_l cipher_suite = 0;

    RWNX_DBG(RWNX_FN_ENTRY_STR);
    rwnx_hw = hw->priv;

    switch (cmd) {
    case SET_KEY:
        /* Retrieve the cipher suite selector */
        switch (key->cipher) {
        case WLAN_CIPHER_SUITE_WEP40:
            cipher_suite = MAC_CIPHER_WEP40;
            break;
        case WLAN_CIPHER_SUITE_WEP104:
            cipher_suite = MAC_CIPHER_WEP104;
            break;
        case WLAN_CIPHER_SUITE_TKIP:
            cipher_suite = MAC_CIPHER_TKIP;
            break;
        case WLAN_CIPHER_SUITE_CCMP:
            cipher_suite = MAC_CIPHER_CCMP;
            break;
        case WLAN_CIPHER_SUITE_AES_CMAC:
            return -EOPNOTSUPP;
        case WLAN_CIPHER_SUITE_SMS4:
            cipher_suite = MAC_CIPHER_WPI_SMS4;
            /* reverse each subkey */
            for (i = 0; i < WPI_SUBKEY_LEN/2; i++) {
                u8 tmp;
                tmp = key->key[i];
                key->key[i] = key->key[WPI_SUBKEY_LEN - 1 - i];
                key->key[WPI_SUBKEY_LEN - 1 - i] = tmp;
            }
            for (i = 0; i < WPI_SUBKEY_LEN/2; i++) {
                u8 tmp;
                tmp = key->key[i + WPI_SUBKEY_LEN];
                key->key[i + WPI_SUBKEY_LEN] = key->key[WPI_KEY_LEN - 1 - i];
                key->key[WPI_KEY_LEN - 1 - i] = tmp;
            }
            /*
              icv_len has been initialized with rwnx_cs.mic_len value which is
              set to CCMP MIC length (see rwnx_cs decalration for details).
              Overwrite it with correct value for WPI.
            */
            key->icv_len = WPI_MIC_LEN;
            break;
        default:
            return -EINVAL;
        }

        if ((error = rwnx_send_key_add(rwnx_hw,
                                       ((struct rwnx_vif *)vif->drv_priv)->vif_index,
                                       sta ? ((struct rwnx_sta *)sta->drv_priv)->sta_idx : 0xFF,
                                       ((key->flags & IEEE80211_KEY_FLAG_PAIRWISE) == IEEE80211_KEY_FLAG_PAIRWISE),
                                       key->key, key->keylen, key->keyidx,
                                       cipher_suite, &key_add_cfm)))
            return error;

        if (key_add_cfm.status != 0) {
            RWNX_PRINT_CFM_ERR(key_add);
            return -EIO;
        }

        /* Save the index retrieved from LMAC */
        key->hw_key_idx = key_add_cfm.hw_key_idx;

        /* Now inform mac80211 about our choices regarding header fields generation:
         * we let mac80211 take care of all generations except for wapi*/
        if (key->cipher == WLAN_CIPHER_SUITE_SMS4) {
            key->flags |= IEEE80211_KEY_FLAG_PUT_IV_SPACE;
            if (rwnx_alloc_wpi_key(rwnx_hw, vif->type == NL80211_IFTYPE_AP, key))
                return -ENOMEM;
        }
        else
            key->flags |= IEEE80211_KEY_FLAG_GENERATE_IV;
        if (key->cipher == WLAN_CIPHER_SUITE_TKIP)
            key->flags |= IEEE80211_KEY_FLAG_GENERATE_MMIC;

        break;

    case DISABLE_KEY:
        error = rwnx_send_key_del(rwnx_hw, key->hw_key_idx);
        if (key->cipher == WLAN_CIPHER_SUITE_SMS4)
            rwnx_free_wpi_key(rwnx_hw, key);
        break;

    default:
        error = -EINVAL;
        break;
    }

    return error;
}

/**
 * @ampdu_action: Perform a certain A-MPDU action
 *	The RA/TID combination determines the destination and TID we want
 *	the ampdu action to be performed for. The action is defined through
 *	ieee80211_ampdu_mlme_action. Starting sequence number (@ssn)
 *	is the first frame we expect to perform the action on. Notice
 *	that TX/RX_STOP can pass NULL for this parameter.
 *	The @buf_size parameter is only valid when the action is set to
 *	%IEEE80211_AMPDU_TX_OPERATIONAL and indicates the peer's reorder
 *	buffer size (number of subframes) for this session -- the driver
 *	may neither send aggregates containing more subframes than this
 *	nor send aggregates in a way that lost frames would exceed the
 *	buffer size. If just limiting the aggregate size, this would be
 *	possible with a buf_size of 8:
 *	 - TX: 1.....7
 *	 - RX:  2....7 (lost frame #1)
 *	 - TX:        8..1...
 *	which is invalid since #1 was now re-transmitted well past the
 *	buffer size of 8. Correct ways to retransmit #1 would be:
 *	 - TX:       1 or 18 or 81
 *	Even "189" would be wrong since 1 could be lost again.
 *
 *	Returns a negative error code on failure.
 *	The callback can sleep.

 */
#ifdef CONFIG_VENDOR_RWNX_AMSDUS_TX
static int rwnx_ops_ampdu_action(struct ieee80211_hw *hw, struct ieee80211_vif *vif,
                                 enum ieee80211_ampdu_mlme_action action,
                                 struct ieee80211_sta *sta,
                                 u16 tid, u16 *ssn, u8 buf_size,
                                 bool *amsdu_supported)
#else
static int rwnx_ops_ampdu_action(struct ieee80211_hw *hw,
                                 struct ieee80211_vif *vif,
                                 struct ieee80211_ampdu_params *params)
#endif
{
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 6, 0))
    enum ieee80211_ampdu_mlme_action action = params->action;
    struct ieee80211_sta *sta = params->sta;
    u16 tid = params->tid;
    u16 *ssn = &params->ssn;
    u8 buf_size = params->buf_size;
#endif
    struct rwnx_hw *rwnx_hw = hw->priv;
    struct rwnx_sta *rwnx_sta = (struct rwnx_sta *)sta->drv_priv;
    struct rwnx_txq *txq = rwnx_txq_sta_get(rwnx_sta, tid);
    struct rwnx_baw *baw = &txq->baw;
    struct mm_ba_add_cfm ba_add_cfm;
    struct mm_ba_del_cfm ba_del_cfm;

    int i, cred, ret = 0;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    if ((!ieee80211_hw_check(hw, AMPDU_AGGREGATION) ||
         !rwnx_hw->mod_params->agg_tx) &&
        action == IEEE80211_AMPDU_TX_START)
        return -ENOTSUPP;

    /*
     * REALLY BELIEVE IN RWNX it's all handled in HW and we simply send up
     * already deaggregated MPDUs with a simple indication they're part of an A-MPDU
     * no other need of informing driver or LMAC of RX BA sessions setup or tear down
     */
    switch (action) {
    /* Setup of BA agreement as a recipient(RX aggregation session) */
    case IEEE80211_AMPDU_RX_START:
        WARN_ON((rwnx_send_ba_add(rwnx_hw, BA_AGMT_RX, rwnx_sta->sta_idx, tid,
                                  *ssn, buf_size, &ba_add_cfm)));
        break;
    /* Tear down of BA agreement as a recipient(RX aggregation session) */
    case IEEE80211_AMPDU_RX_STOP:
        break;

    /* Setup of BA agreement as an originator(TX aggregation session) */
    case IEEE80211_AMPDU_TX_START:
#ifdef CONFIG_RWNX_AGG_TX
        baw->buf_size = 0;
        baw->ssn = *ssn;
        baw->ba_idx++;
        baw->fsn = 0;
#ifdef CONFIG_RWNX_AMSDUS_TX
        *amsdu_supported = true;
#endif
#endif
        /* mandatory callback once setup preparations are done at lower level */
        if (!ret)
            ieee80211_start_tx_ba_cb_irqsafe(vif, sta->addr, tid);
        break;

    case IEEE80211_AMPDU_TX_STOP_CONT:
    case IEEE80211_AMPDU_TX_STOP_FLUSH:
    case IEEE80211_AMPDU_TX_STOP_FLUSH_CONT:
        /* TODO handle flush and different STOP kinds */
        /* Send BA_DEL message to LMAC */
#ifdef CONFIG_RWNX_AGG_TX
        spin_lock_bh(&rwnx_hw->tx_lock);

        if (baw->agg_on) {
            cred = baw->buf_size - NX_TXQ_INITIAL_CREDITS;
            for (i = 0; i < baw->buf_size; i++) {
                if (baw->states[i] == BAW_CONFIRMED) {
                cred--;
                }
            }
            txq->credits -= cred;
            trace_credit_update(txq, -cred);
            if (txq->credits <= 0) {
                rwnx_txq_stop(txq, RWNX_TXQ_STOP_FULL);
            }
        }

        baw->ba_idx++;
        baw->agg_on = false;
        baw->buf_size = 0;
        rwnx_reset_baw_state(baw);

        spin_unlock_bh(&rwnx_hw->tx_lock);
#endif
        if ((ret = rwnx_send_ba_del(rwnx_hw, rwnx_sta->sta_idx, tid, &ba_del_cfm)))
            return ret;
        /* XXX BA_AGMT_DOESNT_EXIST */
        if (ba_del_cfm.status != BA_AGMT_DELETED &&
            ba_del_cfm.status != BA_AGMT_DOESNT_EXIST) {
            RWNX_PRINT_CFM_ERR(ba_del);
            ret = -EIO;
        }
        /* mandatory callback once we've made our own tear down ops */
        if (action != IEEE80211_AMPDU_TX_STOP_FLUSH)
            ieee80211_stop_tx_ba_cb_irqsafe(vif, sta->addr, tid);
        if (ret)
            return -EIO;
        break;

    case IEEE80211_AMPDU_TX_OPERATIONAL:
        /* Send BA_ADD message to LMAC */
        if ((ret = rwnx_send_ba_add(rwnx_hw, BA_AGMT_TX, rwnx_sta->sta_idx, tid,
                                    buf_size, baw->ssn, &ba_add_cfm)))
            return ret;
        if (ba_add_cfm.status != BA_AGMT_ESTABLISHED) {
            RWNX_PRINT_CFM_ERR(ba_add);
            return -EIO;
        }

#ifdef CONFIG_RWNX_AGG_TX
#ifdef CONFIG_RWNX_AMSDUS_TX
        /* check amsdu_supported only to use a local amsdu_forced */
        if (amsdu_supported) {
            bool amsdu_forced = false;

            if (rwnx_hw->mod_params->amsdu_force == 1) {
                amsdu_forced = !*amsdu_supported;
                *amsdu_supported = true;
            } else if (rwnx_hw->mod_params->amsdu_force == 2) {
                *amsdu_supported = false;
            }

            if (*amsdu_supported) {
                /* 4095 is MDPU (i.e. MAC HDR + AMSDU + FCS) limit inside A-MPDU even if
                   peer supports 7935 A-MSDU. (-4 for FCS as driver already include
                   MAC HDR in its computation) */
                txq->amsdu_ht_len_cap =
                    sta->ht_cap.cap & IEEE80211_HT_CAP_MAX_AMSDU ? 4091 : 3839;
                if (sta->vht_cap.vht_supported) {
                    u32 vcap_mlen = sta->vht_cap.cap &
                        (IEEE80211_VHT_CAP_MAX_MPDU_LENGTH_3895 |
                         IEEE80211_VHT_CAP_MAX_MPDU_LENGTH_7991 |
                         IEEE80211_VHT_CAP_MAX_MPDU_LENGTH_11454);
                    switch (vcap_mlen) {
                    case IEEE80211_VHT_CAP_MAX_MPDU_LENGTH_3895:
                        txq->amsdu_vht_len_cap = 3895;
                        break;
                    case IEEE80211_VHT_CAP_MAX_MPDU_LENGTH_7991:
                        txq->amsdu_vht_len_cap = 7991;
                        break;
                    case IEEE80211_VHT_CAP_MAX_MPDU_LENGTH_11454:
                        txq->amsdu_vht_len_cap = 11454;
                        break;
                    }
                } else {
                    txq->amsdu_vht_len_cap = 0;
                }
                RWNX_DBG("sta %pM tid %d A-MSDU len caps: ht:%d vht:%d forced:%d\n",
                        sta->addr, tid, txq->amsdu_ht_len_cap,
                        txq->amsdu_vht_len_cap, amsdu_forced);
            } else {
                txq->amsdu_ht_len_cap = 0;
            }
        }
#endif

        spin_lock_bh(&rwnx_hw->tx_lock);
        txq->credits += buf_size - NX_TXQ_INITIAL_CREDITS;

        if (WARN_ON(txq->credits > buf_size))
            txq->credits = buf_size;
        trace_credit_update(txq, buf_size - NX_TXQ_INITIAL_CREDITS);
        if (txq->credits > 0)
            rwnx_txq_start(txq, RWNX_TXQ_STOP_FULL);
        spin_unlock_bh(&rwnx_hw->tx_lock);


        baw->fsn = baw->ssn;
        baw->buf_size = buf_size;
        wmb();
        baw->agg_on = true;
#endif
        break;

    default:
        RWNX_DBG("### Unknown AMPDU action\n");
    }

    return ret;
}

/**
 * @testmode_cmd: Implement a cfg80211 test mode command. The passed @vif may
 *	be %NULL. The callback can sleep.
 */
#ifdef CONFIG_NL80211_TESTMODE
static int rwnx_op_testmode_cmd(struct ieee80211_hw *hw,
                                struct ieee80211_vif *vif, void *data, int len)
{
    struct nlattr *tb[RWNX_TM_ATTR_MAX];
    struct rwnx_hw *rwnx_hw;
    int result;

    RWNX_DBG(RWNX_FN_ENTRY_STR);
    rwnx_hw = hw->priv;

    /* Parse the nessage and fill in the tb array */
    result =
        nla_parse(tb, RWNX_TM_ATTR_MAX - 1, data, len,
                  rwnx_testmode_gnl_msg_policy, NULL);
    if (result != 0) {
        printk("Error parsing the gnl message : %d\n", result);
        return result;
    }

    /* RWNX_TM_ATTR_COMMAND is absolutely mandatory */
    if (!tb[RWNX_TM_ATTR_COMMAND]) {
        printk("Error finding testmode command type\n");
        return -ENOMSG;
    }

    /* In case multiple accesses to the device happen at the same time */
    mutex_lock(&(rwnx_hw->mutex));

    /* Parse the kind of commands */
    switch (nla_get_u32(tb[RWNX_TM_ATTR_COMMAND])) {
    /* Normal register accesses */
    case RWNX_TM_CMD_APP2DEV_REG_READ:
    case RWNX_TM_CMD_APP2DEV_REG_WRITE:
        result = rwnx_testmode_reg(hw, tb);
        break;

    /* Set Debug filters */
    case RWNX_TM_CMD_APP2DEV_SET_DBGMODFILTER:
    case RWNX_TM_CMD_APP2DEV_SET_DBGSEVFILTER:
        result = rwnx_testmode_dbg_filter(hw, tb);
        break;

    /* Debug register accesses */
    case RWNX_TM_CMD_APP2DEV_REG_READ_DBG:
    case RWNX_TM_CMD_APP2DEV_REG_WRITE_DBG:
        result = rwnx_testmode_reg_dbg(hw, tb);
        break;

    default:
        printk("Unknown testmode command\n");
        result = -ENOSYS;
        break;
    }

    mutex_unlock(&(rwnx_hw->mutex));

    return result;
}
#endif

/**
 * @tx_frames_pending: Check if there is any pending frame in the hardware
 *	queues before entering power save.
 */
static bool rwnx_ops_tx_frames_pending(struct ieee80211_hw *hw)
{
    struct rwnx_hw *rwnx_hw = hw->priv;

    return rwnx_ipc_tx_pending(rwnx_hw);
}

/**
 * @sw_scan_start: Notifier function that is called just before a software scan
 *	is started. Can be NULL, if the driver doesn't need this notification.
 *	The mac_addr parameter allows supporting NL80211_SCAN_FLAG_RANDOM_ADDR,
 *	the driver may set the NL80211_FEATURE_SCAN_RANDOM_MAC_ADDR flag if it
 *	can use this parameter. The callback can sleep.
 */
void rwnx_ops_sw_scan_start(struct ieee80211_hw *hw, struct ieee80211_vif *vif,
                            const u8 *mac_addr)
{
    struct rwnx_hw *rwnx_hw = hw->priv;

    rwnx_hw->scanning = true;
}

/**
 * @sw_scan_complete: Notifier function that is called just after a
 *	software scan finished. Can be NULL, if the driver doesn't need
 *	this notification.
 *	The callback can sleep.
 */
static void rwnx_ops_sw_scan_complete(struct ieee80211_hw *hw,
                                      struct ieee80211_vif *vif)
{
    struct rwnx_hw *rwnx_hw = hw->priv;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 19, 0)
    struct rwnx_vif *rwnx_vif = (struct rwnx_vif *)vif->drv_priv;

    /* reset tx power at fw level as it has been changed during scan */
    if (!vif->bss_conf.idle) {
        struct mm_set_power_cfm cfm;

        rwnx_send_set_power(rwnx_hw, rwnx_vif->vif_index,
                            vif->bss_conf.txpower, &cfm);
        rwnx_vif->txpower = cfm.power;
        rwnx_vif->txpower_idx = cfm.radio_idx;
    }
#endif
    rwnx_hw->scanning = false;
}

/**
 * @hw_scan: Ask the hardware to service the scan request, no need to start
 *	the scan state machine in stack. The scan must honour the channel
 *	configuration done by the regulatory agent in the wiphy's
 *	registered bands. The hardware (or the driver) needs to make sure
 *	that power save is disabled.
 *	The @req ie/ie_len members are rewritten by mac80211 to contain the
 *	entire IEs after the SSID, so that drivers need not look at these
 *	at all but just send them after the SSID -- mac80211 includes the
 *	(extended) supported rates and HT information (where applicable).
 *	When the scan finishes, ieee80211_scan_completed() must be called;
 *	note that it also must be called when the scan cannot finish due to
 *	any error unless this callback returned a negative error code.
 *	The callback can sleep.
 */
static int rwnx_ops_hw_scan(struct ieee80211_hw *hw, struct ieee80211_vif *vif,
                            struct ieee80211_scan_request *hw_req)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 17, 0)
    struct cfg80211_scan_request *req = &hw_req->req;
#endif
    struct rwnx_hw *rwnx_hw = hw->priv;
    struct scan_start_cfm scan_start_cfm;
    int error;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    if ((error = rwnx_send_scan_req(rwnx_hw, vif, req, &scan_start_cfm)))
        return error;

    if (scan_start_cfm.status != 0) {
        RWNX_PRINT_CFM_ERR(scan_start);
        rwnx_ipc_elem_var_deallocs(rwnx_hw, &rwnx_hw->scan_ie);
        return -EIO;
    }

    return 0;
}

/**
 * @cancel_hw_scan: Ask the low-level tp cancel the active hw scan.
 *	The driver should ask the hardware to cancel the scan (if possible),
 *	but the scan will be completed only after the driver will call
 *	ieee80211_scan_completed().
 *	This callback is needed for wowlan, to prevent enqueueing a new
 *	scan_work after the low-level driver was already suspended.
 *	The callback can sleep.
 */
static void rwnx_ops_cancel_hw_scan(struct ieee80211_hw *hw,
                                   struct ieee80211_vif *vif)
{
    struct scan_cancel_cfm scan_cancel_cfm;
    struct rwnx_hw *rwnx_hw = hw->priv;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 8, 0)
    struct cfg80211_scan_info info = {
        .aborted = true,
    };
#endif

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_send_scan_cancel_req(rwnx_hw, &scan_cancel_cfm);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 8, 0)
    ieee80211_scan_completed(hw, &info);
#else
    ieee80211_scan_completed(hw, true);
#endif
}

/**
 * @add_chanctx: Notifies device driver about new channel context creation.
 */
static int rwnx_ops_add_chanctx(struct ieee80211_hw *hw,
                                struct ieee80211_chanctx_conf *ctx)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_chanctx *rwnx_chanctx;
    struct mm_chan_ctxt_add_cfm add_chanctx_cfm;
    int error = 0;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_chanctx = (struct rwnx_chanctx *)ctx->drv_priv;

    rwnx_chanctx->ctx = ctx;

    /* Forward the information to the LMAC */
    if ((error = rwnx_send_add_chanctx(rwnx_hw, ctx, &add_chanctx_cfm)))
        return error;

    if (add_chanctx_cfm.status != 0) {
        RWNX_PRINT_CFM_ERR(add_chanctx);
        return -EIO;
    }

    INIT_LIST_HEAD(&rwnx_chanctx->list);
    INIT_LIST_HEAD(&rwnx_chanctx->vifs);

    /* Save the index retrieved from LMAC */
    rwnx_chanctx->index = add_chanctx_cfm.index;
    rwnx_chanctx->active = false;
    /* Add the new context in the list of used channel contexts */
    spin_lock_bh(&rwnx_hw->cb_lock);
    list_add_tail(&rwnx_chanctx->list, &rwnx_hw->chan_ctxts);
    spin_unlock_bh(&rwnx_hw->cb_lock);

    return 0;
}

/**
 * @remove_chanctx: Notifies device driver about channel context destruction.
 */
static void rwnx_ops_remove_chanctx(struct ieee80211_hw *hw,
                                    struct ieee80211_chanctx_conf *ctx)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_chanctx *rwnx_chanctx;
    struct rwnx_vif *rwnx_vif;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_chanctx = (struct rwnx_chanctx *)ctx->drv_priv;

    rwnx_chanctx->ctx = NULL;

    spin_lock_bh(&rwnx_hw->cb_lock);
    /* Flush the list of VIFS linked with this channel context */
    list_for_each_entry(rwnx_vif, &rwnx_chanctx->vifs, list) {
        list_del(&rwnx_vif->list_chan_ctxt);
    }
    /* Remove the context from the list of used channel contexts */
    list_del(&rwnx_chanctx->list);
    spin_unlock_bh(&rwnx_hw->cb_lock);

    /* Forward the information to the LMAC */
    rwnx_send_del_chanctx(rwnx_hw, rwnx_chanctx->index);
}

/**
 * @change_chanctx: Notifies device driver about channel context changes that
 *	may happen when combining different virtual interfaces on the same
 *	channel context with different settings
 */
static void rwnx_ops_change_chanctx(struct ieee80211_hw *hw,
                                    struct ieee80211_chanctx_conf *ctx,
                                    u32 changed)
{
    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_send_update_chanctx(hw->priv, ctx);
}

/**
 * @remain_on_channel: Starts an off-channel period on the given channel, must
 *	call back to ieee80211_ready_on_channel() when on that channel. Note
 *	that normal channel traffic is not stopped as this is intended for hw
 *	offload. Frames to transmit on the off-channel channel are transmitted
 *	normally except for the %IEEE80211_TX_CTL_TX_OFFCHAN flag. When the
 *	duration (which will always be non-zero) expires, the driver must call
 *	ieee80211_remain_on_channel_expired().
 *	Note that this callback may be called while the device is in IDLE and
 *	must be accepted in this case.
 *	This callback may sleep.
 */
static int rwnx_ops_remain_on_channel(struct ieee80211_hw *hw,
                                       struct ieee80211_vif *vif,
                                       struct ieee80211_channel *chan,
                                       int duration,
                                       enum ieee80211_roc_type type)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_vif *rwnx_vif;
    int error = 0;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw  = hw->priv;
    rwnx_vif = (struct rwnx_vif *)vif->drv_priv;

    /* Forward the information to the LMAC */
    error = rwnx_send_roc(rwnx_hw, rwnx_vif, chan, duration);

    return error;
}

/**
 * @cancel_remain_on_channel: Requests that an ongoing off-channel period is
 *	aborted before it expires. This callback may sleep.
 */
static int rwnx_ops_cancel_remain_on_channel(struct ieee80211_hw *hw)
{
    struct rwnx_hw *rwnx_hw;
    int error = 0;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw  = hw->priv;

    /* Forward the information to the LMAC */
    error = rwnx_send_cancel_roc(rwnx_hw);

    return error;
}

/**
 * @assign_vif_chanctx: Notifies device driver about channel context being
 *      bound to vif. Possible use is for hw queue remapping.
 */
static int rwnx_ops_assign_vif_chanctx(struct ieee80211_hw *hw,
                                       struct ieee80211_vif *vif,
                                       struct ieee80211_chanctx_conf *ctx)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_chanctx *rwnx_chanctx;
    struct rwnx_vif *rwnx_vif;
    int ret;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_chanctx = (struct rwnx_chanctx *)ctx->drv_priv;
    rwnx_vif = (struct rwnx_vif *)vif->drv_priv;

    WARN_ON(rwnx_vif->chanctx);

    /* Forward the information to the LMAC */
    ret = rwnx_send_link_chanctx(rwnx_hw, rwnx_vif->vif_index,
                                 rwnx_chanctx->index, 0);
    if (ret)
        return ret;

    spin_lock_bh(&rwnx_hw->cb_lock);
    /* Link the Channel Context with the VIF */
    rwnx_vif->chanctx = rwnx_chanctx;

    /* Insert the VIF in the list of VIFs linked with the channel context */
    list_add_tail(&rwnx_vif->list_chan_ctxt, &rwnx_chanctx->vifs);

    /* Start TXQ if it is the active channel */
    if (rwnx_chanctx->active) {
        rwnx_txq_vif_start(rwnx_vif, RWNX_TXQ_STOP_CHAN, rwnx_hw);
    }
    spin_unlock_bh(&rwnx_hw->cb_lock);

    return 0;
}

/**
 * @unassign_vif_chanctx: Notifies device driver about channel context being
 *	unbound from vif.
 */
static void rwnx_ops_unassign_vif_chanctx(struct ieee80211_hw *hw,
                                          struct ieee80211_vif *vif,
                                          struct ieee80211_chanctx_conf *ctx)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_vif *rwnx_vif;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_vif = (struct rwnx_vif *)vif->drv_priv;

    WARN_ON(!rwnx_vif->chanctx);

    spin_lock_bh(&rwnx_hw->cb_lock);
    /* Unlink the Channel Context with the VIF */
    rwnx_vif->chanctx = NULL;

    /* Remove the VIF from the list of VIFs linked with the channel context */
    list_del(&rwnx_vif->list_chan_ctxt);
    spin_unlock_bh(&rwnx_hw->cb_lock);

    /* Stop TXQ */
    rwnx_txq_vif_stop(rwnx_vif, RWNX_TXQ_STOP_CHAN, rwnx_hw);

    /* Forward the information to the LMAC */
    rwnx_send_unlink_chanctx(rwnx_hw, rwnx_vif->vif_index);
}

/**
 * @switch_vif_chanctx: switch a number of vifs from one chanctx to
 *	another, as specified in the list of
 *	@ieee80211_vif_chanctx_switch passed to the driver, according
 *	to the mode defined in &ieee80211_chanctx_switch_mode.
 */
int rwnx_ops_switch_vif_chanctx(struct ieee80211_hw *hw,
                                struct ieee80211_vif_chanctx_switch *vifs,
                                int n_vifs,
                                enum ieee80211_chanctx_switch_mode mode)
{
    struct ieee80211_chanctx_conf *new_ctx, *old_ctx;
    struct ieee80211_vif *vif;
    struct rwnx_hw *rwnx_hw = hw->priv;
    int i, ret;

    RWNX_DBG(RWNX_FN_ENTRY_STR);


    for (i = 0; i < n_vifs; i++) {
        new_ctx=vifs[i].new_ctx;
        old_ctx=vifs[i].old_ctx;
        vif=vifs[i].vif;

        if (mode == CHANCTX_SWMODE_SWAP_CONTEXTS){
            /* CHANCTX_SWMODE_SWAP_CONTEXTS means that mac80211 wasn't able to
             create a new ctxt (because no more ctxt can be created for this
             combination) and then it will swap current ctxt with a new one
             that currently "inactive" (and unknown to fw). After the swap the
             old context will be invalid and deleted.
             So first unassign and remove old ctxt (because we may have reached
             limit at fw level) and then create and assign a new one */
            rwnx_ops_unassign_vif_chanctx(hw, vif, old_ctx);
            rwnx_ops_remove_chanctx(hw, old_ctx);

            ret = rwnx_ops_add_chanctx(hw, new_ctx);
            if (ret)
                return ret;
            ret = rwnx_ops_assign_vif_chanctx(hw, vif, new_ctx);
            if (ret)
                return ret;
        } else {
            /* CHANCTX_SWMODE_REASSIGN_VIF means that mac80211 already create
               the new ctxt (i.e already call add_chanctx callback). The old
               ctxt will be be valid after this call (obviously if it is no
               longer used by other vif it will be removed with remove_chanctx
               callback). In this case simply tell fw to assign the new ctxt to
               this vif and unassign old ctxt.
              */
            struct rwnx_vif *rwnx_vif = (struct rwnx_vif *)vif->drv_priv;
            struct rwnx_chanctx *rwnx_chanctx = (struct rwnx_chanctx *)new_ctx->drv_priv;
            WARN_ON(!rwnx_vif->chanctx);

            ret = rwnx_send_link_chanctx(rwnx_hw, rwnx_vif->vif_index,
                                         rwnx_chanctx->index, 1);
            if (ret)
                return ret;

            spin_lock_bh(&rwnx_hw->cb_lock);
            /* Link the Channel Context with the VIF */
            rwnx_vif->chanctx = rwnx_chanctx;

            /* Remove the VIF from the list of VIFs linked with the old channel context */
            list_del(&rwnx_vif->list_chan_ctxt);

            /* Insert the VIF in the list of VIFs linked with the new channel context */
            list_add_tail(&rwnx_vif->list_chan_ctxt, &rwnx_chanctx->vifs);

            /* Update TXQ's status regarding vif's channel context */
            if (rwnx_chanctx->active) {
                rwnx_txq_vif_start(rwnx_vif, RWNX_TXQ_STOP_CHAN, rwnx_hw);
            } else {
                rwnx_txq_vif_stop(rwnx_vif, RWNX_TXQ_STOP_CHAN, rwnx_hw);
            }
            spin_unlock_bh(&rwnx_hw->cb_lock);
        }
    }

    return 0;
}

/**
  * @mgd_prepare_tx: Prepare for transmitting a management frame for association
 *	before associated. In multi-channel scenarios, a virtual interface is
 *	bound to a channel before it is associated, but as it isn't associated
 *	yet it need not necessarily be given airtime, in particular since any
 *	transmission to a P2P GO needs to be synchronized against the GO's
 *	powersave state. mac80211 will call this function before transmitting a
 *	management frame prior to having successfully associated to allow the
 *	driver to give it channel time for the transmission, to get a response
 *	and to be able to synchronize with the GO.
 *	For drivers that set %IEEE80211_HW_DEAUTH_NEED_MGD_TX_PREP, mac80211
 *	would also call this function before transmitting a deauthentication
 *	frame in case that no beacon was heard from the AP/P2P GO.
 *	The callback will be called before each transmission and upon return
 *	mac80211 will transmit the frame right away.
 *      If duration is greater than zero, mac80211 hints to the driver the
 *      duration for which the operation is requested.
 *	The callback is optional and can (should!) sleep.
 */
static void rwnx_ops_mgd_prepare_tx(struct ieee80211_hw *hw,
                                    struct ieee80211_vif *vif,
                                    u16 duration)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_vif *rwnx_vif;
    struct rwnx_chanctx *rwnx_chanctx;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_vif = (struct rwnx_vif *)vif->drv_priv;
    rwnx_chanctx = (struct rwnx_chanctx *)vif->chanctx_conf->drv_priv;

    rwnx_send_sched_chanctx(rwnx_hw, rwnx_vif->vif_index,
                            rwnx_chanctx->index, 1);
}

/**
 * @set_tim: Set TIM bit. mac80211 calls this function when a TIM bit
 *	must be set or cleared for a given STA. Must be atomic.
 */
static int rwnx_ops_set_tim(struct ieee80211_hw *hw, struct ieee80211_sta *sta,
                            bool set)
{
    struct rwnx_hw *rwnx_hw;
    struct rwnx_sta *rwnx_sta;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_hw = hw->priv;
    rwnx_sta = (struct rwnx_sta *)sta->drv_priv;

    /* Forward the information to the LMAC */
    return rwnx_send_tim_update(rwnx_hw, rwnx_sta->vif_idx, sta->aid, set?1:0);
}

/**
 * @get_survey: Return per-channel survey information
 */
static int rwnx_ops_get_survey(struct ieee80211_hw *hw, int idx,
                               struct survey_info *survey)
{
    struct rwnx_hw *rwnx_hw = hw->priv;
    struct ieee80211_supported_band *sband;
    struct rwnx_survey_info *rwnx_survey;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    if (idx >= ARRAY_SIZE(rwnx_hw->survey))
        return -ENOENT;

    rwnx_survey = &rwnx_hw->survey[idx];

    // Check if provided index matches with a supported 2.4GHz channel
    sband = hw->wiphy->bands[NL80211_BAND_2GHZ];
    if (sband && idx >= sband->n_channels) {
        idx -= sband->n_channels;
        sband = NULL;
    }

    if (!sband) {
        // Check if provided index matches with a supported 5GHz channel
        sband = hw->wiphy->bands[NL80211_BAND_5GHZ];

        if (!sband || idx >= sband->n_channels)
            return -ENOENT;
    }

    // Fill the survey
    survey->channel = &sband->channels[idx];
    survey->filled = rwnx_survey->filled;

    if (rwnx_survey->filled != 0) {
        SURVEY_TIME(survey) = (u64)rwnx_survey->chan_time_ms;
        SURVEY_TIME_BUSY(survey) = (u64)rwnx_survey->chan_time_busy_ms;
        survey->noise = rwnx_survey->noise_dbm;

        // Set the survey report as not used
        rwnx_survey->filled = 0;
    }

    return 0;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 19, 0)
/**
 * @tdls_channel_switch: enable TDLS channel switch.
 */
static int rwnx_ops_tdls_channel_switch(struct ieee80211_hw *hw,
                                        struct ieee80211_vif *vif,
                                        struct ieee80211_sta *sta, u8 oper_class,
                                        struct cfg80211_chan_def *chandef,
                                        struct sk_buff *tmpl_skb, u32 ch_sw_tm_ie_off)
{
    struct rwnx_hw *rwnx_hw = hw->priv;
    struct rwnx_vif *rwnx_vif = (struct rwnx_vif *)vif->drv_priv;
    struct rwnx_sta *rwnx_sta = (struct rwnx_sta *)sta->drv_priv;
    struct tdls_chan_switch_cfm cfm;
    int error;

    error = rwnx_send_tdls_chan_switch_req(rwnx_hw, rwnx_vif, rwnx_sta,
                                           STA_TDLS_INITIATOR(sta), oper_class,
                                           chandef, &cfm);
    if (error)
        return error;

    if (!cfm.status) {
        // Set channel switch flags
        rwnx_sta->tdls.chsw_en = true;
        rwnx_hw->tdls_info.chsw_en = true;
    }

    return cfm.status;
}

/**
 * @tdls_cancel_channel_switch: disable TDLS channel switch.
 */
static void rwnx_ops_tdls_cancel_channel_switch(struct ieee80211_hw *hw,
        struct ieee80211_vif *vif,
        struct ieee80211_sta *sta)
{
    struct rwnx_hw *rwnx_hw = hw->priv;
    struct rwnx_vif *rwnx_vif = (struct rwnx_vif *)vif->drv_priv;
    struct rwnx_sta *rwnx_sta = (struct rwnx_sta *)sta->drv_priv;
    struct tdls_cancel_chan_switch_cfm cfm;

    if (!rwnx_send_tdls_cancel_chan_switch_req(rwnx_hw, rwnx_vif,
                                               rwnx_sta, &cfm)) {
        // Set channel switch flags
        rwnx_sta->tdls.chsw_en = false;
        rwnx_hw->tdls_info.chsw_en = false;
    }
}

/**
 * @tdls_recv_channel_switch: a TDLS channel-switch related frame (request or
 *  response) has been received from a remote peer.
 */
static void rwnx_ops_tdls_recv_channel_switch(struct ieee80211_hw *hw,
        struct ieee80211_vif *vif,
        struct ieee80211_tdls_ch_sw_params *params)
{
    return;
}
#endif /* version >= 3.19 */

/**
 * @reconfig_complete: Called after a call to ieee80211_restart_hw() and
 *	during resume, when the reconfiguration has completed.
 *	This can help the driver implement the reconfiguration step (and
 *	indicate mac80211 is ready to receive frames).
 *	This callback may sleep.
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 19, 0)
static void rwnx_reconfig_complete(struct ieee80211_hw *hw,
                                   enum ieee80211_reconfig_type type)
{
    struct rwnx_hw *rwnx_hw = hw->priv;
    if (!WARN_ON(type != IEEE80211_RECONFIG_TYPE_RESTART))
    clear_bit(RWNX_DEV_STACK_RESTARTING, &rwnx_hw->drv_flags);
}
#else
static void rwnx_restart_complete(struct ieee80211_hw *hw)
{
    struct rwnx_hw *rwnx_hw = hw->priv;
    clear_bit(RWNX_DEV_STACK_RESTARTING, &rwnx_hw->drv_flags);
}
#endif /* version >= 3.19 */

struct ieee80211_ops rwnx_ops = {
    .tx                       = rwnx_ops_tx,
#ifdef CONFIG_RWNX_AMSDUS_TX
    .agg_msdu                 = rwnx_ops_agg_msdu,
#endif
    .start                    = rwnx_ops_start,
    .stop                     = rwnx_ops_stop,
#ifdef CONFIG_PM_SLEEP
    /* When Wake On WLAN -WoWLAN - is supported:
     * suspend,
     * resume
     */
#endif
    .add_interface            = rwnx_ops_add_interface,
    .remove_interface         = rwnx_ops_remove_interface,
    .flush                    = rwnx_ops_flush,
    .config                   = rwnx_ops_config,
    .bss_info_changed         = rwnx_ops_bss_info_changed,
    .prepare_multicast        = rwnx_prepare_multicast,
    .configure_filter         = rwnx_ops_configure_filter,
    .set_key                  = rwnx_ops_set_key,
    .sta_add                  = rwnx_ops_sta_add,
    .sta_remove               = rwnx_ops_sta_remove,
    .sta_notify               = rwnx_ops_sta_notify,
    .conf_tx                  = rwnx_ops_conf_tx,
    .ampdu_action             = rwnx_ops_ampdu_action,
    CFG80211_TESTMODE_CMD(rwnx_op_testmode_cmd)
    .tx_frames_pending        = rwnx_ops_tx_frames_pending,
    .sw_scan_start            = rwnx_ops_sw_scan_start,
    .sw_scan_complete         = rwnx_ops_sw_scan_complete,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 19, 0)
    .reconfig_complete        = rwnx_reconfig_complete,
    .tdls_channel_switch      = rwnx_ops_tdls_channel_switch,
    .tdls_cancel_channel_switch = rwnx_ops_tdls_cancel_channel_switch,
    .tdls_recv_channel_switch = rwnx_ops_tdls_recv_channel_switch,
#else
    .restart_complete         = rwnx_restart_complete,
#endif
#ifdef CONFIG_MAC80211_TXQ
    .wake_tx_queue            = rwnx_ops_wake_tx_queue,
#endif
};

static void rwnx_reg_notifier(struct wiphy *wiphy,
                              struct regulatory_request *request)
{
    struct ieee80211_hw *hw = wiphy_to_ieee80211_hw(wiphy);
    struct rwnx_hw *rwnx_hw = hw->priv;

    rwnx_radar_set_domain(&rwnx_hw->radar, request->dfs_region);
}

static void rwnx_set_vers(struct rwnx_hw *rwnx_hw)
{
    u32 vers = rwnx_hw->version_cfm.version_lmac;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    snprintf(rwnx_hw->hw->wiphy->fw_version,
             sizeof(rwnx_hw->hw->wiphy->fw_version), "%d.%d.%d.%d",
             (vers & (0xff << 24)) >> 24, (vers & (0xff << 16)) >> 16,
             (vers & (0xff <<  8)) >>  8, (vers & (0xff <<  0)) >>  0);
}

void rwnx_enable_wapi(struct rwnx_hw *rwnx_hw)
{
    struct ieee80211_hw *hw = rwnx_hw->hw;
    hw->n_cipher_schemes = ARRAY_SIZE(rwnx_cs);
    hw->cipher_schemes = rwnx_cs;
    hw->wiphy->flags |= WIPHY_FLAG_CONTROL_PORT_PROTOCOL;
}


void rwnx_enable_mesh(struct rwnx_hw *rwnx_hw)
{
    struct ieee80211_hw *hw = rwnx_hw->hw;

    if (!rwnx_mod_params.mesh)
        return;

    hw->wiphy->interface_modes |= BIT(NL80211_IFTYPE_MESH_POINT);
    rwnx_limits[0].types |= BIT(NL80211_IFTYPE_MESH_POINT);
    rwnx_limits_dfs[0].types |= BIT(NL80211_IFTYPE_MESH_POINT);
}

/**
 *
 */
int rwnx_mac80211_init(struct rwnx_plat *rwnx_plat, void **platform_data)
{
    struct rwnx_hw *rwnx_hw;
    struct ieee80211_hw *hw;
    struct rwnx_conf_file init_conf;
    int ret = 0;
    int i;

    RWNX_DBG(RWNX_FN_ENTRY_STR);

    if (rwnx_mod_params.hwscan) {
        rwnx_combinations[0].num_different_channels = 3;
        rwnx_ops.hw_scan              = rwnx_ops_hw_scan;
        rwnx_ops.cancel_hw_scan       = rwnx_ops_cancel_hw_scan,
        rwnx_ops.add_chanctx          = rwnx_ops_add_chanctx;
        rwnx_ops.remove_chanctx       = rwnx_ops_remove_chanctx;
        rwnx_ops.change_chanctx       = rwnx_ops_change_chanctx;
        rwnx_ops.assign_vif_chanctx   = rwnx_ops_assign_vif_chanctx;
        rwnx_ops.unassign_vif_chanctx = rwnx_ops_unassign_vif_chanctx;
        rwnx_ops.switch_vif_chanctx   = rwnx_ops_switch_vif_chanctx;
        rwnx_ops.mgd_prepare_tx       = rwnx_ops_mgd_prepare_tx;
        rwnx_ops.remain_on_channel        = rwnx_ops_remain_on_channel;
        rwnx_ops.cancel_remain_on_channel = rwnx_ops_cancel_remain_on_channel;
        rwnx_ops.get_survey               = rwnx_ops_get_survey;
    }

    if (rwnx_mod_params.autobcn) {
        rwnx_ops.set_tim = rwnx_ops_set_tim;
    }

    hw = ieee80211_alloc_hw(sizeof(struct rwnx_hw), &rwnx_ops);
    if (!hw) {
        dev_err(rwnx_platform_get_dev(rwnx_plat), "ieee80211_alloc_hw failed\n");
        ret = -ENOMEM;
        goto err_out;
    }

    rwnx_hw = hw->priv;
    rwnx_hw->hw = hw;
    rwnx_hw->plat = rwnx_plat;
    rwnx_hw->dev = rwnx_platform_get_dev(rwnx_plat);
    SET_IEEE80211_DEV(hw, rwnx_hw->dev);
    rwnx_hw->mod_params = &rwnx_mod_params;
    rwnx_hw->tcp_pacing_shift = 7;

    /* Create cache to allocate sw_txhdr */
    rwnx_hw->sw_txhdr_cache = KMEM_CACHE(rwnx_sw_txhdr, 0);
    if (!rwnx_hw->sw_txhdr_cache) {
        wiphy_err(hw->wiphy, "Cannot allocate cache for sw TX header\n");
        ret = -ENOMEM;
        goto err_cache;
    }

    if ((ret = rwnx_parse_configfile(rwnx_hw, RWNX_CONFIG_FW_NAME, &init_conf))) {
        wiphy_err(hw->wiphy, "rwnx_parse_configfile failed\n");
        goto err_config;
    }

    rwnx_hw->scan_ie.addr = NULL;

    SET_IEEE80211_PERM_ADDR(hw, init_conf.mac_addr);
    /* ATM - NX_VIRT_DEV_MAX is a power of 2
     *     - mac80211 doesn't handle non-contiguous masks */
    if (!WARN_ON(NX_VIRT_DEV_MAX & (NX_VIRT_DEV_MAX - 1)))
        *(u32 *)(hw->wiphy->addr_mask + 2) = cpu_to_be32(NX_VIRT_DEV_MAX - 1);

    ieee80211_hw_set(hw, SIGNAL_DBM);
    ieee80211_hw_set(hw, REPORTS_TX_ACK_STATUS);
    ieee80211_hw_set(hw, QUEUE_CONTROL);
    ieee80211_hw_set(hw, WANT_MONITOR_VIF);
    ieee80211_hw_set(hw, SPECTRUM_MGMT);
    ieee80211_hw_set(hw, SUPPORTS_HT_CCK_RATES);
    ieee80211_hw_set(hw, SUPPORT_FAST_XMIT);
#ifdef CONFIG_MAC80211_AMSDUS_TX
    ieee80211_hw_set(hw, TX_AMSDU);
    ieee80211_hw_set(hw, TX_FRAG_LIST);
    hw->max_tx_fragments = rwnx_hw->mod_params->amsdu_maxnb;
#endif

    hw->wiphy->bands[NL80211_BAND_2GHZ] = &rwnx_band_2GHz;
    hw->wiphy->bands[NL80211_BAND_5GHZ] = &rwnx_band_5GHz;
    hw->wiphy->interface_modes =
        BIT(NL80211_IFTYPE_STATION)     |
        BIT(NL80211_IFTYPE_AP)          |
        // If set, wpa_supplicant always create a dedicated interface for p2p
        //BIT(NL80211_IFTYPE_P2P_DEVICE)  |
        BIT(NL80211_IFTYPE_P2P_CLIENT)  |
        BIT(NL80211_IFTYPE_P2P_GO);
    hw->wiphy->flags |= WIPHY_FLAG_HAS_REMAIN_ON_CHANNEL |
                        WIPHY_FLAG_HAS_CHANNEL_SWITCH;

    hw->wiphy->max_remain_on_channel_duration = rwnx_hw->mod_params->roc_dur_max;
    hw->wiphy->features |= NL80211_FEATURE_AP_MODE_CHAN_WIDTH_CHANGE;

    if (rwnx_mod_params.tdls) {
        /* TDLS support */
        hw->wiphy->features |= NL80211_FEATURE_TDLS_CHANNEL_SWITCH;
    }

    hw->wiphy->iface_combinations   = rwnx_combinations;
    /* -1 not to include combination with radar detection, will be re-added in
       rwnx_handle_dynparams if supported */
    hw->wiphy->n_iface_combinations = ARRAY_SIZE(rwnx_combinations) - 1;
    hw->wiphy->reg_notifier = rwnx_reg_notifier;

    BUILD_BUG_ON(RWNX_TX_MAX_RATES > NX_TX_MAX_RATES);
    hw->max_rates      = hw->max_report_rates = min(RWNX_TX_MAX_RATES,
                                                    IEEE80211_TX_MAX_RATES);
    hw->max_rate_tries = 1;

    hw->max_listen_interval = rwnx_mod_params.listen_itv;

    hw->max_rx_aggregation_subframes = IEEE80211_MAX_AMPDU_BUF;

    hw->vif_data_size     = sizeof(struct rwnx_vif);
    hw->sta_data_size     = sizeof(struct rwnx_sta);
#ifdef CONFIG_MAC80211_TXQ
    hw->txq_data_size     = sizeof(struct rwnx_txq);
#endif
    hw->chanctx_data_size = sizeof(struct rwnx_chanctx);
    hw->extra_tx_headroom = sizeof(struct rwnx_txhdr) + RWNX_SWTXHDR_ALIGN_SZ;
    hw->queues                 = RWNX_HWQ_NB;
    hw->offchannel_tx_hw_queue = RWNX_HWQ_VO;
    // Set UAPSD queues
    hw->uapsd_queues = (rwnx_mod_params.uapsd_queues & IEEE80211_WMM_IE_STA_QOSINFO_AC_MASK);

    tasklet_init(&rwnx_hw->task, rwnx_task, (unsigned long)rwnx_hw);

    INIT_LIST_HEAD(&rwnx_hw->vifs);
    INIT_LIST_HEAD(&rwnx_hw->chan_ctxts);

    rwnx_hwq_init(rwnx_hw);

    for (i = 0; i < SCAN_CHANNEL_MAX; i++) {
        rwnx_hw->survey[i].filled = 0;
    }

    rwnx_mu_group_init(rwnx_hw);

    mutex_init(&rwnx_hw->mutex);
    mutex_init(&rwnx_hw->dbgdump_elem.mutex);
    spin_lock_init(&rwnx_hw->tx_lock);
    spin_lock_init(&rwnx_hw->cb_lock);
#ifdef CONFIG_RWNX_AMSDUS_TX
    spin_lock_init(&rwnx_hw->amsdu_lock);
#endif

    if ((ret = rwnx_platform_on(rwnx_hw, NULL)))
        goto err_platon;

    /* Reset LMAC */
    if ((ret = rwnx_send_reset(rwnx_hw)))
        goto err_lmac_reqs;
    if ((ret = rwnx_send_version_req(rwnx_hw, &rwnx_hw->version_cfm)))
        goto err_lmac_reqs;

    rwnx_set_vers(rwnx_hw);

    if ((ret = rwnx_handle_dynparams(rwnx_hw, hw->wiphy)))
        goto err_lmac_reqs;

    rwnx_enable_mesh(rwnx_hw);
    rwnx_radar_detection_init(&rwnx_hw->radar);

    /* ieee80211_register_hw() will take care of calling wiphy_register() and
     * also ieee80211_if_add() (because IFTYPE_STATION is supported)
     * which will internally call register_netdev() */
    if ((ret = ieee80211_register_hw(hw))) {
        wiphy_err(hw->wiphy, "Could not register ieee80211 device (err=%d)\n",
                  ret);
        goto err_register_hw;
    }
    rwnx_custregd(rwnx_hw, hw->wiphy);

    *platform_data = rwnx_hw;

    if ((ret = rwnx_dbgfs_register(rwnx_hw, "rwnx"))) {
        wiphy_err(hw->wiphy, "Failed to register debugfs entries");
        goto err_debugfs;
    }

    return 0;

err_debugfs:
err_register_hw:
err_lmac_reqs:
    rwnx_platform_off(rwnx_hw, NULL);
err_platon:
err_config:
    kmem_cache_destroy(rwnx_hw->sw_txhdr_cache);
err_cache:
    ieee80211_free_hw(hw);
err_out:
    return ret;
}

/**
 *
 */
void rwnx_mac80211_deinit(struct rwnx_hw *rwnx_hw)
{
    RWNX_DBG(RWNX_FN_ENTRY_STR);

    rwnx_dbgfs_unregister(rwnx_hw);
    ieee80211_unregister_hw(rwnx_hw->hw);
    rwnx_radar_detection_deinit(&rwnx_hw->radar);
    rwnx_platform_off(rwnx_hw, NULL);
    kmem_cache_destroy(rwnx_hw->sw_txhdr_cache);
    ieee80211_free_hw(rwnx_hw->hw);
}

static void __used rwnx_restart_hw(struct rwnx_hw *rwnx_hw)
{
    struct rwnx_vif *rwnx_vif, *__rwnx_vif;
    int ret;
    unsigned long now = jiffies;
    static struct {
        unsigned long last;
        unsigned int cnt;
    } restart_recs = { .last = 0, .cnt = 0 };

    printk(RWNX_FN_ENTRY_STR);

    rwnx_hw->drv_flags = BIT(RWNX_DEV_RESTARTING);

    mutex_lock(&rwnx_hw->dbgdump_elem.mutex);

    rwnx_platform_off(rwnx_hw, NULL);

    rwnx_hw->debugfs.trace_prst = false;
    mutex_unlock(&rwnx_hw->dbgdump_elem.mutex);

    ieee80211_stop_queues(rwnx_hw->hw);

    if (restart_recs.cnt) {
        if (jiffies_to_msecs(now - restart_recs.last) > 3000) {
            restart_recs.cnt = 0;
        } else if (restart_recs.cnt > 5) {
            printk(KERN_CRIT "%s: Too many failures .. aborting\n", __func__);
            return;
        }
    }
    restart_recs.cnt++;
    restart_recs.last = now;

    if (rwnx_platform_on(rwnx_hw, NULL)) {
        printk(KERN_CRIT "%s: Couldn't turn platform on .. aborting\n", __func__);
        return;
    }

    if ((ret = rwnx_send_reset(rwnx_hw)))  {
        printk(KERN_CRIT "%s: Couldn't reset the LMAC .. aborting\n", __func__);
        return;
    }

    /* driver housekeeping */
    list_for_each_entry_safe(rwnx_vif, __rwnx_vif, &rwnx_hw->vifs, list) {
        list_del(&rwnx_vif->list);
        rwnx_vif->vif = NULL;
    }

    set_bit(RWNX_DEV_STACK_RESTARTING, &rwnx_hw->drv_flags);
    clear_bit(RWNX_DEV_RESTARTING, &rwnx_hw->drv_flags);

    /* hand over to mac80211 from here */
    ieee80211_restart_hw(rwnx_hw->hw);
}

/**
 *
 */
static int __init rwnx_mod_init(void)
{
    RWNX_DBG(RWNX_FN_ENTRY_STR);
    rwnx_print_version();
    return rwnx_platform_register_drv();
}

/**
 *
 */
static void __exit rwnx_mod_exit(void)
{
    RWNX_DBG(RWNX_FN_ENTRY_STR);
    rwnx_platform_unregister_drv();
}

module_init(rwnx_mod_init);
module_exit(rwnx_mod_exit);

MODULE_FIRMWARE(RWNX_CONFIG_FW_NAME);

MODULE_DESCRIPTION(RW_DRV_DESCRIPTION);
MODULE_VERSION(RWNX_VERS_MOD);
MODULE_AUTHOR(RW_DRV_COPYRIGHT " " RW_DRV_AUTHOR);
MODULE_LICENSE("GPL");
