/**
 *****************************************************************************
 *
 * @file rwnx_rx.c
 *
 * Copyright (C) RivieraWaves 2012-2019
 *
 *****************************************************************************
 */
#include <linux/dma-mapping.h>
#include <net/mac80211.h>

#include "rwnx_defs.h"
#include "rwnx_rx.h"
#include "hal_desc.h"
#include "rwnx_prof.h"
#include "ipc_host.h"
#include "rwnx_compat.h"

static const u8 legrates_lut[] = {
    0,                          /* 0 */
    1,                          /* 1 */
    2,                          /* 2 */
    3,                          /* 3 */
    -1,                         /* 4 */
    -1,                         /* 5 */
    -1,                         /* 6 */
    -1,                         /* 7 */
    10,                         /* 8 */
    8,                          /* 9 */
    6,                          /* 10 */
    4,                          /* 11 */
    11,                         /* 12 */
    9,                          /* 13 */
    7,                          /* 14 */
    5                           /* 15 */
};

/**
 * rwnx_rx_vector_convert - Convert a legacy RX vector into a new RX vector format
 *
 * @rwnx_hw: main driver data.
 * @rx_vect1: Rx vector 1 descriptor of the received frame.
 * @rx_vect2: Rx vector 2 descriptor of the received frame.
 */
static void rwnx_rx_vector_convert(struct rwnx_hw *rwnx_hw,
                                   struct rx_vector_1 *rx_vect1,
                                   struct rx_vector_2 *rx_vect2)
{
    struct rx_vector_1_old rx_vect1_leg;
    struct rx_vector_2_old rx_vect2_leg;
    u32_l phy_vers = rwnx_hw->version_cfm.version_phy_2;

    // Check if we need to do the conversion. Only if old modem is used
    if (__MDM_MAJOR_VERSION(phy_vers) > 0) {
        rx_vect1->rssi1 = rx_vect1->rssi_leg;
        return;
    }

    // Copy the received vector locally
    memcpy(&rx_vect1_leg, rx_vect1, sizeof(struct rx_vector_1_old));

    // Reset it
    memset(rx_vect1, 0, sizeof(struct rx_vector_1));

    // Perform the conversion
    rx_vect1->format_mod = rx_vect1_leg.format_mod;
    rx_vect1->ch_bw = rx_vect1_leg.ch_bw;
    rx_vect1->antenna_set = rx_vect1_leg.antenna_set;
    rx_vect1->leg_length = rx_vect1_leg.leg_length;
    rx_vect1->leg_rate = rx_vect1_leg.leg_rate;
    rx_vect1->rssi1 = rx_vect1_leg.rssi1;

    switch (rx_vect1->format_mod) {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
            rx_vect1->leg.lsig_valid = rx_vect1_leg.lsig_valid;
            rx_vect1->leg.chn_bw_in_non_ht = rx_vect1_leg.num_extn_ss;
            rx_vect1->leg.dyn_bw_in_non_ht = rx_vect1_leg.dyn_bw;
            break;
        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
            rx_vect1->ht.aggregation = rx_vect1_leg.aggregation;
            rx_vect1->ht.fec = rx_vect1_leg.fec_coding;
            rx_vect1->ht.lsig_valid = rx_vect1_leg.lsig_valid;
            rx_vect1->ht.length = rx_vect1_leg.ht_length;
            rx_vect1->ht.mcs = rx_vect1_leg.mcs;
            rx_vect1->ht.num_extn_ss = rx_vect1_leg.num_extn_ss;
            rx_vect1->ht.short_gi = rx_vect1_leg.short_gi;
            rx_vect1->ht.smoothing = rx_vect1_leg.smoothing;
            rx_vect1->ht.sounding = rx_vect1_leg.sounding;
            rx_vect1->ht.stbc = rx_vect1_leg.stbc;
            break;
        case FORMATMOD_VHT:
            rx_vect1->vht.beamformed = !rx_vect1_leg.smoothing;
            rx_vect1->vht.fec = rx_vect1_leg.fec_coding;
            rx_vect1->vht.length = rx_vect1_leg.ht_length | rx_vect1_leg._ht_length << 8;
            rx_vect1->vht.mcs = rx_vect1_leg.mcs & 0x0F;
            rx_vect1->vht.nss = rx_vect1_leg.stbc ? rx_vect1_leg.n_sts/2 : rx_vect1_leg.n_sts;
            rx_vect1->vht.doze_not_allowed = rx_vect1_leg.doze_not_allowed;
            rx_vect1->vht.short_gi = rx_vect1_leg.short_gi;
            rx_vect1->vht.sounding = rx_vect1_leg.sounding;
            rx_vect1->vht.stbc = rx_vect1_leg.stbc;
            rx_vect1->vht.group_id = rx_vect1_leg.group_id;
            rx_vect1->vht.partial_aid = rx_vect1_leg.partial_aid;
            rx_vect1->vht.first_user = rx_vect1_leg.first_user;
            break;
    }

    if (!rx_vect2)
        return;

    // Copy the received vector 2 locally
    memcpy(&rx_vect2_leg, rx_vect2, sizeof(struct rx_vector_2_old));

    // Reset it
    memset(rx_vect2, 0, sizeof(struct rx_vector_2));

    rx_vect2->rcpi1 = rx_vect2_leg.rcpi;
    rx_vect2->rcpi2 = rx_vect2_leg.rcpi;
    rx_vect2->rcpi3 = rx_vect2_leg.rcpi;
    rx_vect2->rcpi4 = rx_vect2_leg.rcpi;

    rx_vect2->evm1 = rx_vect2_leg.evm1;
    rx_vect2->evm2 = rx_vect2_leg.evm2;
    rx_vect2->evm3 = rx_vect2_leg.evm3;
    rx_vect2->evm4 = rx_vect2_leg.evm4;
}

/**
 *
 */
void rwnx_fill_rxstatus_rxvect_phyinfo(struct ieee80211_rx_status *status,
                                       struct rx_vector_1 *rxvect,
                                       struct phy_channel_info_desc* phy_info)
{
    int nb_chain, signal;

    /* check for HE frames */
    if (rxvect->format_mod > FORMATMOD_VHT) {

        status->rate_idx = rxvect->he.mcs;
        RX_ENC_HE(status);
        RX_NSS(status) = rxvect->he.nss + 1;

        /* check RX_FLAG_LDPC */
        if (rxvect->he.fec)
            RX_ENC_FLAG_LDPC(status);

    /* check for VHT frames */
    } else if (rxvect->format_mod > FORMATMOD_HT_GF) {

        status->rate_idx = rxvect->vht.mcs;
        RX_ENC_VHT(status);
        RX_NSS(status) = rxvect->vht.nss + 1;

        /* check RX_FLAG_GI */
        if (rxvect->vht.short_gi)
            RX_ENC_FLAG_SHORT_GI(status);

        /* check RX_FLAG_LDPC */
        if (rxvect->vht.fec)
            RX_ENC_FLAG_LDPC(status);

    /* check for HT frames */
    } else if (rxvect->format_mod > FORMATMOD_NON_HT_DUP_OFDM) {

        status->rate_idx = rxvect->ht.mcs;

        /* check RX_FLAG_{HT,HT_GF} */
        if (rxvect->format_mod == FORMATMOD_HT_MF)
            RX_ENC_HT(status);
        else
            RX_ENC_HT_GF(status);

        /* check RX_FLAG_GI */
        if (rxvect->ht.short_gi)
            RX_ENC_FLAG_SHORT_GI(status);

        /* check RX_FLAG_LDPC */
        if (rxvect->ht.fec)
            RX_ENC_FLAG_LDPC(status);

    } else {
        BUG_ON((status->rate_idx = legrates_lut[rxvect->leg_rate]) == -1);
        if (phy_info->phy_band == NL80211_BAND_5GHZ)
            status->rate_idx -= 4;  /* rwnx_ratetable_5ghz[0].hw_value == 4 */
        /* check RX_FLAG_SHORTPRE */
        if (!rxvect->pre_type)
            RX_ENC_FLAG_SHORT_PRE(status);
    }

    /* check RX_FLAG_{40,80,160,80P80}MHZ */
    if (rxvect->ch_bw == PHY_CHNL_BW_40)
        RX_BW_40MHZ(status);
    else if (rxvect->ch_bw == PHY_CHNL_BW_80)
        RX_BW_80MHZ(status);
    else if (rxvect->ch_bw == PHY_CHNL_BW_160)
        RX_BW_160MHZ(status);

    status->band = phy_info->phy_band;
    status->freq = phy_info->phy_prim20_freq;

#define SET_CHAIN_SIGNAL(c)                             \
    if (rxvect->antenna_set & BIT(c - 1)) {             \
        nb_chain++;                                     \
        status->chain_signal[c - 1] = rxvect->rssi1;    \
        signal += rxvect->rssi1;                        \
    }

    nb_chain = 0;
    signal = 0;
    SET_CHAIN_SIGNAL(1);
    SET_CHAIN_SIGNAL(2);
    SET_CHAIN_SIGNAL(3);
    SET_CHAIN_SIGNAL(4);
#undef SET_CHAIN_SIGNAL

    status->chains = rxvect->antenna_set;
    status->signal = signal / nb_chain;
}

/**
 *
 */
static inline int rwnx_fill_rxstatus(struct rwnx_hw *rwnx_hw,
                                     struct ieee80211_rx_status *status,
                                     struct hw_rxhdr *rxhdr)
{
    int ret = 0;
    struct hw_vect *hw_vect = &rxhdr->hwvect;
    bool aggregation = false;

    memset(status, 0, sizeof(struct ieee80211_rx_status));

    if (unlikely(!hw_vect->frm_successful_rx)) {
        RWNX_DBG("addr_mismatch:%d fcs_err:%d phy_err:%d undef_err:%d rxfifo_oflow:%d\n",
                 hw_vect->addr_mismatch, hw_vect->fcs_err, hw_vect->phy_err,
                 hw_vect->undef_err, hw_vect->rx_fifo_oflow);

        /* As of now errors we get should only be fcs errors (c.f hw ppl) */
        if (hw_vect->fcs_err)
            status->flag |= RX_FLAG_FAILED_FCS_CRC;
        else                    /* ??? */
            status->flag |= RX_FLAG_FAILED_PLCP_CRC;

        return -1;
    }

    status->mactime =
        le64_to_cpu(((u64)(hw_vect->tsf_hi) << 32) | hw_vect->tsf_lo);
    status->flag |= RX_FLAG_MACTIME_END;

    rwnx_fill_rxstatus_rxvect_phyinfo(status, &(hw_vect->rx_vect1), &(rxhdr->phy_info));
    /* check for HT or VHT frames */
    if (hw_vect->rx_vect1.format_mod >= FORMATMOD_VHT) {
        aggregation = true;
    } else if (hw_vect->rx_vect1.format_mod > FORMATMOD_NON_HT_DUP_OFDM) {
        aggregation = hw_vect->rx_vect1.ht.aggregation;
    }
    if (aggregation) {
        int idx = hw_vect->mpdu_cnt;
        status->flag |= RX_FLAG_AMPDU_DETAILS;
        status->ampdu_reference = hw_vect->ampdu_cnt;
        if (likely(idx < IEEE80211_MAX_AMPDU_BUF))
            rwnx_hw->stats.rx_in_ampdu[idx].cnt++;
        else
            printk(KERN_CRIT "%s:%d: mpdu_cnt = 0x%08x\n", __func__, __LINE__, idx);
    }

    switch (hw_vect->decr_status) {
    case RWNX_RX_HD_DECR_UNENC:
        /* XXX drop if protected frame ? */
        break;

    case RWNX_RX_HD_DECR_ICVFAIL:
    case RWNX_RX_HD_DECR_AMSDUDISCARD:
    case RWNX_RX_HD_DECR_NULLKEY:
    case RWNX_RX_HD_DECR_CCMPFAIL:
        ret = -1;
        break;

    case RWNX_RX_HD_DECR_WEPSUCCESS:
    case RWNX_RX_HD_DECR_TKIPSUCCESS:
    case RWNX_RX_HD_DECR_CCMPSUCCESS:
        status->flag |= RX_FLAG_DECRYPTED;
        break;
    }

#if 1
    RWNX_DBG("tsf %llx / type %d / subtype %d / rate %d\n",
             status->mactime, hw_vect->type, hw_vect->subtype, hw_vect->leg_rate);
#endif

    return ret;
}

/**
 * rwnx_unsup_rx_vec_ind() - IRQ handler callback for %IPC_IRQ_E2A_UNSUP_RX_VEC
 *
 * LMAC has triggered an IT saying that a rx vector of an unsupported frame has been
 * captured and sent to upper layer. Then we need to fill the rx status, create a vendor
 * specific header and fill it with the HT packet length. Finally, we need to specify at
 * least 2 bytes of data and send the sk_buff to mac80211.
 *
 * @pthis: Pointer to main driver data
 * @hostid: Pointer to IPC elem from e2aradars_pool
 */
u8 rwnx_unsup_rx_vec_ind(void *pthis, void *hostid) {
    struct rwnx_hw *rwnx_hw = pthis;
    struct rwnx_ipc_skb_elem *elem = hostid;
    struct rx_vector_desc *rx_desc;
    struct sk_buff *skb;
    struct rx_vector_1 *rx_vect1;
    struct phy_channel_info_desc *phy_info;
    u16 ht_length;
    struct ieee80211_rx_status *rx_status;

    dma_sync_single_for_cpu(rwnx_hw->dev, elem->dma_addr,
                            sizeof(struct rx_vector_desc), DMA_FROM_DEVICE);

    skb = elem->skb;
    if (((struct rx_vector_desc *) (skb->data))->pattern == 0) {
        /*sync is needed even if the driver did not modify the memory*/
        dma_sync_single_for_device(rwnx_hw->dev, elem->dma_addr,
                                     sizeof(struct rx_vector_desc), DMA_FROM_DEVICE);
        return -1;
    }

    rx_desc = (struct rx_vector_desc *) (skb->data);
    rx_vect1 = (struct rx_vector_1 *) (rx_desc->rx_vect1);
    rwnx_rx_vector_convert(rwnx_hw, rx_vect1, NULL);
    phy_info = (struct phy_channel_info_desc *) (&rx_desc->phy_info);
    if (rx_vect1->format_mod >= FORMATMOD_VHT)
        ht_length = 0;
    else
        ht_length = (u16) le32_to_cpu(rx_vect1->ht.length);

    /* Fill Rx status*/
    rx_status = IEEE80211_SKB_RXCB(skb);
    memset(rx_status, 0, sizeof(struct ieee80211_rx_status));
    rwnx_fill_rxstatus_rxvect_phyinfo(rx_status, rx_vect1, phy_info);


    // Reserve space for radiotap
    skb_reserve(skb, RADIOTAP_HDR_MAX_LEN);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 19, 0)
    {
        struct ieee80211_vendor_radiotap *rtap;
        rx_status->flag |= RX_FLAG_RADIOTAP_VENDOR_DATA;

        /* Fill vendor specific header with fake values */
        rtap = (struct ieee80211_vendor_radiotap *) skb->data;
        rtap->present = BIT(0);
        rtap->align = 1;
        rtap->oui[0] = 0x00;
        rtap->oui[1] = 0x25;
        rtap->oui[2] = 0x3A;
        rtap->subns  = 0;
        rtap->pad = 0;
        rtap->len = sizeof(ht_length);
        put_unaligned_le16(ht_length, rtap->data);
        skb_put(skb, sizeof(*rtap) + rtap->pad + rtap->len);

        /* Copy fake data */
        put_unaligned_le16(0, rtap->data + rtap->len);
        skb_put(skb, UNSUP_RX_VEC_DATA_LEN);
    }
#endif

    /* Unmap will synchronize buffer for CPU */
    dma_unmap_single(rwnx_hw->dev, elem->dma_addr, rwnx_hw->ipc_env->unsuprxvec_bufsz,
                     DMA_FROM_DEVICE);
    elem->skb = NULL;

    ieee80211_rx(rwnx_hw->hw, skb);

    /* Allocate and push a new buffer to fw to replace this one */
    if (rwnx_ipc_unsup_rx_vec_elem_allocs(rwnx_hw, elem))
        dev_err(rwnx_hw->dev, "Failed to alloc new unsupported rx vector buf\n");

    return 0;
}

/**
 * Callback for data reception indication
 * FIXME: memory allocation failures handling is buggy
 */
u8 rwnx_rxdataind(void *pthis, void *hostid)
{
    struct rwnx_hw *rwnx_hw = (struct rwnx_hw *)pthis;
    struct hw_rxhdr *hw_rxhdr;
    struct hw_vect *hw_vect;
    struct rwnx_ipc_skb_elem *rxelem = hostid;
    struct ieee80211_rx_status *rx_status;
    struct sk_buff *skb;
    u8 ret = 0;
    u32 rx_pattern;
    int mpdu_offset = sizeof(struct hw_rxhdr) + 2;
    int peek_len    = mpdu_offset + sizeof(struct ieee80211_hdr);
    int push_back = 0;
    REG_SW_SET_PROFILING(rwnx_hw, SW_PROF_RWNXDATAIND);

    if (!rxelem->skb) {
        ret = -1;
        goto alloc_new;
    }

    dma_sync_single_for_cpu(rwnx_hw->dev, rxelem->dma_addr,
                            peek_len, DMA_FROM_DEVICE);

    skb = rxelem->skb;
    rx_pattern = ((struct hw_rxhdr *)skb->data)->pattern;

    /* Look for pattern saying that this buf has been used */
    if (rx_pattern != RX_DMA_OVER_PATTERN) {
        /*sync is needed even if the driver did not modify the memory*/
        dma_sync_single_for_device(rwnx_hw->dev, rxelem->dma_addr,
                                   peek_len, DMA_FROM_DEVICE);
        ret = -1;
        goto end;
    }

    hw_rxhdr = (struct hw_rxhdr *)skb->data;
    hw_vect = &hw_rxhdr->hwvect;

    rwnx_rx_vector_convert(rwnx_hw, &hw_vect->rx_vect1, &hw_vect->rx_vect2);

    rx_status = IEEE80211_SKB_RXCB(skb);

    /* For now, push back if invalid
     * soon to arrive hw versions will discard (some of ?) them
     * then we will be able to discard these checks */

    if (unlikely(rwnx_fill_rxstatus(rwnx_hw, rx_status, hw_rxhdr)))
        push_back = 1;
    else if (unlikely(hw_vect->decr_status == RWNX_RX_HD_DECR_UNENC)) {
        struct ieee80211_hdr *mac_hdr =
            (struct ieee80211_hdr *)(skb->data + mpdu_offset);
        if (ieee80211_has_protected(mac_hdr->frame_control))
            push_back = 1;
    }

    if (push_back) {
        /* The buffer is not valid, re-push it to fw */
        rwnx_ipc_rxbuf_elem_repush(rwnx_hw, rxelem);
        goto end;
    }

    /* Unmap will synchronize buffer for CPU */
    dma_unmap_single(rwnx_hw->dev, rxelem->dma_addr, rwnx_hw->ipc_env->rx_bufsz,
                     DMA_FROM_DEVICE);

    skb_reserve(skb, mpdu_offset);
    skb_put(skb, le32_to_cpu(hw_vect->len));
    rxelem->skb = NULL;

    /* XXX
     * w/o RX_FLAG_IV_STRIPPED mac80211 removes both IV and ICV
     * our lower layers trim : WEP/TKIP {ext,}IV and IV | CCMP MIC */
    if (hw_vect->decr_status == RWNX_RX_HD_DECR_WEPSUCCESS ||
        hw_vect->decr_status == RWNX_RX_HD_DECR_TKIPSUCCESS)
        skb_put(skb, IEEE80211_WEP_ICV_LEN);
    if (hw_vect->decr_status == RWNX_RX_HD_DECR_CCMPSUCCESS)
        skb_put(skb, IEEE80211_CCMP_MIC_LEN);

    REG_SW_SET_PROFILING(rwnx_hw, SW_PROF_IEEE80211RX);
    ieee80211_rx(rwnx_hw->hw, skb);
    REG_SW_CLEAR_PROFILING(rwnx_hw, SW_PROF_IEEE80211RX);

alloc_new:
    /* Allocate and push a new rx buffer to fw to replace this one */
    if (rwnx_ipc_rxbuf_elem_allocs(rwnx_hw, rxelem)) {
        dev_err(rwnx_hw->dev, "Failed to alloc new RX buf\n");
        /* return an error only if we did not consume */
    }

end:
    REG_SW_CLEAR_PROFILING(rwnx_hw, SW_PROF_RWNXDATAIND);

    return ret;
}


