/**
 ******************************************************************************
 *
 * @file rwnx_tx.c
 *
 * Copyright (C) RivieraWaves 2012-2019
 *
 ******************************************************************************
 */
#include <linux/dma-mapping.h>
#include <net/mac80211.h>
#include <net/sock.h>

#include "rwnx_tx.h"
#include "rwnx_txq.h"
#include "hal_desc.h"
#include "rwnx_baws.h"
#include "rwnx_prof.h"
#ifdef CONFIG_RWNX_BFMER
#include "rwnx_bfmer.h"
#endif //(CONFIG_RWNX_BFMER)
#include "rwnx_events.h"

/**
 * Masks for bits in tx_flags value in struct umacdesc
 */
#define RWNX_TX_UMAC_BEAMFORM_MASK     (CO_BIT(0))

/**
 * rwnx_update_wpi_header - Write WPI header if needed
 *
 * @rwnx_hw: Driver main data
 * @sw_txhdr: Initailzed Software descriptor for the frame
 *
 * For WPI (aka WAPI), mac80211 only leaves empty space in the frame for the
 * security header. This function updates it.
 * Does nothing for non WPI frame.
 */
static int rwnx_update_wpi_header(struct rwnx_hw *rwnx_hw,
                                  struct rwnx_sw_txhdr *sw_txhdr)
{
    struct ieee80211_key_conf *conf;
    struct rwnx_wpi_key *key;
    u8 *wpi_hdr;
    bool overflow, carry;
    int i;

    conf = IEEE80211_SKB_CB(sw_txhdr->skb)->control.hw_key;
    if (!conf || conf->cipher != WLAN_CIPHER_SUITE_SMS4)
        return 0;

    wpi_hdr = skb_mac_header(sw_txhdr->skb) + ieee80211_hdrlen(sw_txhdr->frame_ctl);

    hlist_for_each_entry(key, &rwnx_hw->wpi_keys, list) {
        if (key->conf == conf)
            break;
    }

    if (!key)
        return -EINVAL;

     /* increment pn */
    if (key->conf->flags & IEEE80211_KEY_FLAG_PAIRWISE) {
        key->pn[0] += 2;
        carry = (key->pn[0] <= 1);
    } else {
        key->pn[0] += 1;
        carry = (key->pn[0] == 0);
    }

    if (carry) {
        for(i = 1; i < WPI_PN_LEN; i++) {
            key->pn[i]++;
            if (key->pn[i])
                break;
        }
    }

    /* update wpi header */
    wpi_hdr[0] = key->conf->keyidx;
    wpi_hdr[1] = 0;
    memcpy(&wpi_hdr[2], key->pn, sizeof(key->pn));

    /* check overflow */
    overflow = 1;
    for(i = 1; i < WPI_PN_LEN && overflow; i++) {
        overflow = (key->pn[i] == 0xff);
    }

    if (overflow) {
        /* TODO: inform user space of needed rekey */
        pr_info("PN overflow is coming");
    }

    return 0;
}

#ifdef CONFIG_RWNX_AMSDUS_TX
/**
 * amsdu_headroom - Return headroom needed for AMSDU software decriptor
 *
 * @pad: Padding to include before the header
 */
static inline unsigned int amsdu_headroom(u8 pad)
{
    return sizeof(struct rwnx_amsdu_txhdr) + ((4 - pad) & 3) + 2;
}

/**
 * rwnx_free_amsdusubs - Free amsdu subframes after confirmation
 *
 * @rwnx_hw: Main driver data
 * @sw_txhdr: Software descriptor for the AMSDU frame
 */
static void rwnx_free_amsdusubs(struct rwnx_hw *rwnx_hw,
                                struct rwnx_sw_txhdr *sw_txhdr)
{
    struct rwnx_amsdu_txhdr *atxhdr;

    list_for_each_entry(atxhdr, &sw_txhdr->amsdu.hdrs, list) {
        struct sk_buff *sub_skb = atxhdr->skb;
        struct ieee80211_amsdu_info *ainfo =
            (struct ieee80211_amsdu_info *)sub_skb->cb;
        dma_unmap_single(rwnx_hw->dev, atxhdr->dma_addr, atxhdr->map_len,
                         DMA_TO_DEVICE);
        skb_pull(sub_skb, amsdu_headroom(ainfo->hdr_pad));
        dev_kfree_skb(sub_skb);
    }
}

/**
 * rwnx_ops_agg_msdu - mac80211 operation callback when a subframe is added to
 * an AMSDU frame
 *
 * @hw: mac80211's main driver data
 * @ctl: Control structure for the AMSDU
 * @skb: Buffer containing the subframe (with AMSDU header already included)
 */
int rwnx_ops_agg_msdu(struct ieee80211_hw *hw, struct amsdu_ctl *ctl,
                      struct sk_buff *skb)
{
    struct rwnx_hw *rwnx_hw = hw->priv;
    struct ieee80211_amsdu_info *ainfo = (struct ieee80211_amsdu_info *)skb->cb;
    struct rwnx_txq *rwnx_txq =
        &((struct rwnx_sta *)ainfo->sta->drv_priv)->txqs[ainfo->tid];
    struct rwnx_sw_txhdr *sw_txhdr;
    struct rwnx_amsdu *amsdu;
    struct rwnx_amsdu_txhdr *amsdu_txhdr;
    dma_addr_t dma_addr;
    int map_len;

    map_len = skb->len;
    dma_addr = dma_map_single(rwnx_hw->dev, skb->data, map_len,
                              DMA_BIDIRECTIONAL);
    if (WARN_ON(dma_mapping_error(rwnx_hw->dev, dma_addr)))
        return -1;

    /* get an aligned header - use hdr_pad in case we want to use it in cfm */
    amsdu_txhdr = (void *)skb_push(skb, amsdu_headroom(ainfo->hdr_pad));
    amsdu_txhdr->dma_addr = dma_addr;
    amsdu_txhdr->map_len  = map_len;
    amsdu_txhdr->skb      = skb;

    /* Take lock to prevent driver ending this amsdu while updating it */
    spin_lock_bh(&rwnx_hw->amsdu_lock);
    sw_txhdr = rwnx_txq->amsdu_anchor;
    if (unlikely(!sw_txhdr || sw_txhdr->skb != ctl->anchor)) {
        spin_unlock(&rwnx_hw->amsdu_lock);
        dma_unmap_single(rwnx_hw->dev, dma_addr, map_len, DMA_TO_DEVICE);
        skb_pull(skb, amsdu_headroom(ainfo->hdr_pad));
        return -1;
    }
    amsdu = &sw_txhdr->amsdu;
    list_add_tail(&amsdu_txhdr->list, &amsdu->hdrs);

    if (WARN_ON(amsdu->rem_len < map_len))
        amsdu->rem_len = 0;
    else
        amsdu->rem_len -= map_len;

    if (++amsdu->nb == NX_TX_PAYLOAD_MAX ||
        amsdu->nb >= rwnx_hw->mod_params->amsdu_maxnb) {
        ctl->anchor = NULL;
        sw_txhdr->txq->amsdu_anchor = NULL;
    } else {
        ctl->left = amsdu->rem_len;
    }

    spin_unlock_bh(&rwnx_hw->amsdu_lock);

    return 0;
}

/**
 * rwnx_start_amsdu - Try to set software desriptor as AMSDU anchor
 *
 * @rwnx_hw: Main driver data
 * @sw_txhdr: Software descriptor
 *
 * When a new software descriptor is processed, this function test if it can
 * be used as an anchor for an AMSDU frame (i.e. the first frame).
 * Also when queuing the first buffer in a txq it inform mac80211 if AMSDU is
 * possible
 */
void rwnx_start_amsdu(struct rwnx_hw *rwnx_hw, struct rwnx_sw_txhdr *sw_txhdr)
{
    struct rwnx_txq *txq = sw_txhdr->txq;
    struct ieee80211_tx_info *tx_info = IEEE80211_SKB_CB(sw_txhdr->skb);

    /* On first buffer queued indicate mac80211 if AMSDU are ready for this sta */
    if ((skb_queue_len(&txq->sk_list) == 1) &&
        sw_txhdr->baw && sw_txhdr->baw->agg_on &&
        txq->amsdu_ht_len_cap && rwnx_hw->mod_params->amsdu_maxnb) {
        ieee80211_amsdu_ctl(rwnx_hw->hw, txq->sta, sw_txhdr->tid, NULL, 0, 0,
                            true);
    }

    /* Check if this buffer can be the new amsdu anchor */
    if ((tx_info->control.flags & IEEE80211_TX_CTRL_AMSDU) &&
        sw_txhdr->baw && sw_txhdr->baw->agg_on &&
        (rwnx_hw->mod_params->amsdu_maxnb > 1)) {

        /* no need for lock as long as it is done before informing mac80211
           on the new anchor */
        txq->amsdu_anchor = sw_txhdr;

        if (ieee80211_amsdu_ctl(rwnx_hw->hw, txq->sta, sw_txhdr->tid,
                                sw_txhdr->skb, tx_info->control.amsdu_next_pad,
                                sw_txhdr->amsdu.rem_len, true)) {
            txq->amsdu_anchor = NULL;
            return;
        }

        INIT_LIST_HEAD(&sw_txhdr->amsdu.hdrs);
        sw_txhdr->amsdu.nb = 1;
        sw_txhdr->amsdu.rem_len -= sw_txhdr->frame_len;
    }
}

/**
 * rwnx_end_amsdu - End an AMSDU frame before pushing it to fw
 *
 * @rwnx_hw: Main driver data
 * @sw_txhdr: Software descriptor
 *
 * Before pushing the software descriptor to the fw check if it is an AMSDU anchor.
 * If so, inform mac80211 that AMSDU frame is closed and complete descriptor
 * with subframe addesses. (It could have been done in .agg_msdu but when
 * security is used it's easier to move icv_len from first to last buffer when
 * the size of the AMSDU is known)
 */
void rwnx_end_amsdu(struct rwnx_hw *rwnx_hw, struct rwnx_sw_txhdr *sw_txhdr)
{
    struct rwnx_txq *txq = sw_txhdr->txq;

    if (txq->amsdu_anchor == sw_txhdr) {
        /* Inform mac80211 we end this amsdu */
        ieee80211_amsdu_ctl(rwnx_hw->hw, txq->sta, sw_txhdr->tid, NULL, 0, 0,
                            txq->amsdu_ht_len_cap &&
                            (rwnx_hw->mod_params->amsdu_maxnb > 1));

        /* Take lock to ensure agg_amsdu is not currently running */
        spin_lock_bh(&rwnx_hw->amsdu_lock);
        txq->amsdu_anchor = NULL;
        spin_unlock_bh(&rwnx_hw->amsdu_lock);
    }

    if (sw_txhdr->amsdu.nb > 1) {
        struct rwnx_amsdu *amsdu = &sw_txhdr->amsdu;
        struct rwnx_amsdu_txhdr *atxhdr;
        struct hostdesc *hostdesc = &sw_txhdr->desc.host;
        int i = 1;

        list_for_each_entry(atxhdr, &amsdu->hdrs, list) {
            hostdesc->packet_addr[i] = (u32_l)atxhdr->dma_addr;
            hostdesc->packet_len[i] = atxhdr->map_len;
            hostdesc->packet_cnt++;
            i++;
        }

        if (amsdu->icv_len) {
            hostdesc->packet_len[0] -= amsdu->icv_len;
            hostdesc->packet_len[i - 1] += amsdu->icv_len;
        }
    }
}

/**
 * rwnx_is_amsdu - Test if software descriptor contains an AMSDU
 *
 * @sw_txhdr: Software descriptor
 *
 * @return true if software descriptor contains an AMSDU and false otherwise.
 */
static inline bool rwnx_is_amsdu(struct rwnx_sw_txhdr *sw_txhdr)
{
    return (sw_txhdr->amsdu.nb);
}

#elif defined(CONFIG_MAC80211_AMSDUS_TX)
/**
 * rwnx_is_amsdu - Test if software descriptor contains an AMSDU
 *
 * @sw_txhdr: Software descriptor
 *
 * @return true if software descriptor contains an AMSDU and false otherwise.
 */
static inline bool rwnx_is_amsdu(struct rwnx_sw_txhdr *sw_txhdr)
{
    struct ieee80211_tx_info *tx_info = IEEE80211_SKB_CB(sw_txhdr->skb);
    return (tx_info->control.flags & IEEE80211_TX_CTRL_AMSDU);
}

/**
 * rwnx_free_amsdusubs - Unmap amsdu subframes after confirmation or mapping error
 *
 * @rwnx_hw: Main driver data
 * @sw_txhdr: Software descriptor for the AMSDU frame
 */
static void rwnx_free_amsdusubs(struct rwnx_hw *rwnx_hw,
                                struct rwnx_sw_txhdr *sw_txhdr)
{
    int pkt_cnt = sw_txhdr->desc.host.packet_cnt;

    while (pkt_cnt > 1) {
        pkt_cnt--;
        dma_unmap_single(rwnx_hw->dev,
                         sw_txhdr->desc.host.packet_addr[pkt_cnt],
                         sw_txhdr->desc.host.packet_len[pkt_cnt],
                         DMA_TO_DEVICE);
    }
}

/**
 * rwnx_prep_amsdu_dma_tx - Map AMSDU subframes for DMA transfer
 *
 * @rwnx_hw: Main driver data
 * @sw_txhdr: Software descriptor for the AMSDU frame
 *
 * Map all amsdu subframes, stored as skb fragments, for DMA transfer and update
 * software descriptor accordingly.
 * return immediately if this is not an AMSDU frame
 */
static inline int rwnx_prep_amsdu_dma_tx(struct rwnx_hw *rwnx_hw,
                                         struct rwnx_sw_txhdr *sw_txhdr)
{
    struct sk_buff *frag, *skb = sw_txhdr->skb;

    /* To reflect runtime parameter changes to mac80211 */
    rwnx_hw->hw->max_tx_fragments = rwnx_hw->mod_params->amsdu_maxnb;

    if (!rwnx_is_amsdu(sw_txhdr))
        return 0;

    skb_walk_frags(skb, frag) {
        dma_addr_t dma_addr;
        int pkt_cnt = sw_txhdr->desc.host.packet_cnt;

        dma_addr = dma_map_single(rwnx_hw->dev, frag->data, frag->len,
                                  DMA_BIDIRECTIONAL);

        if (WARN_ON(dma_mapping_error(rwnx_hw->dev, dma_addr))) {
            rwnx_free_amsdusubs(rwnx_hw, sw_txhdr);
            return -1;
        }

        sw_txhdr->desc.host.packet_addr[pkt_cnt] = dma_addr;
        sw_txhdr->desc.host.packet_len[pkt_cnt] = frag->len;
        sw_txhdr->desc.host.packet_cnt++;
    }

    return 0;
}

#endif /* CONFIG_RWNX_AMSDUS_TX */

#ifdef CONFIG_XXX_AMSDUS_TX
/**
 * rwnx_amsdu_stat_update - Update AMSDU statistics
 *
 * @rwnx_hw: Main driver data
 * @sw_txhdr: Software descriptor
 * @failure: True if frame hasn't been successfully transmitted
 */
static inline void rwnx_amsdu_stat_update(struct rwnx_hw *rwnx_hw,
                                          struct rwnx_sw_txhdr *sw_txhdr,
                                          bool failure)
{
    int amsdu_size;

    if (!rwnx_is_amsdu(sw_txhdr))
        return;

    amsdu_size = sw_txhdr->desc.host.packet_cnt - 1;
    rwnx_hw->stats.amsdus[amsdu_size].done++;
    if (failure)
        rwnx_hw->stats.amsdus[amsdu_size].failed++;
}
#endif

/**
 * rwnx_fill_pol_mcs_rate - Fill one rate control information on the policy
 * table with a HT/VHT rate
 *
 * @rinfo: Pointer on rate info to update
 * @rate: Rate selected
 * @gf: True if GreenField should be used (HT only)
 * @return: Number of NSS (-1) for the selected rate
 */
static inline u8 rwnx_fill_pol_mcs_rate(union rwnx_rate_ctrl_info *rinfo,
                                        struct ieee80211_tx_rate *rate,
                                        bool gf)
{
    u8 nss;

    rinfo->mcsIndexTx = rate->idx;
    rinfo->giAndPreTypeTx = !!(rate->flags & IEEE80211_TX_RC_SHORT_GI);

    if (rate->flags & IEEE80211_TX_RC_VHT_MCS) {
        nss = rate->idx >> 4;
        rinfo->formatModTx = FORMATMOD_VHT;
    } else {
        nss = rate->idx >> 3;
        /* GF
         * TODO let GF and SGI be exclusive only for single SS MCSs
         *      for the moment make it always exclusive */
        if (gf && !rinfo->giAndPreTypeTx) {
            rinfo->formatModTx = FORMATMOD_HT_GF;
            /* for radiotap feedback - SW RC minstrel doesn't handle this */
            rate->flags |= IEEE80211_TX_RC_GREEN_FIELD;
        } else {
            rinfo->formatModTx = FORMATMOD_HT_MF;
        }
    }

    if (rate->flags & IEEE80211_TX_RC_40_MHZ_WIDTH ||
        rate->flags & IEEE80211_TX_RC_DUP_DATA)
        rinfo->bwTx = PHY_CHNL_BW_40;
    else if (rate->flags & IEEE80211_TX_RC_80_MHZ_WIDTH)
        rinfo->bwTx = PHY_CHNL_BW_80;
    else if (rate->flags & IEEE80211_TX_RC_160_MHZ_WIDTH)
        rinfo->bwTx = PHY_CHNL_BW_160;

    return nss;
}

/**
 * rwnx_fill_pol_mcs_rate - Fill one rate control information in the policy
 * table with a legary rate
 *
 * @rinfo: Pointer on rate info to update
 * @rate: Rate selected
 * @bitrates: Table of basic rates supported
 */
static inline void rwnx_fill_pol_leg_rate(union rwnx_rate_ctrl_info *rinfo,
                                          struct ieee80211_tx_rate *rate,
                                          struct ieee80211_rate *bitrates)
{
    rinfo->mcsIndexTx = bitrates[rate->idx].hw_value;
    rinfo->formatModTx = FORMATMOD_NON_HT;
    if (!(rate->flags & IEEE80211_TX_RC_USE_SHORT_PREAMBLE))
        rinfo->giAndPreTypeTx = 1 << 1;
}

/**
 * rwnx_fill_pol_prot - Fill protection fields of one rate control information
 * in the policy table
 *
 * @rinfo: Pointer on rate info to update
 * @prot_ridx: rate to use for protection frames
 * @use_rts: True is RTS/CTS protection has to be used used and
 * false if selfCTS must be used.
 */
static inline void rwnx_fill_pol_prot(union rwnx_rate_ctrl_info *rinfo,
                                      u8 prot_ridx, int use_rts)
{
    /* Choose rts over cts */
    rinfo->navProtFrmEx = use_rts ? NAV_PROT_RTS_CTS_BIT : NAV_PROT_SELF_CTS_BIT;
    rinfo->bwProtTx = rinfo->bwTx;
    if (rinfo->bwProtTx == PHY_CHNL_BW_20)
        rinfo->formatModProtTx = FORMATMOD_NON_HT;
    else
        rinfo->formatModProtTx = FORMATMOD_NON_HT_DUP_OFDM;
    rinfo->mcsIndexProtTx = prot_ridx;
}

/**
 * rwnx_fill_pol_power - Fill one power control information in the policy table
 *
 * @pinfo: pointer to power info to update
 * @tx_power: TX power to configure
 * @use_prot: True is RTS or selfCTS protection has been configured
 */
static inline void rwnx_fill_pol_power(struct rwnx_power_ctrl_info *pinfo,
                                       u8 tx_power, bool use_prot)
{
    pinfo->txPwrLevelPT = tx_power;
    if (use_prot)
         pinfo->txPwrLevelProtPT = pinfo->txPwrLevelPT;
}

/**
 * rwnx_fill_pol_tbl - Initialize policy table
 *
 * @rwnx_hw: Driver main data
 * @tx_info: TX configuration provided by mac80211
 * @sw_txhdr: Pointer to the software descriptor
 * @hw_txhdr: Pointer to the hardware descriptor
 * @sta: Pointer to the destination STA (can be null)
 *
 * FIXME once the old behavior has been validated (c.f old rc limitations)
 * Sticking to the old behavior means trying one entry rinfo at most once,
 * and the last one for every other attempts
 *
 * Though the mac man states wrt new rc facilities :
 * nRetryRC = "Number of trial which can be done using this Rate Control",
 * the behavior of the mac is more like, FWIU, :
 *     tx(rinfo[0]);         x 1 time
 *     for (i = 0, tx_left = {short,long}retry_cnt;
 *                      tx_left > 0 && i < NX_TX_MAX_RATES; i++) {
 *          if (!rinfo[i].nRetry)
 *              continue
 *          tx(rinfo[i]);    x min(tx_left, rinfo[i].nRetry) time(s)
 *          tx_left -= rinfo[i].nRetry;
 *     }
 *     if (tx_left > 0)
 *          tx(rinfo[3]);    x min(tx_left, xxx) time(s)
 *
 * ATM we are using hw->wiphy->retry_{short,long} for {short,long}retry_cnt,
 * we could use the SW RC parameters instead and skip 1a)
 *
 * Beware of minstrel w/ IEEE80211_HW_SUPPORTS_HT_CCK_RATES wrt A-MPDUs/A-MSDUs
 */
static void rwnx_fill_pol_tbl(struct rwnx_hw *rwnx_hw,
                              struct ieee80211_tx_info *tx_info,
                              struct rwnx_sw_txhdr *sw_txhdr,
                              struct rwnx_hw_txhdr *hw_txhdr,
                              struct ieee80211_sta *sta)
{
    struct tx_policy_tbl *pol_tbl = &hw_txhdr->policy;
    struct ieee80211_bss_conf *bss_conf;
    struct ieee80211_rate *bitrates;
    u8 prot_leg_ridx, prot_mcs_ridx;
    u8 nss;
    u8 stbc_nss = 0;
    u8 sta_stbc_nss = 0;
    u8 hw_sta_idx = 0;
    u8 tx_power;
    bool use_stbc = false;
#ifdef CONFIG_RWNX_BFMER
    bool use_bfm = false;
    // Highest Number of Spatial Streams for VHT
    u8 max_nss = 0;
#endif //(CONFIG_RWNX_BFMER)
    bool gf_sta = false, gf_able = false;
    int i;

    bitrates = rwnx_hw->hw->wiphy->bands[tx_info->band]->bitrates;
    bss_conf = tx_info->control.vif ? &tx_info->control.vif->bss_conf : NULL;

    /* Get STA params */
    if (sta) {
        struct rwnx_sta *rwnx_sta = (struct rwnx_sta *)sta->drv_priv;
        sta_stbc_nss = rwnx_sta->stbc_nss;
        hw_sta_idx = rwnx_sta->hw_sta_idx;
#ifdef CONFIG_RWNX_BFMER
        if (rwnx_sta->bfm_report) {
            use_bfm = true;
        }
#endif //(CONFIG_RWNX_BFMER)

        /* GF */
        if (rwnx_hw->mod_params->gf_on && bss_conf &&
            (sta->ht_cap.cap & IEEE80211_HT_CAP_GRN_FLD)) {
            int ht_op = bss_conf->ht_operation_mode;
            if (((ht_op & IEEE80211_HT_OP_MODE_PROTECTION) ==
                 IEEE80211_HT_OP_MODE_PROTECTION_NONMEMBER) ||
                ((ht_op & IEEE80211_HT_OP_MODE_PROTECTION) ==
                 IEEE80211_HT_OP_MODE_PROTECTION_NONHT_MIXED) ||
                (ht_op & (IEEE80211_HT_OP_MODE_NON_GF_STA_PRSNT |
                          IEEE80211_HT_OP_MODE_NON_HT_STA_PRSNT)))
                gf_able = false;
            else
                gf_able = true;
            gf_sta = true;
        }
    }

    /* RTS/SelfCTS rates */
    if (bss_conf && bss_conf->basic_rates)
        prot_mcs_ridx = bitrates[fls(bss_conf->basic_rates) - 1].hw_value;
    else
        prot_mcs_ridx = bitrates[0].hw_value;
    prot_leg_ridx = ieee80211_get_rts_cts_rate(rwnx_hw->hw, tx_info)->hw_value;

    /* TX power */
    if (unlikely(rwnx_hw->scanning && ieee80211_is_probe_req(sw_txhdr->frame_ctl)))
        tx_power = rwnx_hw->scan_txpower_idx;
    else
        tx_power = ((struct rwnx_vif *)(tx_info->control.vif->drv_priv))->txpower_idx;

    /* pattern */
    pol_tbl->upatterntx = POLICY_TABLE_PATTERN;

    /* ratectrlinfos / powerctrlinfos */
    for (i = 0 ; i < rwnx_hw->hw->max_rates; i++) {
        union rwnx_rate_ctrl_info *rinfo = &pol_tbl->ratectrlinfos[i];
        struct rwnx_power_ctrl_info *pinfo = &pol_tbl->powerctrlinfos[i];
        struct ieee80211_tx_rate *rate = &tx_info->control.rates[i];
        bool use_rts = false, use_cts = false;
        u8 prot_ridx;

        if (rate->idx == -1) {
            int j;
            for (j = i; j < NX_TX_MAX_RATES; j++, rinfo++, pinfo++) {
                *rinfo = pol_tbl->ratectrlinfos[i - 1];
                *pinfo = pol_tbl->powerctrlinfos[i - 1];
                rinfo->nRetry = 0;
            }
            break;
        }

        rinfo->value = 0;
        rinfo->nRetry = !!i; /* rate->count; */

        if ((rate->flags & IEEE80211_TX_RC_USE_RTS_CTS) && sta)
            use_rts = true;
        else if (rate->flags & IEEE80211_TX_RC_USE_CTS_PROTECT)
            use_cts = true;

        if (rate->flags >= IEEE80211_TX_RC_MCS) {
            prot_ridx = prot_mcs_ridx;
            nss = rwnx_fill_pol_mcs_rate(rinfo, rate,
                                         (gf_sta && (use_rts || use_cts || gf_able)));

#ifdef CONFIG_RWNX_BFMER
            if (nss > max_nss)
                max_nss = nss;
#endif //(CONFIG_RWNX_BFMER)
            if ((i == 0) && (nss < sta_stbc_nss)) {
                stbc_nss = nss;
                use_stbc = true;
            }
            else if (use_stbc && (nss != stbc_nss))
                use_stbc = false;

        } else {
            prot_ridx = prot_leg_ridx;
            rwnx_fill_pol_leg_rate(rinfo, rate, bitrates);
        }

        if (use_rts || use_cts)
            rwnx_fill_pol_prot(rinfo, prot_ridx, use_rts);

        rwnx_fill_pol_power(pinfo, tx_power, (use_rts || use_cts));
    }

    /* phyctrlinfo_1 */
    pol_tbl->phyctrlinfo_1 = rwnx_hw->phy.ctrlinfo_1;
    if (tx_info->flags & IEEE80211_TX_CTL_LDPC)
        pol_tbl->phyctrlinfo_1.fecCoding = 1;
    if (use_stbc)
        pol_tbl->phyctrlinfo_1.stbc = (stbc_nss + 1);

     /* phyctrlinfo_2 */
    pol_tbl->phyctrlinfo_2 = rwnx_hw->phy.ctrlinfo_2;

    /* macctrlinfo_1 */
    pol_tbl->macctrlinfo_1.value = 0;
    pol_tbl->macctrlinfo_1.keySRamIndexRA = hw_sta_idx;
    if (tx_info->control.hw_key) {
        struct ieee80211_key_conf *key_conf = tx_info->control.hw_key;
        pol_tbl->macctrlinfo_1.keySRamIndex = key_conf->hw_key_idx;
    }

    /* macctrlinfo_2 */
    pol_tbl->macctrlinfo_2.value = 0;
    pol_tbl->macctrlinfo_2.rtsThreshold = rwnx_hw->hw->wiphy->rts_threshold;
    pol_tbl->macctrlinfo_2.shortRetryLimit = rwnx_hw->hw->wiphy->retry_short;
    pol_tbl->macctrlinfo_2.longRetryLimit = rwnx_hw->hw->wiphy->retry_long;

#ifdef CONFIG_RWNX_BFMER
    if (use_bfm) {
        struct rwnx_sta *rwnx_sta = (struct rwnx_sta *)sta->drv_priv;
        // Get report Nc Index - First 3 bits of the header
        int nc_idx = (int)(rwnx_sta->bfm_report->report[0] & 0x7);

        if (nc_idx < max_nss) {
            use_bfm = false;
        }
    }

    if (use_bfm) {
        sw_txhdr->desc.umac.tx_flags |= RWNX_TX_UMAC_BEAMFORM_MASK;
    } else {
        sw_txhdr->desc.umac.tx_flags &= ~RWNX_TX_UMAC_BEAMFORM_MASK;
    }
#endif //(CONFIG_RWNX_BFMER)
}

/**
 * rwnx_fill_mac_config - Update MAC control info of a hardware descriptor
 *
 * @mac_ctrl: Pointer to MAC control
 * @tx_info: Tx information
 */
static inline void rwnx_fill_mac_config( union rwnx_thd_mac_ctrl_info_1 *mac_ctrl,
                                         struct ieee80211_tx_info *tx_info)
{
    mac_ctrl->value = 0;
    mac_ctrl->lowRateRetry = 1;
    if (tx_info->flags & IEEE80211_TX_CTL_NO_ACK)
        mac_ctrl->expectedAck = EXPECTED_ACK_NO_ACK;
    else
        mac_ctrl->expectedAck = EXPECTED_ACK_NORMAL_ACK;
}

/**
 * rwnx_fill_phy_config - Update PHY control info of a hardware descriptor
 *
 * @rwnx_hw: Main driver data
 * @phy_ctrl: Pointer to PHY control
 * @sta: Pointer to mac80211 STA
 * @rwnx_sta: Pointer to rwnx STA
 */
static inline void rwnx_fill_phy_config(struct rwnx_hw *rwnx_hw,
                                 union rwnx_thd_phy_ctrl_info *phy_ctrl,
                                 struct ieee80211_sta *sta,
                                 struct rwnx_sta *rwnx_sta)
{
    phy_ctrl->value = 0;
    if (sta && sta->vht_cap.vht_supported) {
        phy_ctrl->groupIDTx = rwnx_sta->gid;
        phy_ctrl->partialAIDTx = rwnx_sta->paid;
        /* bw signaling */
        if (rwnx_hw->mod_params->bwsig_on) {
            phy_ctrl->useBWSignalingTx = 1;
            if (rwnx_hw->mod_params->dynbw_on)
                phy_ctrl->dynBWTx = 1;
        }
    } else {
        phy_ctrl->groupIDTx = 63;
    }
}

/**
 * rwnx_prep_hw_txhdr - Initialize HW descriptor
 *
 * @rwnx_hw: Main driver data
 * @txhdr: TX descriptors
 */
static void rwnx_prep_hw_txhdr(struct rwnx_hw *rwnx_hw, struct rwnx_txhdr *txhdr)
{
    struct rwnx_sw_txhdr *sw_txhdr = txhdr->sw_hdr;
    struct rwnx_hw_txhdr *hw_txhdr = &txhdr->hw_hdr;
    struct ieee80211_tx_info *tx_info = IEEE80211_SKB_CB(sw_txhdr->skb);
    struct ieee80211_sta *sta;
    struct rwnx_sta *rwnx_sta;

    hw_txhdr->status.value = 0;
    rwnx_sta = sw_txhdr->rwnx_sta;
    sta = rwnx_to_ieee80211_sta(rwnx_sta);

    rwnx_fill_phy_config(rwnx_hw, &hw_txhdr->phy_ctrl_info, sta, rwnx_sta);
    rwnx_fill_mac_config(&hw_txhdr->mac_ctrl_info, tx_info);
    rwnx_fill_pol_tbl(rwnx_hw, tx_info, sw_txhdr, hw_txhdr, sta);
}

/**
 * rwnx_drop_skb - Drop a buffer
 *
 * @rwnx_hw: Main driver data
 * @sw_txhdr: Software descriptor allocated for the buffer (may be NULL)
 * @skb: Buffer to drop
 * @headroom: Headroom used in the buffer to store descriptors
 * @sta: Destination STA for the buffer (may be NULL)
 */
static void rwnx_drop_skb(struct rwnx_hw *rwnx_hw, struct rwnx_sw_txhdr *sw_txhdr,
                          struct sk_buff *skb, unsigned int headroom,
                          struct ieee80211_sta *sta)
{
    /* If we drop the last buffer of a PS Service Period, we must clear the
       'service period' at mac80211 level */
    if ((IEEE80211_SKB_CB(skb)->flags & IEEE80211_TX_STATUS_EOSP) && sta)
        ieee80211_sta_eosp(sta);

    if (sw_txhdr)
        kmem_cache_free(rwnx_hw->sw_txhdr_cache, sw_txhdr);

    skb_pull(skb, headroom);
    ieee80211_free_txskb(rwnx_hw->hw, skb);
}

/**
 * rwnx_drop_sw_txhdr -  Drop a buffer for which software descriptor has been
 * initialized
 *
 * @rwnx_hw: Main driver data
 * @sw_txhdr: Software descriptor initialized for the buffer
 */
static inline void rwnx_drop_sw_txhdr(struct rwnx_hw *rwnx_hw,
                                      struct rwnx_sw_txhdr *sw_txhdr)
{
    rwnx_drop_skb(rwnx_hw, sw_txhdr, sw_txhdr->skb, sw_txhdr->headroom,
                  rwnx_to_ieee80211_sta(sw_txhdr->rwnx_sta));
}

/**
 * rwnx_prep_sw_txhdr - Allocate TX descriptors and initialize software
 * descriptor.
 *
 * @rwnx_hw: Main driver data
 * @sta: Destination STA (may be NULL)
 * @skb: Buffer to send
 *
 * @return: Pointer to allocated TX descriptors and NULL if allocation/
 * initialization failed.
 */
static struct rwnx_txhdr *rwnx_prep_sw_txhdr(struct rwnx_hw *rwnx_hw,
                                             struct ieee80211_sta *sta,
                                             struct sk_buff *skb)
{
    struct rwnx_txhdr *txhdr;
    struct rwnx_sw_txhdr *sw_txhdr;
    struct ieee80211_hdr *mac_hdr;
    struct ieee80211_tx_info *tx_info;
    struct rwnx_vif *rwnx_vif;
    unsigned int hdr_pads, headroom;

    mac_hdr = (struct ieee80211_hdr *)skb->data;
    hdr_pads = RWNX_SWTXHDR_ALIGN_PADS((long)mac_hdr);
    headroom = sizeof(struct rwnx_txhdr) + hdr_pads;

    txhdr = (struct rwnx_txhdr *)skb_push(skb, headroom);
    txhdr->sw_hdr = kmem_cache_alloc(rwnx_hw->sw_txhdr_cache, GFP_ATOMIC);
    if (unlikely(txhdr->sw_hdr == NULL))
        goto error;

    tx_info = IEEE80211_SKB_CB(skb);
    rwnx_vif = (struct rwnx_vif *)(tx_info->control.vif->drv_priv);

    sw_txhdr = txhdr->sw_hdr;
    sw_txhdr->skb = skb;
    sw_txhdr->frame_len = (u16)(skb_headlen(skb) - headroom);
    sw_txhdr->frame_ctl = mac_hdr->frame_control;
    sw_txhdr->sn = IEEE80211_SEQ_TO_SN(mac_hdr->seq_ctrl);
    sw_txhdr->tid = skb->priority & IEEE80211_QOS_CTL_TID_MASK;
    sw_txhdr->baw = NULL;
    sw_txhdr->ampdu_status.total_cnt = 0;
    sw_txhdr->headroom = headroom;
    sw_txhdr->map_len = skb_headlen(skb) - offsetof(struct rwnx_txhdr, hw_hdr);

    if (sta) {
        sw_txhdr->rwnx_sta = (struct rwnx_sta *)sta->drv_priv;
        sw_txhdr->txq = rwnx_txq_sta_get(sw_txhdr->rwnx_sta, sw_txhdr->tid);
        sw_txhdr->desc.host.staid = sw_txhdr->rwnx_sta->sta_idx;
    } else {
        /* Drop QOS frames with no sta (11aa) except for Mesh Point */
        if (unlikely((rwnx_vif->vif->type != NL80211_IFTYPE_MESH_POINT) &&
                     ieee80211_is_data_qos(sw_txhdr->frame_ctl)))
            goto error;

        sw_txhdr->rwnx_sta = NULL;
        sw_txhdr->txq = rwnx_txq_vif_get(rwnx_vif, tx_info->hw_queue);
        sw_txhdr->desc.host.staid = 0xFF;
    }

    /* Sanity check: Ensure HWQ linked to txq match hwq selected by mac80211 */
    WARN_ON(tx_info->hw_queue != sw_txhdr->txq->hwq->id);

    sw_txhdr->desc.host.tid = sw_txhdr->tid;
    sw_txhdr->desc.host.vif_idx = rwnx_vif->vif_index;
    sw_txhdr->desc.host.padding = hdr_pads;
#ifdef CONFIG_RWNX_MUMIMO_TX
    sw_txhdr->desc.host.mumimo_info = 0;
#endif
#ifdef CONFIG_RWNX_AMSDUS_TX
    sw_txhdr->amsdu.nb = 0;
    sw_txhdr->amsdu.icv_len = 0;
#endif

    if (tx_info->control.hw_key) {
        sw_txhdr->frame_len += tx_info->control.hw_key->icv_len;
#ifdef CONFIG_RWNX_AMSDUS_TX
        if (tx_info->control.flags & IEEE80211_TX_CTRL_AMSDU)
            sw_txhdr->amsdu.icv_len = tx_info->control.hw_key->icv_len;
#endif
    }

    sw_txhdr->first_jiffies = jiffies;
    return txhdr;

error:
    rwnx_drop_skb(rwnx_hw, txhdr->sw_hdr, skb, headroom, sta);
    return NULL;
}

/**
 *  rwnx_prep_ampdu - Initialize AMPDU part in the software desctiptor
 *
 * @rwnx_hw: Main driver data
 * @txhdr: TX descriptors
 *
 * Does nothing if buffer cannot be send inside an AMPDU.
 * Must be called HW descriptor already initialized.
 *
 * @return: 0 on success and !=0 if buffer must be dropped
 */
static int rwnx_prep_ampdu(struct rwnx_hw *rwnx_hw, struct rwnx_txhdr *txhdr)
{
#ifdef CONFIG_RWNX_AGG_TX
    struct rwnx_sw_txhdr *sw_txhdr = txhdr->sw_hdr;
    struct rwnx_txq *txq = sw_txhdr->txq;
    struct ieee80211_tx_info *tx_info = IEEE80211_SKB_CB(sw_txhdr->skb);
    union rwnx_rate_ctrl_info *rinfo;

    sw_txhdr->desc.umac.flags = 0;

    if (unlikely(!(tx_info->flags & IEEE80211_TX_CTL_AMPDU) ||
                 !txq->baw.agg_on ||
                 !ieee80211_is_data_qos(sw_txhdr->frame_ctl) ||
                 ieee80211_is_qos_nullfunc(sw_txhdr->frame_ctl)))
        return 0;

    /* Special case: singleton was first sent before BA was established
       and is now re-transmitted after creation of BA.
       No need to re-transmit as it is out of BAW => ignore it. */
    if (unlikely((tx_info->flags & IEEE80211_TX_INTFL_RETRANSMISSION) &&
                 !rwnx_is_in_baw(&txq->baw, sw_txhdr->sn)))
        return 1;

    sw_txhdr->baw = &txq->baw;
    sw_txhdr->ba_idx = txq->baw.ba_idx;

    /* Note: ba_idx will be checked in rwnx_tx_push */
    rinfo = &txhdr->hw_hdr.policy.ratectrlinfos[0];

    sw_txhdr->desc.host.sn = sw_txhdr->sn;
    if (likely(rinfo->formatModTx != FORMATMOD_NON_HT)) {
        if (unlikely(rwnx_hw->mod_params->rc_probes_on &&
                     tx_info->flags & IEEE80211_TX_CTL_RATE_CTRL_PROBE))
            sw_txhdr->desc.umac.flags = WHICHDESC_AMPDU_FIRST;
        else
            sw_txhdr->desc.umac.flags = AMPDU_BIT;
    }
    sw_txhdr->desc.umac.phy_flags = rinfo->value;

#ifdef CONFIG_RWNX_AMSDUS_TX
    if (rinfo->formatModTx == FORMATMOD_VHT)
        sw_txhdr->amsdu.rem_len = sw_txhdr->txq->amsdu_vht_len_cap;
    else
        sw_txhdr->amsdu.rem_len = sw_txhdr->txq->amsdu_ht_len_cap;
#endif /* CONFIG_RWNX_AMSDUS_TX */
#endif /* CONFIG_RWNX_AGG_TX */

    return 0;
}

/**
 * rwnx_prep_dma_tx - Map buffer and TX descriptors for DMA transfer
 *
 * @rwnx_hw: Main driver data
 * @txhdr: TX descriptors
 * @return: 0 on success and !=0 if DMA map failed
 */
static int rwnx_prep_dma_tx(struct rwnx_hw *rwnx_hw,
                            struct rwnx_txhdr *txhdr)
{
    struct rwnx_sw_txhdr *sw_txhdr = txhdr->sw_hdr;
    struct rwnx_hw_txhdr *hw_txhdr = &txhdr->hw_hdr;
    dma_addr_t dma_addr;

    dma_addr = dma_map_single(rwnx_hw->dev, hw_txhdr, sw_txhdr->map_len,
                              DMA_BIDIRECTIONAL);
    if (WARN_ON(dma_mapping_error(rwnx_hw->dev, dma_addr)))
        return -1;

    sw_txhdr->dma_addr = dma_addr;
#ifdef CONFIG_RWNX_SPLIT_TX_BUF
    sw_txhdr->desc.host.packet_addr[0] = dma_addr;
    sw_txhdr->desc.host.packet_len[0] = sw_txhdr->frame_len;
    sw_txhdr->desc.host.packet_cnt = 1;
#else
    sw_txhdr->desc.host.packet_addr = dma_addr;
    sw_txhdr->desc.host.packet_len = sw_txhdr->frame_len;
#endif

    return 0;
}

/**
 * rwnx_tx_prep - Initializes TX descriptor for a buffer
 *
 * @rwnx_hw: Main driver data
 * @sta: Destination STA (may be NULL)
 * @skb: Date buffer
 *
 * Simple wrapper that calls all rwnx_prep_xxx functions.
 *
 * @return pointer to fully initialized TX descriptor and NULL if buffer
 * has been dropped.
 */
static inline struct rwnx_txhdr *rwnx_tx_prep(struct rwnx_hw *rwnx_hw,
                                              struct ieee80211_sta *sta,
                                              struct sk_buff *skb)
{
    struct rwnx_txhdr *txhdr;

    txhdr = rwnx_prep_sw_txhdr(rwnx_hw, sta, skb);
    if (txhdr == NULL)
        return NULL;

    if (unlikely(rwnx_update_wpi_header(rwnx_hw, txhdr->sw_hdr)))
        goto drop;

    rwnx_prep_hw_txhdr(rwnx_hw, txhdr);

    if (unlikely(rwnx_prep_ampdu(rwnx_hw, txhdr)))
        goto drop;

    if (unlikely(rwnx_prep_dma_tx(rwnx_hw, txhdr)))
        goto drop;

#ifdef CONFIG_MAC80211_AMSDUS_TX
    if (unlikely(rwnx_prep_amsdu_dma_tx(rwnx_hw, txhdr->sw_hdr)))
        goto drop;
#endif

    return txhdr;

drop:
    rwnx_drop_sw_txhdr(rwnx_hw, txhdr->sw_hdr);
    return NULL;
}

/**
 * rwnx_process_txhdr - Process a buffer once TX descriptor have been initialized
 *
 * @rwnx_hw: Main driver data
 * @txhdr: TX descriptors
 */
static void rwnx_process_txhdr(struct rwnx_hw *rwnx_hw, struct rwnx_txhdr *txhdr)
{
    struct rwnx_sw_txhdr *sw_txhdr = txhdr->sw_hdr;
    struct rwnx_txq *txq = sw_txhdr->txq;
    struct rwnx_hwq *hwq = txq->hwq;

    if (rwnx_txq_is_ready_for_push(txq)) {
        hwq->len++;
        rwnx_tx_push(rwnx_hw, txhdr, RWNX_PUSH_IMMEDIATE);
        return;
    }

    /* Sanity check: We should not process new txhdr when hwq is stopped */
    WARN_ON_ONCE(hwq->len >= hwq->len_stop);

    REG_SW_SET_PROFILING_CHAN(rwnx_hw, SW_PROF_CHAN_CTXT_TX_PAUSE_BIT);
    rwnx_txq_queue_skb(sw_txhdr->skb, txq, rwnx_hw, false);
    REG_SW_CLEAR_PROFILING_CHAN(rwnx_hw, SW_PROF_CHAN_CTXT_TX_PAUSE_BIT);

#ifdef CONFIG_RWNX_AMSDUS_TX
    rwnx_start_amsdu(rwnx_hw, sw_txhdr);
#endif
}

/**
 * rwnx_tx_push_ps - Update MAC header for buffer pushed during PS Service Period
 *
 * @txq: TXQ used for transmission
 * @sb: Buffer to pushed
 *
 * When pushing buffer for PS service period "managed" by the driver (i.e. after
 * mac80211 called rwnx_ops_release_buffered_frames) the driver is responsible
 * for setting the MORE_DATA and EOSP flags
 * (see doc for ieee80211_ops.release_buffered_frames for details)
 *
 * If txq->push_limit is 0 then this buffer is not part of a PS SP (or at
 * least no managed by the driver) and the function returns immediately
 */
static void rwnx_tx_push_ps(struct rwnx_txq *txq, struct sk_buff *skb)
{
    struct rwnx_sta *rwnx_sta;
    struct ieee80211_tx_info *tx_info;
    struct ieee80211_qos_hdr *mac_hdr;

    if (!txq->push_limit)
        return;

    tx_info = IEEE80211_SKB_CB(skb);
    rwnx_sta = (struct rwnx_sta *)txq->sta->drv_priv;
    mac_hdr = (struct ieee80211_qos_hdr *)skb_mac_header(skb);

    txq->push_limit--;
    rwnx_sta->ps.sp_cnt--;

    if (rwnx_sta->ps.sp_cnt > 0) {
        mac_hdr->frame_control |= cpu_to_le16(IEEE80211_FCTL_MOREDATA);
    } else {
        /* This is the last buffer of the SP */
        tx_info->flags |= IEEE80211_TX_STATUS_EOSP;

        if (rwnx_sta->ps.uapsd && ieee80211_is_data_qos(mac_hdr->frame_control))
            mac_hdr->qos_ctrl |= cpu_to_le16(IEEE80211_QOS_CTL_EOSP);
        /* TODO: Handle case where last buffer of UAPSD SP is not a QOS frame */

        if (rwnx_sta->ps.more_data)
            mac_hdr->frame_control |= cpu_to_le16(IEEE80211_FCTL_MOREDATA);
    }
}

/**
 *  rwnx_tx_push - Push one packet to fw
 *
 * @rwnx_hw: Driver main data
 * @txhdr: tx desc of the buffer to push
 * @flags: Push flag (cf @rwnx_push_flags)
 *
 * Push one packet to fw.
 */
void rwnx_tx_push(struct rwnx_hw *rwnx_hw, struct rwnx_txhdr *txhdr, int flags)
{
    struct rwnx_sw_txhdr *sw_txhdr = txhdr->sw_hdr;
    struct sk_buff *skb = sw_txhdr->skb;
    struct rwnx_txq *txq = sw_txhdr->txq;
    u16 hw_queue = txq->hwq->id;
    int user = 0;

    lockdep_assert_held(&rwnx_hw->tx_lock);

    REG_SW_SET_PROFILING_CHAN(rwnx_hw, SW_PROF_CHAN_CTXT_PUSH_BIT);

    /* RETRY flag is not always set so retest here */
    if (txq->nb_retry) {
        flags |= RWNX_PUSH_RETRY;
        txq->nb_retry--;
        if (txq->nb_retry == 0) {
            WARN(skb != txq->last_retry_skb,
                 "last retry buffer is not the expected one");
            txq->last_retry_skb = NULL;
        }
    }

    if (flags & RWNX_PUSH_RETRY) {
        int peek_off = offsetof(struct rwnx_hw_txhdr, status);
        int peek_len = sizeof(((struct rwnx_hw_txhdr *)0)->status);

        txhdr->hw_hdr.status.value = 0;
        dma_sync_single_for_device(rwnx_hw->dev, sw_txhdr->dma_addr + peek_off,
                                   peek_len, DMA_BIDIRECTIONAL);
#ifdef CONFIG_RWNX_AGG_TX
        /* We might want to retry non-BAWed frames, e.g. any frame discarded by
         * LMAC because of non-matching chanctxt */
        if (sw_txhdr->baw && likely(sw_txhdr->desc.umac.flags & AMPDU_BIT)) {
                sw_txhdr->desc.umac.sn_win = sw_txhdr->baw->fsn;
        }
#endif
    } else {
#ifdef CONFIG_RWNX_AMSDUS_TX
        rwnx_end_amsdu(rwnx_hw, sw_txhdr);
#endif /* CONFIG_RWNX_AMSDUS_TX */

#ifdef CONFIG_RWNX_AGG_TX
        if (sw_txhdr->baw) {
            struct rwnx_baw *baw = sw_txhdr->baw;

            if (likely(sw_txhdr->ba_idx == baw->ba_idx)) {
                struct ieee80211_tx_info *tx_info = IEEE80211_SKB_CB(sw_txhdr->skb);
                u16 sn = sw_txhdr->sn;

                /* There's this one OoBAW left from mac80211 internal buffering */
                if (unlikely(rwnx_set_baw_state(baw, sn, BAW_PENDING, tx_info))) {
                    txq->hwq->len--;
                    rwnx_drop_sw_txhdr(rwnx_hw, sw_txhdr);
                    return;
                }
                sw_txhdr->desc.umac.sn_win = baw->fsn;
            } else {
                sw_txhdr->baw = NULL;
                sw_txhdr->desc.umac.flags &= ~AMPDU_BIT;
            }
        }
#endif /* CONFIG_RWNX_AGG_TX */
        txq->pkt_sent++;
        txq->hwq->len--;
    }

#ifdef CONFIG_RWNX_MUMIMO_TX
    /* MU group is only selected during hwq processing */
    sw_txhdr->desc.host.mumimo_info = txq->mumimo_info;
    user = RWNX_TXQ_POS_ID(txq);
#endif /* CONFIG_RWNX_MUMIMO_TX */

    rwnx_tx_push_ps(txq, skb);

    trace_push_desc(skb, sw_txhdr, flags);
    txq->credits--;
    txq->pkt_pushed[user]++;
    if (txq->credits <= 0)
        rwnx_txq_stop(txq, RWNX_TXQ_STOP_FULL);

    rwnx_ipc_txdesc_push(rwnx_hw, &sw_txhdr->desc, skb, hw_queue, user);
    rwnx_hw->hwq[hw_queue].credits[user]--;
    rwnx_hw->stats.cfm_balance[hw_queue]++;

    REG_SW_CLEAR_PROFILING_CHAN(rwnx_hw, SW_PROF_CHAN_CTXT_PUSH_BIT);
}

/**
 *  rwnx_ops_tx - mac80211 tx ops
 *
 *  @tx: Handler that 802.11 module calls for each transmitted frame.
 *  skb contains the buffer starting from the IEEE 802.11 header.
 *  The low-level driver should send the frame out based on
 *  configuration in the TX control data. This handler should,
 *  preferably, never fail and stop queues appropriately.
 *  This must be implemented if @tx_frags is not.
 *  Must be atomic
 *
 *  We don't memset the hdrs so we must pay attention to properly set the fields
 */
void rwnx_ops_tx(struct ieee80211_hw *hw, struct ieee80211_tx_control *control,
                 struct sk_buff *skb)
{
    struct rwnx_hw *rwnx_hw = hw->priv;
    struct rwnx_txhdr *txhdr;

    REG_SW_SET_PROFILING_CHAN(rwnx_hw, SW_PROF_CHAN_CTXT_TX_BIT);
    sk_pacing_shift_update(skb->sk, rwnx_hw->tcp_pacing_shift);

    txhdr = rwnx_tx_prep(rwnx_hw, control->sta, skb);
    if (txhdr == NULL)
        goto end;

    spin_lock(&rwnx_hw->tx_lock);
    rwnx_process_txhdr(rwnx_hw, txhdr);
    spin_unlock(&rwnx_hw->tx_lock);

end:
    REG_SW_CLEAR_PROFILING_CHAN(rwnx_hw, SW_PROF_CHAN_CTXT_TX_BIT);
}

#ifdef CONFIG_MAC80211_TXQ
/**
 * rwnx_ops_wake_tx_queue - mac80211 wake_tx_queue ops
 *
 * @wake_tx_queue: Called when new packets have been added to the queue.
 */
void rwnx_ops_wake_tx_queue(struct ieee80211_hw *hw, struct ieee80211_txq *mac_txq)
{
    struct rwnx_hw *rwnx_hw = hw->priv;
    struct rwnx_txq *txq = (struct rwnx_txq *)mac_txq->drv_priv;

    txq->nb_ready_mac80211++;
    trace_txq_wake(txq);

    spin_lock(&rwnx_hw->tx_lock);
    if (!rwnx_txq_is_stopped(txq) &&
        !rwnx_txq_is_scheduled(txq)) {

        /* Check mac80211 txq len here, as it may have been dequeued
           while waiting for spinlock */
        ieee80211_txq_get_depth(mac_txq, &txq->nb_ready_mac80211, NULL);
        if (txq->nb_ready_mac80211) {
            rwnx_txq_add_to_hw_list(txq);
            rwnx_hwq_process(rwnx_hw, txq->hwq);
        }
    }
    spin_unlock(&rwnx_hw->tx_lock);
}

/**
 * rwnx_tx_dequeue_prep - Dequeue one buffer from mac80211 and prepare it
 *
 * @rwnx_hw: Main driver data
 * @txq: mac80211 txq
 *
 * If buffer is dropped in rwnx_tx_prep then a new one is dequeued.
 *
 * @return Pointer to skb with TX descriptor initialized and NULL if
 * no more buffer is ready in the txq
 */
struct sk_buff *rwnx_tx_dequeue_prep(struct rwnx_hw *rwnx_hw,
                                     struct ieee80211_txq *mac_txq)
{
    struct rwnx_txq *txq = (struct rwnx_txq *)mac_txq->drv_priv;
    struct sk_buff *skb;
    struct rwnx_txhdr *txhdr;

    while (1) {
        skb = ieee80211_tx_dequeue(rwnx_hw->hw, mac_txq);
        if (skb == NULL) {
            if (txq->nb_ready_mac80211 > 0) {
                trace_txq_drop(txq, txq->nb_ready_mac80211);
                txq->nb_ready_mac80211 = 0;
            }
            return NULL;
        }

        WARN(txq->hwq->id != IEEE80211_SKB_CB(skb)->hw_queue,
             "Skb will be pushed on AC %d instead of AC %d\n",
             txq->hwq->id, IEEE80211_SKB_CB(skb)->hw_queue);

        txq->nb_ready_mac80211--;
        txhdr = rwnx_tx_prep(rwnx_hw, txq->sta, skb);
        txq->hwq->len++;
        if (txhdr)
            return skb;
    }

    return NULL;
}
#endif

/**
 * rwnx_ops_release_buffered_frames - mac80211 release_buffered_frames ops
 *
 * @release_buffered_frames: Release buffered frames according to the given
 *	parameters. In the case where the driver buffers some frames for
 *	sleeping stations mac80211 will use this callback to tell the driver
 *	to release some frames, either for PS-poll or uAPSD.
 *	Note that if the @more_data parameter is %false the driver must check
 *	if there are more frames on the given TIDs, and if there are more than
 *	the frames being released then it must still set the more-data bit in
 *	the frame. If the @more_data parameter is %true, then of course the
 *	more-data bit must always be set.
 *	The @tids parameter tells the driver which TIDs to release frames
 *	from, for PS-poll it will always have only a single bit set.
 *	In the case this is used for a PS-poll initiated release, the
 *	@num_frames parameter will always be 1 so code can be shared. In
 *	this case the driver must also set %IEEE80211_TX_STATUS_EOSP flag
 *	on the TX status (and must report TX status) so that the PS-poll
 *	period is properly ended. This is used to avoid sending multiple
 *	responses for a retried PS-poll frame.
 *	In the case this is used for uAPSD, the @num_frames parameter may be
 *	bigger than one, but the driver may send fewer frames (it must send
 *	at least one, however). In this case it is also responsible for
 *	setting the EOSP flag in the QoS header of the frames. Also, when the
 *	service period ends, the driver must set %IEEE80211_TX_STATUS_EOSP
 *	on the last frame in the SP. Alternatively, it may call the function
 *	ieee80211_sta_eosp() to inform mac80211 of the end of the SP.
 *	This callback must be atomic.
 *
 * Dispatch the request among all buffered frames in txq of requested tids,
 * using higher priority txq first. This is done by setting the push_limit in the
 * txq struct and adding it to its hwq schedule list.
 * MORE_DATA and EOSP flags will only be set when the buffer is actually pushed
 * to the firmware (@rwnx_tx_push_ps) as there is no way to know the push order.
 * To do so the reason, service period length and the more data flag (updated with
 * txq buffered status) are save in rwnx_sta_ps structure.
 *
 */
void rwnx_ops_release_buffered_frames(struct ieee80211_hw *hw,
                                      struct ieee80211_sta *sta,
                                      u16 tids, int num_frames,
                                      enum ieee80211_frame_release_type reason,
                                      bool more_data)
{
    struct rwnx_hw *rwnx_hw = hw->priv;
    struct rwnx_sta *rwnx_sta = (struct rwnx_sta *)sta->drv_priv;
    struct rwnx_txq *txq;
    int i, tid, remain_frames = num_frames;

    if (reason == IEEE80211_FRAME_RELEASE_UAPSD)
        rwnx_sta->ps.uapsd = true;
    else
        rwnx_sta->ps.uapsd = false;
    rwnx_sta->ps.more_data = more_data;

    spin_lock(&rwnx_hw->tx_lock);

    foreach_sta_txq_prio(rwnx_sta, txq, tid, i, rwnx_hw) {
        int nb_skb;

        if (!(tids & BIT(tid)))
            continue;

        nb_skb = skb_queue_len(&txq->sk_list);

        if (remain_frames < nb_skb) {
            rwnx_sta->ps.more_data = true;
            nb_skb = remain_frames;
        }

        if (!nb_skb)
            continue;

        txq->push_limit = nb_skb;
        rwnx_txq_add_to_hw_list(txq);

        remain_frames -= nb_skb;
        if (remain_frames == 0)
            break;
    }

    rwnx_sta->ps.sp_cnt = num_frames - remain_frames;

    if (rwnx_sta->ps.sp_cnt == 0)
        ieee80211_sta_eosp(sta);
    else
        rwnx_hwq_process_all(rwnx_hw);

    spin_unlock(&rwnx_hw->tx_lock);
}

/**
 * rwnx_down_rate - Update rate provided by Rate Controller after transmission
 *
 * @rate: Rate to update
 * @bw: Actual bandwidth used for tranmission
 *
 * Used for HW downgrade of tx bandwidth down to bw in reported transmission_bw.
 * In that case the VHT MCS might get downgraded too.
 */
static void rwnx_down_rate(struct ieee80211_tx_rate *rate, u32 bw)
{
    int nss;

    // FIXME
    WARN_ON(rate->flags & IEEE80211_TX_RC_DUP_DATA);
    rate->flags &= ~(IEEE80211_TX_RC_40_MHZ_WIDTH |
                     IEEE80211_TX_RC_80_MHZ_WIDTH |
                     IEEE80211_TX_RC_160_MHZ_WIDTH);
    switch (bw) {
    case PHY_CHNL_BW_20:
        if ((rate->flags & IEEE80211_TX_RC_VHT_MCS) && rate->idx == 9)
            rate->idx = 8;
        break;
    case PHY_CHNL_BW_40:
        rate->flags |= IEEE80211_TX_RC_40_MHZ_WIDTH;
        break;
    case PHY_CHNL_BW_80:
        nss = ieee80211_rate_get_vht_nss(rate);
        if (rate->idx == 6 && (nss == 3 || nss == 7))
            rate->idx = 5;
        else if (rate->idx == 9 && nss == 6)
            rate->idx = 8;
        rate->flags |= IEEE80211_TX_RC_80_MHZ_WIDTH;
        break;
    default:
        WARN_ON(1);
    }
}

/**
 * rwnx_tx_retry - Resend a packet that FW failed to send
 *
 * @rwnx_hw: Driver main data
 * @txhdr: TX descriptor of the buffer to re-send
 */
static inline void rwnx_tx_retry(struct rwnx_hw *rwnx_hw,
                                 struct rwnx_txhdr *txhdr)
{
    struct rwnx_sw_txhdr *sw_txhdr = txhdr->sw_hdr;
    struct rwnx_txq *txq = sw_txhdr->txq;

    /* Pkt to retry is equivalent of one txq credit in all cases */
    txq->credits ++;
    if (txq->credits > 0)
        rwnx_txq_start(txq, RWNX_TXQ_STOP_FULL);

    if (!rwnx_txq_is_stopped(txq) && (txq->nb_retry == 0)) {
        rwnx_tx_push(rwnx_hw, txhdr, RWNX_PUSH_IMMEDIATE | RWNX_PUSH_RETRY);
    } else {
        rwnx_txq_queue_skb(sw_txhdr->skb, txq, rwnx_hw, true);
    }
}


/**
 * rwnx_get_hw_txstatus - Get HW transmission status
 *
 * @rwnx_hw: Main driver data
 * @txhdr: TX descriptor
 * @rwnx_txst: Updated with HW tx status
 *
 * @return: 0 if status has been updated and != 0 if transmission is not yet
 * complete.
 */
static inline int rwnx_get_hw_txstatus(struct rwnx_hw *rwnx_hw,
                                       struct rwnx_txhdr *txhdr,
                                       union rwnx_hw_txstatus *rwnx_txst)
{
    dma_addr_t status_addr = (txhdr->sw_hdr->dma_addr +
                              offsetof(struct rwnx_hw_txhdr, status));
    int status_len = sizeof(((struct rwnx_hw_txhdr *)0)->status);

    dma_sync_single_for_cpu(rwnx_hw->dev, status_addr, status_len,
                            DMA_FROM_DEVICE);

    if (txhdr->hw_hdr.status.value == 0) {
        dma_sync_single_for_device(rwnx_hw->dev, status_addr, status_len,
                                   DMA_FROM_DEVICE);
        return -1;
    }

    *rwnx_txst = txhdr->hw_hdr.status;
    return 0;
}


/**
 * rwnx_can_retry_mpdu - Test if MPDU can be re-pushed to FW
 *
 * @rwnx_hw: Main driver data
 * @sw_txhdr: Software descriptor
 *
 * A MPDU can be re-pushed to FW if it's lifetime is not done and
 * Block Ack Agreement didn't change since first push
 *
 * @return: true if MPDU can be retried and false otherwise
 */
static inline bool rwnx_can_retry_mpdu(struct rwnx_hw *rwnx_hw,
                                       struct rwnx_sw_txhdr *sw_txhdr)
{
    return (time_is_after_jiffies(sw_txhdr->first_jiffies +
                                  msecs_to_jiffies(rwnx_hw->mod_params->tx_lft)) &&
            sw_txhdr->baw && (sw_txhdr->ba_idx == sw_txhdr->baw->ba_idx));
}

/**
 * rwnx_update_ampdu_txstatus - Update txinfo with rate used for AMPDU transmission
 *
 * @sw_txhdr: Software descriptor
 * @tx_info: mac80211 structure to be updated
 * @is_last: True is this the the software descriptor of the last MPDU inside
 * the AMPDU
 *
 * tx_info is updated with the status of the whole A-MPDU so this can only be
 * done when processing confirmation of the last MPDU (i.e. is_last is true).
 *
 * However if the last MPDU had to be retried, AMPDU status is not sent to
 * mac80211 but saved in the software descriptor. The AMPDU status is sent
 * when the frame is finally confirmed (because it has been successfully
 * re-transmitted or its lifetime reached 0). In this case (is_last is false)
 * AMPDU status is read from the software descriptor.
 */
static inline void rwnx_update_ampdu_txstatus(struct rwnx_sw_txhdr *sw_txhdr,
                                              struct ieee80211_tx_info *tx_info,
                                              bool is_last)
{
    struct rwnx_baw_ampdu_status *status;

    if (is_last)
        status = &sw_txhdr->baw->rc_status;
    else
        status = &sw_txhdr->ampdu_status;

    tx_info->flags |= IEEE80211_TX_STAT_AMPDU;
    tx_info->status.ampdu_ack_len = status->ok_cnt;
    tx_info->status.ampdu_len = status->total_cnt;
    tx_info->status.rates[0] = status->rate;
    tx_info->status.rates[0].count = 1;
    tx_info->status.rates[1].idx = -1;
}

/**
 * rwnx_process_ampdu_cfm - Process confirmation for a MPDU transmitted
 * in an AMPDU
 *
 * @rwnx_hw: Main driver data
 * @txhdr: TX descriptor
 * @tx_info: mac80211 tx information, updated with TX rate used if processed
 * is set to true
 * @rwnx_txst: HW transmission status
 * @cred: Updated with the number of credit this frame give.
 * @processed: Set to true if tx_info and cred have been updated and left
 * untouch otherwise
 *
 * @return: != 0 if the frame has been retried, and 0 otherwise. In the latter
 * case, it is necessary to read the value of 'processed' to see if cred and
 * tx_info have been updated.
 *
 */
static int rwnx_process_ampdu_cfm(struct rwnx_hw *rwnx_hw,
                                  struct rwnx_txhdr *txhdr,
                                  struct ieee80211_tx_info *tx_info,
                                  union rwnx_hw_txstatus rwnx_txst,
                                  int *cred, bool *processed)
{
#ifdef CONFIG_RWNX_AGG_TX
    struct rwnx_sw_txhdr *sw_txhdr = txhdr->sw_hdr;
    struct rwnx_baw *baw = sw_txhdr->baw;

    if (!baw )
        return 0;

    if (sw_txhdr->rwnx_sta->ps.on && !rwnx_txst.frm_successful_tx) {
        /* If transmission failed to a STA in PS, mac80211 will try to
           retransmit it, so skip retry and baw update. */
        if (!(tx_info->flags & IEEE80211_TX_CTL_NO_PS_BUFFER))
            tx_info->flags |= IEEE80211_TX_STAT_TX_FILTERED;

        return 0;
    }

    /* Check if the frame was part of an A-MPDU */
    if (RWNX_WD_IS_AMPDU(rwnx_txst.which_descriptor_sw)) {
        bool is_last = false;

        /* Get the rate used to transmit the AMPDU from cfm of the 1st MPDU */
        if (RWNX_WD_IS_FIRST(rwnx_txst.which_descriptor_sw)) {
            u32 bwTx = txhdr->hw_hdr.policy.ratectrlinfos[0].bwTx;

            baw->rc_status.total_cnt = 0;
            baw->rc_status.ok_cnt = 0;
            baw->rc_status.rate = tx_info->control.rates[0];
            if (rwnx_txst.transmission_bw != bwTx) {
                WARN_ON(rwnx_txst.transmission_bw > bwTx);
                rwnx_down_rate(&baw->rc_status.rate,
                               rwnx_txst.transmission_bw);
            }
        } else {
            is_last = RWNX_WD_IS_LAST(rwnx_txst.which_descriptor_sw);
        }

        baw->rc_status.total_cnt++;
        rwnx_hw->stats.agg_done++;
        rwnx_hw->stats.in_ampdu[baw->rc_status.total_cnt - 1].done++;

        if (rwnx_txst.frm_successful_tx) {
            baw->rc_status.ok_cnt++;
        } else {
            rwnx_hw->stats.in_ampdu[baw->rc_status.total_cnt - 1].failed++;
            if (rwnx_can_retry_mpdu(rwnx_hw, sw_txhdr)) {
                if (is_last) {
                    rwnx_hw->stats.agg_retries_last++;

                    if (unlikely(!baw->rc_status.ok_cnt)) {
                        /* Whole AMPDU failed, make it singleton */
                        sw_txhdr->desc.umac.flags &= ~(WHICHDESC_AMPDU_FIRST);
                        rwnx_hw->stats.ampdu_all_ko++;
                    }

                    /* save AMPDU counter in sw_txhdr to send them to mac80211
                       when this frame is re-confirmed */
                    sw_txhdr->ampdu_status = baw->rc_status;
                }

                rwnx_hw->stats.agg_retries++;
                rwnx_tx_retry(rwnx_hw, txhdr);
                return 1;
            } else {
                rwnx_hw->stats.agg_died++;
            }
        }

        if (is_last)
            rwnx_update_ampdu_txstatus(sw_txhdr, tx_info, true);
        else if (sw_txhdr->ampdu_status.total_cnt)
            rwnx_update_ampdu_txstatus(sw_txhdr, tx_info, false);

        *processed = true;
    }

    /* Or possibly aggregated but sent by the FW as a singleton MPDU */
    else if (tx_info->flags & IEEE80211_TX_CTL_AMPDU) {
#ifdef CONFIG_RWNX_SPLIT_TX_BUF
        /* Even for singleton, FW disables retries in MACHW when packet_cnt > 1
           so do it here */
        if ((sw_txhdr->desc.host.packet_cnt > 1) &&
            !rwnx_txst.frm_successful_tx &&
            rwnx_can_retry_mpdu(rwnx_hw, sw_txhdr)) {
            rwnx_tx_retry(rwnx_hw, txhdr);
            return 1;
        }
#endif
        if (sw_txhdr->ampdu_status.total_cnt) {
            rwnx_update_ampdu_txstatus(sw_txhdr, tx_info, false);
            *processed = true;
        } else {
            tx_info->flags &= ~IEEE80211_TX_CTL_AMPDU;
        }
    }

    if (sw_txhdr->ba_idx == baw->ba_idx) {
        rwnx_set_baw_state(baw, sw_txhdr->sn, BAW_CONFIRMED, tx_info);
        if (rwnx_is_first_in_baw(baw, sw_txhdr->sn))
            *cred = rwnx_move_baw(baw);
        else
            *cred = 0;
    }

#endif /* CONFIG_RWNX_AGG_TX */
    return 0;
}

/**
 * rwnx_update_txstatus - Update txinfo with rate used for singleton transmission
 *
 * @rwnx_txst: Transmission status
 * @tx_info: mac80211 structure to be updated
 *
 */
static inline void rwnx_update_txstatus(struct rwnx_hw *rwnx_hw,
                                        struct rwnx_txhdr *txhdr,
                                        struct ieee80211_tx_info *tx_info,
                                        union rwnx_hw_txstatus rwnx_txst)
{
    int i, tries;
    u32 bwTx;

    for (i = 0, tries = rwnx_txst.num_mpdu_retries + 1;
         i < rwnx_hw->hw->max_rates && tries && tx_info->status.rates[i].idx != -1;
         i++, tries--)
        tx_info->status.rates[i].count = 1;

    if (tries)
        tx_info->status.rates[i - 1].count += tries;
    else
        tx_info->status.rates[i].idx = -1;

    bwTx = txhdr->hw_hdr.policy.ratectrlinfos[i - 1].bwTx;
    if (rwnx_txst.transmission_bw != bwTx) {
        WARN_ON(rwnx_txst.transmission_bw > bwTx);
        rwnx_down_rate(&tx_info->status.rates[i - 1],
                       rwnx_txst.transmission_bw);
    }
}

/**
 * rwnx_free_sw_txhdr - Free resources used by a frame TX descriptor
 *
 * @rwnx_hw: Main driver data
 * @sw_txhdr: Software descriptor to free
 */
static inline void rwnx_free_sw_txhdr(struct rwnx_hw *rwnx_hw,
                                      struct rwnx_sw_txhdr *sw_txhdr)
{

#ifdef CONFIG_XXX_AMSDUS_TX
    if (rwnx_is_amsdu(sw_txhdr))
        rwnx_free_amsdusubs(rwnx_hw, sw_txhdr);
#endif

    /* unmap with the least costly DMA_TO_DEVICE since we don't need to inval */
    dma_unmap_single(rwnx_hw->dev, sw_txhdr->dma_addr, sw_txhdr->map_len,
                     DMA_TO_DEVICE);

    skb_pull(sw_txhdr->skb, sw_txhdr->headroom);
    kmem_cache_free(rwnx_hw->sw_txhdr_cache, sw_txhdr);
}


/**
 * rwnx_txdatacfm - TX confirmation callback function
 *
 * @pthis: Pointer on driver main data
 * @host_id: Pointer of the skb
 *
 * @return: 0 if confirmation has been processed and !=0 if transmission is
 * not done yet.
 */
int rwnx_txdatacfm(void *pthis, void *host_id)
{
    struct rwnx_hw *rwnx_hw = (struct rwnx_hw *)pthis;
    struct sk_buff *skb = host_id;
    struct ieee80211_tx_info *tx_info = IEEE80211_SKB_CB(skb);
    struct rwnx_txhdr *txhdr = (struct rwnx_txhdr *)skb->data;
    union rwnx_hw_txstatus rwnx_txst;
    struct rwnx_sw_txhdr *sw_txhdr;
    struct rwnx_txq *txq;
    int cred = 1;
    bool processed = false;

    txhdr = (struct rwnx_txhdr *)skb->data;
    if (rwnx_get_hw_txstatus(rwnx_hw, txhdr, &rwnx_txst))
        return -1;

    sw_txhdr = txhdr->sw_hdr;
    txq = sw_txhdr->txq;
    rwnx_txq_confirm_any(rwnx_hw, txq, txq->hwq, sw_txhdr);

    REG_SW_SET_PROFILING_CHAN(rwnx_hw, SW_PROF_CHAN_CTXT_CFM_BIT);

    if (rwnx_txst.descriptor_done_swtx) {
        if (time_is_after_jiffies(sw_txhdr->first_jiffies +
                                  msecs_to_jiffies(rwnx_hw->mod_params->tx_lft))) {
            rwnx_tx_retry(rwnx_hw, txhdr);
            goto end;
        }
        rwnx_txst.frm_successful_tx = 0;
        rwnx_txst.which_descriptor_sw = 0;
    }

#ifdef CONFIG_XXX_AMSDUS_TX
    rwnx_amsdu_stat_update(rwnx_hw, sw_txhdr, !rwnx_txst.frm_successful_tx);
#endif

    if (rwnx_process_ampdu_cfm(rwnx_hw, txhdr, tx_info, rwnx_txst, &cred, &processed))
        goto end;

    if (!processed) {
        rwnx_update_txstatus(rwnx_hw, txhdr, tx_info, rwnx_txst);
    }

    if (rwnx_txst.frm_successful_tx && !(tx_info->flags & IEEE80211_TX_CTL_NO_ACK))
        tx_info->flags |= IEEE80211_TX_STAT_ACK;

    trace_skb_confirm(skb, txq, txq->hwq, (u8)cred);
    txq->credits += cred;
    if (txq->credits > 0)
        rwnx_txq_start(txq, RWNX_TXQ_STOP_FULL);

    /* continue service period */
    if (unlikely(txq->push_limit && !rwnx_txq_is_full(txq))) {
        rwnx_txq_add_to_hw_list(txq);
    }

    rwnx_free_sw_txhdr(rwnx_hw, sw_txhdr);

#ifdef CONFIG_RWNX_BCN
    if (unlikely(ieee80211_is_beacon(sw_txhdr->frame_ctl))) {
        dev_kfree_skb(skb);
        goto end;
    }
#endif

    ieee80211_tx_status(rwnx_hw->hw, skb);

end:
    REG_SW_CLEAR_PROFILING_CHAN(rwnx_hw, SW_PROF_CHAN_CTXT_CFM_BIT);
    return 0;
}

#ifdef CONFIG_RWNX_BCN
/**
 * rwnx_tx_bcns - Send Beacon frame
 *
 * @rwnx: Main driver data
 */
void rwnx_tx_bcns(struct rwnx_hw *rwnx_hw)
{
    struct sk_buff *skb;
    struct rwnx_vif *rwnx_vif;

    list_for_each_entry(rwnx_vif, &rwnx_hw->vifs, list) {
        struct ieee80211_vif *vif;
        struct ieee80211_tx_info *tx_info;

        if (unlikely(!rwnx_vif->bcn_on) ||
            unlikely(ieee80211_queue_stopped(rwnx_hw->hw, RWNX_HWQ_BCN)) ||
            !(skb = ieee80211_beacon_get(rwnx_hw->hw, vif = rwnx_vif->vif)))
            continue;
        tx_info = IEEE80211_SKB_CB(skb);
        tx_info->hw_queue = RWNX_HWQ_BCN;

        rwnx_ops_tx(rwnx_hw->hw, NULL, skb);

        while (!ieee80211_queue_stopped(rwnx_hw->hw, RWNX_HWQ_BCN) &&
               (skb = ieee80211_get_buffered_bc(rwnx_hw->hw, vif))) {
            rwnx_ops_tx(rwnx_hw->hw, NULL, skb);
        }

        if (vif->csa_active && ieee80211_csa_is_complete(vif))
            ieee80211_csa_finish(vif);
    }
}
#endif
