/**
 ****************************************************************************************
 *
 * @file rwnx_tx.h
 *
 * Copyright (C) RivieraWaves 2012-2019
 *
 ****************************************************************************************
 */
#ifndef _RWNX_TX_H_
#define _RWNX_TX_H_

#include "rwnx_defs.h"
#include "hal_desc.h"
#include "ipc_shared.h"

#define RWNX_HWQ_BK  0
#define RWNX_HWQ_BE  1
#define RWNX_HWQ_VI  2
#define RWNX_HWQ_VO  3
#define RWNX_HWQ_BCN 4
#define RWNX_HWQ_NB  NX_TXQ_CNT
#define RWNX_HWQ_ALL_ACS (RWNX_HWQ_BK | RWNX_HWQ_BE | RWNX_HWQ_VI | RWNX_HWQ_VO)

#define RWNX_TX_LIFETIME_MS 100
#define RWNX_TX_MAX_RATES   NX_TX_MAX_RATES

#define RWNX_SWTXHDR_ALIGN_SZ 4
#define RWNX_SWTXHDR_ALIGN_MSK (RWNX_SWTXHDR_ALIGN_SZ - 1)
#define RWNX_SWTXHDR_ALIGN_PADS(x) \
                    ((RWNX_SWTXHDR_ALIGN_SZ - ((x) & RWNX_SWTXHDR_ALIGN_MSK)) \
                     & RWNX_SWTXHDR_ALIGN_MSK)
#if RWNX_SWTXHDR_ALIGN_SZ & RWNX_SWTXHDR_ALIGN_MSK
#error bad RWNX_SWTXHDR_ALIGN_SZ
#endif

#if defined(CONFIG_RWNX_AMSDUS_TX) || defined(CONFIG_MAC80211_AMSDUS_TX)
#define CONFIG_XXX_AMSDUS_TX
#endif

#ifdef CONFIG_RWNX_AMSDUS_TX
/**
 * struct rwnx_amsdu_txhdr - Tx descriptor for AMSDU subframe.
 * Stored inside skb headroom instead fo rwnx_txhdr
 *
 * @list: List of amsdu subframe (in rwnx_sw_txhdr.amsdu.hdrs)
 * @map_len: Size mapped for DMA transfer (AMDSU header + data)
 * @dma_addr: DMA mapped address
 * @skb: Buffer of the AMSDU subframe
 */
struct rwnx_amsdu_txhdr {
    struct list_head list;
    size_t map_len;
    dma_addr_t dma_addr;
    struct sk_buff *skb;
};

/**
 * struct rwnx_amsdu - AMSDU descriptor
 *
 * @hdrs: List of subframe (struct rwnx_amsdu_txhdr)
 * @rem_len: Remaining size, in bytes, available for more subframes in this AMSDU
 * @nb: Number of subframe in this AMSDU. (0 means that this frame is not an AMSDU)
 * @icv_len: Size, in bytes, needed for ICV
 */
struct rwnx_amsdu {
    struct list_head hdrs;
    u16 rem_len;
    u16 nb;
    u8 icv_len;
};

#endif /* CONFIG_RWNX_AMSDUS_TX */

/**
 * struct rwnx_sw_txhdr - Software TX descriptor.
 * contains information needed by driver to send a frame
 *
 * @skb: Buffer containing frame data
 * @rwnx_sta: Pointer to the destination STA (may be NULL)
 * @txq: TX queue to be used to transmit this buffer
 * @baw: Block Ack Window to use for transmission (may be NULL)
 * @ba_idx: Block Ack ID. Used to check if Block Ack changed between the time
 * the TX descriptor has been prepared and the time the frame is actually
 * pushed to the firmware. If so the frame is sent as singleton.
 * @ampdu_status: AMDPU transmission status. Only used when the last MPDU of
 * the AMPDU needs to be retried. In this case AMPDU status cannot be sent to
 * mac80211 until this frame is successfully retransmitted (or its lifetime
 * expired), so it is saved here.
 * @amsdu: AMSDU descriptor. Updated only if the frame is the first subframe
 * of an AMSDU
 * @frame_len: Size, in bytes, of the frame
 * @frame_ctl: Frame control field of the frame (copied from MAC header)
 * @sn: Sequence number of the frame (copied from MAC header)
 * @tid: TID to use for transmission
 * @headroom: Size used in the skb to store TX descriptor
 * @desc: Part of the descriptor that is copied in shared memory for the firmware
 * @map_len: Size mapped for DMA transfer (TX descriptor + frame)
 * @dma_addr: DMA mapped address
 * @first_jiffies: Date when the buffer was received from mac80211.
 * Used to respect buffer lifetime
 */
struct rwnx_sw_txhdr {
    struct sk_buff *skb;
    struct rwnx_sta *rwnx_sta;
    struct rwnx_txq *txq;
    struct rwnx_baw *baw;
    int ba_idx;
    struct rwnx_baw_ampdu_status ampdu_status;
#ifdef CONFIG_RWNX_AMSDUS_TX
    struct rwnx_amsdu amsdu;
#endif
    u16 frame_len;
    __le16 frame_ctl;
    u16 sn;
    u8 tid;
    u16 headroom;
    struct txdesc_api desc;
    size_t map_len;
    dma_addr_t dma_addr;
    unsigned long first_jiffies;
};

/**
 * struct rwnx_txhdr - TX descriptor (Stored inside skb headroom)
 *
 * @sw_hdr: Pointer to software descriptor. To limit needed headroom software
 * descriptor is allocated elsewhere. Indeed, if headroom needed is too large
 * the IP/TCP stack will have to expand the buffer (i.e. make a copy).
 * @cache_guard:
 * @hw_hdr: Hardware descriptor. Downloaded by the firmware while downloading
 * the frame as it is placed just before.
 */
struct rwnx_txhdr {
    struct rwnx_sw_txhdr *sw_hdr;
    char cache_guard[L1_CACHE_BYTES];
    struct rwnx_hw_txhdr hw_hdr;
};


void rwnx_ops_tx(struct ieee80211_hw *hw, struct ieee80211_tx_control *control,
                 struct sk_buff *skb);
#ifdef CONFIG_RWNX_AMSDUS_TX
int rwnx_ops_agg_msdu(struct ieee80211_hw *hw, struct amsdu_ctl *ctl,
                      struct sk_buff *skb);
#endif
#ifdef CONFIG_MAC80211_TXQ
void rwnx_ops_wake_tx_queue(struct ieee80211_hw *hw, struct ieee80211_txq *txq);
struct sk_buff *rwnx_tx_dequeue_prep(struct rwnx_hw *rwnx_hw,
                                     struct ieee80211_txq *txq);
#endif

int rwnx_txdatacfm(void *pthis, void *host_id);

void rwnx_tx_bcns(struct rwnx_hw *rwnx_hw);
void rwnx_tx_flushq(struct rwnx_hw *rwnx_hw, struct rwnx_txq *txq);
void rwnx_tx_push(struct rwnx_hw *rwnx_hw, struct rwnx_txhdr *txhdr, int flags);

#endif /* _RWNX_TX_H_ */
