//========= Copyright  1996-2005, Valve Corporation, All rights reserved. ============//
//
// Purpose: Activities that are available to all NPCs.
//
//=============================================================================//

#include "cbase.h"
#include "ai_activity.h"
#include "ai_basenpc.h"
#include "stringregistry.h"

// memdbgon must be the last include file in a .cpp file!!!
#include "tier0/memdbgon.h"

//=============================================================================
// Init static variables
//=============================================================================
CStringRegistry* CAI_BaseNPC::m_pActivitySR	= NULL;
int				 CAI_BaseNPC::m_iNumActivities = 0;

//-----------------------------------------------------------------------------
// Purpose: Add an activity to the activity string registry and increment
//			the acitivty counter
//-----------------------------------------------------------------------------
void CAI_BaseNPC::AddActivityToSR(const char *actName, int actID) 
{
	Assert( m_pActivitySR );
	if ( !m_pActivitySR )
		return;

	// technically order isn't dependent, but it's too damn easy to forget to add new ACT_'s to all three lists.

	// NOTE: This assertion generally means that the activity enums are out of order or that new enums were not added to all
	//		 relevant tables.  Make sure that you have included all new enums in:
	//			game_shared/ai_activity.h
	//			game_shared/activitylist.cpp
	//			dlls/ai_activity.cpp
	MEM_ALLOC_CREDIT();

	static int lastActID = -2;
	Assert( actID >= LAST_SHARED_ACTIVITY || actID == lastActID + 1 || actID == ACT_INVALID );
	lastActID = actID;

	if ( m_pActivitySR->GetStringID( actName ) != -1 )
	{
		printf( "Duplicate\n" );
	}

	m_pActivitySR->AddString(actName, actID);
	m_iNumActivities++;
}

//-----------------------------------------------------------------------------
// Purpose: Given and activity ID, return the activity name
//-----------------------------------------------------------------------------
const char *CAI_BaseNPC::GetActivityName(int actID) 
{
	if ( actID == -1 )
		return "ACT_INVALID";

	// m_pActivitySR only contains public activities, ActivityList_NameForIndex() has them all
	const char *name = ActivityList_NameForIndex(actID);	

	if( !name )
	{
		AssertOnce( !"CAI_BaseNPC::GetActivityName() returning NULL!" );
	}

	return name;
}

//-----------------------------------------------------------------------------
// Purpose: Given and activity name, return the activity ID
//-----------------------------------------------------------------------------
int CAI_BaseNPC::GetActivityID(const char* actName) 
{
	Assert( m_pActivitySR );
	if ( !m_pActivitySR )
		return ACT_INVALID;

	return m_pActivitySR->GetStringID(actName);
}

#define ADD_ACTIVITY_TO_SR(activityname) AddActivityToSR(#activityname,activityname)

//-----------------------------------------------------------------------------
// Purpose:
//-----------------------------------------------------------------------------
void CAI_BaseNPC::InitDefaultActivitySR(void) 
{
	ADD_ACTIVITY_TO_SR(ACT_INVALID);
	ADD_ACTIVITY_TO_SR(ACT_RESET);				// Set m_Activity to this invalid value to force a reset to m_IdealActivity
	ADD_ACTIVITY_TO_SR(ACT_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_TRANSITION);
	ADD_ACTIVITY_TO_SR(ACT_COVER);					// FIXME: obsolete? redundant with ADD_ACTIVITY_TO_SR( ACT_COVER_LOW?
	ADD_ACTIVITY_TO_SR(ACT_COVER_MED);				// FIXME: unsupported?
	ADD_ACTIVITY_TO_SR(ACT_COVER_LOW);				// FIXME: rename ADD_ACTIVITY_TO_SR( ACT_IDLE_CROUCH?
	ADD_ACTIVITY_TO_SR(ACT_WALK);
	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM);
	ADD_ACTIVITY_TO_SR(ACT_WALK_CROUCH);
	ADD_ACTIVITY_TO_SR(ACT_WALK_CROUCH_AIM);
	ADD_ACTIVITY_TO_SR(ACT_RUN);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM);
	ADD_ACTIVITY_TO_SR(ACT_RUN_CROUCH);
	ADD_ACTIVITY_TO_SR(ACT_RUN_CROUCH_AIM);
	ADD_ACTIVITY_TO_SR(ACT_RUN_PROTECTED);
	ADD_ACTIVITY_TO_SR(ACT_SCRIPT_CUSTOM_MOVE);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK1);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK2);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK1_LOW);		// FIXME: not used yet ); crouched versions of the range attack
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK2_LOW);		// FIXME: not used yet ); crouched versions of the range attack
	ADD_ACTIVITY_TO_SR(ACT_DIESIMPLE);
	ADD_ACTIVITY_TO_SR(ACT_DIEBACKWARD);
	ADD_ACTIVITY_TO_SR(ACT_DIEFORWARD);
	ADD_ACTIVITY_TO_SR(ACT_DIEVIOLENT);
	ADD_ACTIVITY_TO_SR(ACT_DIERAGDOLL);
	ADD_ACTIVITY_TO_SR(ACT_FLY);				// Fly (and flap if appropriate)
	ADD_ACTIVITY_TO_SR(ACT_HOVER);
	ADD_ACTIVITY_TO_SR(ACT_GLIDE);
	ADD_ACTIVITY_TO_SR(ACT_SWIM);
	ADD_ACTIVITY_TO_SR(ACT_JUMP);
	ADD_ACTIVITY_TO_SR(ACT_HOP);				// vertical jump
	ADD_ACTIVITY_TO_SR(ACT_LEAP);				// long forward jump
	ADD_ACTIVITY_TO_SR(ACT_LAND);
	ADD_ACTIVITY_TO_SR(ACT_CLIMB_UP);
	ADD_ACTIVITY_TO_SR(ACT_CLIMB_DOWN);
	ADD_ACTIVITY_TO_SR(ACT_CLIMB_DISMOUNT);
	ADD_ACTIVITY_TO_SR(ACT_SHIPLADDER_UP);
	ADD_ACTIVITY_TO_SR(ACT_SHIPLADDER_DOWN);
	ADD_ACTIVITY_TO_SR(ACT_STRAFE_LEFT);
	ADD_ACTIVITY_TO_SR(ACT_STRAFE_RIGHT);
	ADD_ACTIVITY_TO_SR(ACT_ROLL_LEFT);			// tuck and roll ); left
	ADD_ACTIVITY_TO_SR(ACT_ROLL_RIGHT);			// tuck and roll ); right
	ADD_ACTIVITY_TO_SR(ACT_TURN_LEFT);			// turn quickly left (stationary)
	ADD_ACTIVITY_TO_SR(ACT_TURN_RIGHT);			// turn quickly right (stationary)
	ADD_ACTIVITY_TO_SR(ACT_CROUCH);				// FIXME: obsolete? only used be soldier (the act of crouching down from a standing position)
	ADD_ACTIVITY_TO_SR(ACT_CROUCHIDLE);			// FIXME: obsolete? only used be soldier (holding body in crouched position (loops))
	ADD_ACTIVITY_TO_SR(ACT_STAND);				// FIXME: obsolete? should be transition (the act of standing from a crouched position)
	ADD_ACTIVITY_TO_SR(ACT_USE);
	ADD_ACTIVITY_TO_SR(ACT_ALIEN_BURROW_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_ALIEN_BURROW_OUT);

	ADD_ACTIVITY_TO_SR(ACT_SIGNAL1);
	ADD_ACTIVITY_TO_SR(ACT_SIGNAL2);
	ADD_ACTIVITY_TO_SR(ACT_SIGNAL3);

	ADD_ACTIVITY_TO_SR(ACT_SIGNAL_ADVANCE);		// Squad handsignals ); specific.
	ADD_ACTIVITY_TO_SR(ACT_SIGNAL_FORWARD);
	ADD_ACTIVITY_TO_SR(ACT_SIGNAL_GROUP);
	ADD_ACTIVITY_TO_SR(ACT_SIGNAL_HALT);
	ADD_ACTIVITY_TO_SR(ACT_SIGNAL_LEFT);
	ADD_ACTIVITY_TO_SR(ACT_SIGNAL_RIGHT);
	ADD_ACTIVITY_TO_SR(ACT_SIGNAL_TAKECOVER);

	ADD_ACTIVITY_TO_SR(ACT_LOOKBACK_RIGHT);		// look back over shoulder without turning around.
	ADD_ACTIVITY_TO_SR(ACT_LOOKBACK_LEFT);
	ADD_ACTIVITY_TO_SR(ACT_COWER);				// FIXME: unused ); should be more extreme version of crouching
	ADD_ACTIVITY_TO_SR(ACT_SMALL_FLINCH);		// FIXME: needed? shouldn't flinching be down with overlays?
	ADD_ACTIVITY_TO_SR(ACT_BIG_FLINCH);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_ATTACK1);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_ATTACK2);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_START);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_FINISH);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_LOW);
	ADD_ACTIVITY_TO_SR(ACT_ARM);				// pull out gun ); for instance
	ADD_ACTIVITY_TO_SR(ACT_DISARM);				// reholster gun
	ADD_ACTIVITY_TO_SR(ACT_DROP_WEAPON);
	ADD_ACTIVITY_TO_SR(ACT_DROP_WEAPON_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_PICKUP_GROUND);		// pick up something in front of you on the ground
	ADD_ACTIVITY_TO_SR(ACT_PICKUP_RACK);		// pick up something from a rack or shelf in front of you.
	ADD_ACTIVITY_TO_SR(ACT_IDLE_ANGRY);			// FIXME: being used as an combat ready idle?  alternate idle animation in which the monster is clearly agitated. (loop)

	ADD_ACTIVITY_TO_SR(ACT_IDLE_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_AGITATED);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_STEALTH);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_HURT);

	ADD_ACTIVITY_TO_SR(ACT_WALK_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_WALK_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_WALK_AGITATED);
	ADD_ACTIVITY_TO_SR(ACT_WALK_STEALTH);

	ADD_ACTIVITY_TO_SR(ACT_RUN_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_RUN_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AGITATED);
	ADD_ACTIVITY_TO_SR(ACT_RUN_STEALTH);

	ADD_ACTIVITY_TO_SR(ACT_IDLE_AIM_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_AIM_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_AIM_AGITATED);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_AIM_STEALTH);

	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM_AGITATED);
	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM_STEALTH);

	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM_AGITATED);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM_STEALTH);

	ADD_ACTIVITY_TO_SR(ACT_CROUCHIDLE_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_CROUCHIDLE_AIM_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_CROUCHIDLE_AGITATED);

	ADD_ACTIVITY_TO_SR(ACT_WALK_HURT);			// limp  (loop)
	ADD_ACTIVITY_TO_SR(ACT_RUN_HURT);			// limp  (loop)
	ADD_ACTIVITY_TO_SR(ACT_SPECIAL_ATTACK1);	// very monster specific special attacks.
	ADD_ACTIVITY_TO_SR(ACT_SPECIAL_ATTACK2);
	ADD_ACTIVITY_TO_SR(ACT_COMBAT_IDLE);		// FIXME: unused?  agitated idle.
	ADD_ACTIVITY_TO_SR(ACT_WALK_SCARED);
	ADD_ACTIVITY_TO_SR(ACT_RUN_SCARED);
	ADD_ACTIVITY_TO_SR(ACT_VICTORY_DANCE);		// killed a player ); do a victory dance.
	ADD_ACTIVITY_TO_SR(ACT_DIE_HEADSHOT);		// die ); hit in head. 
	ADD_ACTIVITY_TO_SR(ACT_DIE_CHESTSHOT);		// die ); hit in chest
	ADD_ACTIVITY_TO_SR(ACT_DIE_GUTSHOT);		// die ); hit in gut
	ADD_ACTIVITY_TO_SR(ACT_DIE_BACKSHOT);		// die ); hit in back
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_HEAD);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_CHEST);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_STOMACH);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_LEFTARM);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_RIGHTARM);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_LEFTLEG);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_RIGHTLEG);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_PHYSICS);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_HEAD_BACK);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_CHEST_BACK);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_STOMACH_BACK);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_CROUCH_FRONT);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_CROUCH_BACK);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_CROUCH_LEFT);
	ADD_ACTIVITY_TO_SR(ACT_FLINCH_CROUCH_RIGHT);

	ADD_ACTIVITY_TO_SR(ACT_IDLE_ON_FIRE);		// ON FIRE animations
	ADD_ACTIVITY_TO_SR(ACT_WALK_ON_FIRE);
	ADD_ACTIVITY_TO_SR(ACT_RUN_ON_FIRE);

	ADD_ACTIVITY_TO_SR(ACT_RAPPEL_LOOP);		// Rappel down a rope!

	ADD_ACTIVITY_TO_SR(ACT_180_LEFT);			// 180 degree left turn
	ADD_ACTIVITY_TO_SR(ACT_180_RIGHT);

	ADD_ACTIVITY_TO_SR(ACT_90_LEFT);			// 90 degree turns
	ADD_ACTIVITY_TO_SR(ACT_90_RIGHT);

	ADD_ACTIVITY_TO_SR(ACT_STEP_LEFT);			// Single steps
	ADD_ACTIVITY_TO_SR(ACT_STEP_RIGHT);
	ADD_ACTIVITY_TO_SR(ACT_STEP_BACK);
	ADD_ACTIVITY_TO_SR(ACT_STEP_FORE);

	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK1);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK2);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_MELEE_ATTACK1);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_MELEE_ATTACK2);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK1_LOW);	// FIXME: not used yet ); crouched versions of the range attack
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK2_LOW);	// FIXME: not used yet ); crouched versions of the range attack

	ADD_ACTIVITY_TO_SR(ACT_MELEE_ATTACK_SWING_GESTURE);

	ADD_ACTIVITY_TO_SR(ACT_GESTURE_SMALL_FLINCH);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_BIG_FLINCH);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_FLINCH_BLAST);			// Startled by an explosion
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_FLINCH_BLAST_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_FLINCH_BLAST_DAMAGED);	// Damaged by an explosion
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_FLINCH_BLAST_DAMAGED_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_FLINCH_HEAD);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_FLINCH_CHEST);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_FLINCH_STOMACH);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_FLINCH_LEFTARM);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_FLINCH_RIGHTARM);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_FLINCH_LEFTLEG);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_FLINCH_RIGHTLEG);

	ADD_ACTIVITY_TO_SR(ACT_GESTURE_TURN_LEFT);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_TURN_RIGHT);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_TURN_LEFT45);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_TURN_RIGHT45);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_TURN_LEFT90);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_TURN_RIGHT90);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_TURN_LEFT45_FLAT);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_TURN_RIGHT45_FLAT);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_TURN_LEFT90_FLAT);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_TURN_RIGHT90_FLAT);

	// HALF-LIFE 1 compatability stuff goes here. Temporary!
	ADD_ACTIVITY_TO_SR(ACT_BARNACLE_HIT);		// barnacle tongue hits a monster
	ADD_ACTIVITY_TO_SR(ACT_BARNACLE_PULL);		// barnacle is lifting the monster ( loop )
	ADD_ACTIVITY_TO_SR(ACT_BARNACLE_CHOMP);		// barnacle latches on to the monster
	ADD_ACTIVITY_TO_SR(ACT_BARNACLE_CHEW);		// barnacle is holding the monster in its mouth ( loop )

														// Sometimes ); you just want to set an NPC's sequence to a sequence that doesn't actually
														// have an activity. The AI will reset the NPC's sequence to whatever its IDEAL activity
														// is ); though. So if you set ideal activity to DO_NOT_DISTURB ); the AI will not interfere
														// with the NPC's current sequence. (SJB)
	ADD_ACTIVITY_TO_SR(ACT_DO_NOT_DISTURB);

	ADD_ACTIVITY_TO_SR(ACT_SPECIFIC_SEQUENCE);

	// viewmodel (weapon) activities
	// FIXME: move these to the specific viewmodels ); no need to make global
	ADD_ACTIVITY_TO_SR(ACT_VM_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_VM_HOLSTER);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_VM_FIDGET);
	ADD_ACTIVITY_TO_SR(ACT_VM_PULLBACK);
	ADD_ACTIVITY_TO_SR(ACT_VM_PULLBACK_HIGH);
	ADD_ACTIVITY_TO_SR(ACT_VM_PULLBACK_LOW);
	ADD_ACTIVITY_TO_SR(ACT_VM_THROW);
	ADD_ACTIVITY_TO_SR(ACT_VM_PULLPIN);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK);		// fire
	ADD_ACTIVITY_TO_SR(ACT_VM_SECONDARYATTACK);		// alt. fire
	ADD_ACTIVITY_TO_SR(ACT_VM_RELOAD);
	ADD_ACTIVITY_TO_SR(ACT_VM_DRYFIRE);				// fire with no ammo loaded.
	ADD_ACTIVITY_TO_SR(ACT_VM_HITLEFT);				// bludgeon ); swing to left - hit (primary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_HITLEFT2);			// bludgeon ); swing to left - hit (secondary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_HITRIGHT);			// bludgeon ); swing to right - hit (primary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_HITRIGHT2);			// bludgeon ); swing to right - hit (secondary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_HITCENTER);			// bludgeon ); swing center - hit (primary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_HITCENTER2);			// bludgeon ); swing center - hit (secondary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_MISSLEFT);			// bludgeon ); swing to left - miss (primary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_MISSLEFT2);			// bludgeon ); swing to left - miss (secondary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_MISSRIGHT);			// bludgeon ); swing to right - miss (primary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_MISSRIGHT2);			// bludgeon ); swing to right - miss (secondary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_MISSCENTER);			// bludgeon ); swing center - miss (primary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_MISSCENTER2);			// bludgeon ); swing center - miss (secondary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_HAULBACK);			// bludgeon ); haul the weapon back for a hard strike (secondary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_SWINGHARD);			// bludgeon ); release the hard strike (secondary attk)
	ADD_ACTIVITY_TO_SR(ACT_VM_SWINGMISS);
	ADD_ACTIVITY_TO_SR(ACT_VM_SWINGHIT);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_TO_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_VM_LOWERED_TO_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_VM_RECOIL1);
	ADD_ACTIVITY_TO_SR(ACT_VM_RECOIL2);
	ADD_ACTIVITY_TO_SR(ACT_VM_RECOIL3);
	ADD_ACTIVITY_TO_SR(ACT_VM_PICKUP);
	ADD_ACTIVITY_TO_SR(ACT_VM_RELEASE);

	ADD_ACTIVITY_TO_SR(ACT_VM_ATTACH_SILENCER);
	ADD_ACTIVITY_TO_SR(ACT_VM_DETACH_SILENCER);

	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_L);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_R);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_RL);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_L_RECOIL1);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_L_RECOIL2);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_L_RECOIL3);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_R_RECOIL1);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_R_RECOIL2);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_R_RECOIL3);

	//===========================
	// HL2 Specific Activities
	//===========================
	// SLAM	Specialty Activities
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_ND_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_ATTACH);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_ATTACH2);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_ND_ATTACH);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_ND_ATTACH2);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_DETONATE);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_DETONATOR_HOLSTER);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_ND_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_TO_THROW);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_TO_THROW_ND);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_STICKWALL_TO_TRIPMINE_ND);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_ND_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_THROW);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_THROW2);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_THROW_ND);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_THROW_ND2);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_ND_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_TO_STICKWALL);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_TO_STICKWALL_ND);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_DETONATE);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_DETONATOR_HOLSTER);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_THROW_TO_TRIPMINE_ND);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_TRIPMINE_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_TRIPMINE_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_TRIPMINE_ATTACH);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_TRIPMINE_ATTACH2);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_TRIPMINE_TO_STICKWALL_ND);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_TRIPMINE_TO_THROW_ND);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_DETONATOR_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_DETONATOR_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_DETONATOR_DETONATE);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_DETONATOR_HOLSTER);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_DETONATOR_STICKWALL_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_SLAM_DETONATOR_THROW_DRAW);

	// Shotgun Specialty Activities
	ADD_ACTIVITY_TO_SR(ACT_SHOTGUN_RELOAD_START);
	ADD_ACTIVITY_TO_SR(ACT_SHOTGUN_RELOAD_FINISH);
	ADD_ACTIVITY_TO_SR(ACT_SHOTGUN_PUMP);

	// SMG2 special activities
	ADD_ACTIVITY_TO_SR(ACT_SMG2_IDLE2);
	ADD_ACTIVITY_TO_SR(ACT_SMG2_FIRE2);
	ADD_ACTIVITY_TO_SR(ACT_SMG2_DRAW2);
	ADD_ACTIVITY_TO_SR(ACT_SMG2_RELOAD2);
	ADD_ACTIVITY_TO_SR(ACT_SMG2_DRYFIRE2);
	ADD_ACTIVITY_TO_SR(ACT_SMG2_TOAUTO);
	ADD_ACTIVITY_TO_SR(ACT_SMG2_TOBURST);

	// Physcannon special activities
	ADD_ACTIVITY_TO_SR(ACT_PHYSCANNON_UPGRADE);

	// weapon override activities
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_AR1);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_AR2);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_AR2_LOW);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_AR2_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_HMG1);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_ML);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_SMG1_LOW);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_SMG2);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_SHOTGUN_LOW);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_PISTOL_LOW);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_SLAM);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_TRIPWIRE);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_THROW);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_SNIPER_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_RPG);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_ATTACK_SWING);

	ADD_ACTIVITY_TO_SR(ACT_RANGE_AIM_LOW);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_AIM_SMG1_LOW);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_AIM_PISTOL_LOW);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_AIM_AR2_LOW);

	ADD_ACTIVITY_TO_SR(ACT_COVER_PISTOL_LOW);
	ADD_ACTIVITY_TO_SR(ACT_COVER_SMG1_LOW);

	// weapon override activities
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_AR1);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_AR2);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_AR2_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_HMG1);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_ML);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_SMG1_LOW);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_SMG2);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_PISTOL_LOW);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_SLAM);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_TRIPWIRE);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_THROW);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RANGE_ATTACK_SNIPER_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_MELEE_ATTACK_SWING);

	ADD_ACTIVITY_TO_SR(ACT_IDLE_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_ANGRY_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_ANGRY_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_ANGRY_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_STEALTH_PISTOL);

	ADD_ACTIVITY_TO_SR(ACT_IDLE_PACKAGE);
	ADD_ACTIVITY_TO_SR(ACT_WALK_PACKAGE);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_SUITCASE);
	ADD_ACTIVITY_TO_SR(ACT_WALK_SUITCASE);

	ADD_ACTIVITY_TO_SR(ACT_IDLE_SMG1_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_SMG1_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_WALK_RIFLE_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_RUN_RIFLE_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_WALK_RIFLE_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_RUN_RIFLE_STIMULATED);

	ADD_ACTIVITY_TO_SR(ACT_IDLE_AIM_RIFLE_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM_RIFLE_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM_RIFLE_STIMULATED);

	ADD_ACTIVITY_TO_SR(ACT_IDLE_SHOTGUN_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_SHOTGUN_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_SHOTGUN_AGITATED);

	// Policing activities
	ADD_ACTIVITY_TO_SR(ACT_WALK_ANGRY);
	ADD_ACTIVITY_TO_SR(ACT_POLICE_HARASS1);
	ADD_ACTIVITY_TO_SR(ACT_POLICE_HARASS2);

	// Manned guns
	ADD_ACTIVITY_TO_SR(ACT_IDLE_MANNEDGUN);

	// Melee weapon
	ADD_ACTIVITY_TO_SR(ACT_IDLE_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_ANGRY_MELEE);

	// RPG activities
	ADD_ACTIVITY_TO_SR(ACT_IDLE_RPG_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_RPG);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_ANGRY_RPG);
	ADD_ACTIVITY_TO_SR(ACT_COVER_LOW_RPG);
	ADD_ACTIVITY_TO_SR(ACT_WALK_RPG);
	ADD_ACTIVITY_TO_SR(ACT_RUN_RPG);
	ADD_ACTIVITY_TO_SR(ACT_WALK_CROUCH_RPG);
	ADD_ACTIVITY_TO_SR(ACT_RUN_CROUCH_RPG);
	ADD_ACTIVITY_TO_SR(ACT_WALK_RPG_RELAXED);
	ADD_ACTIVITY_TO_SR(ACT_RUN_RPG_RELAXED);

	ADD_ACTIVITY_TO_SR(ACT_WALK_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_WALK_CROUCH_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_WALK_CROUCH_AIM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_RUN_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_RUN_CROUCH_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_RUN_CROUCH_AIM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_RUN_STEALTH_PISTOL);

	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM_SHOTGUN);

	ADD_ACTIVITY_TO_SR(ACT_WALK_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_RUN_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_WALK_STEALTH_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM_STEALTH_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM_STEALTH_PISTOL);

	// Reloads
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_PISTOL_LOW);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_SMG1_LOW);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_SHOTGUN_LOW);

	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RELOAD);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RELOAD_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RELOAD_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_RELOAD_SHOTGUN);

	// Busy animations
	ADD_ACTIVITY_TO_SR(ACT_BUSY_LEAN_LEFT);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_LEAN_LEFT_ENTRY);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_LEAN_LEFT_EXIT);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_LEAN_BACK);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_LEAN_BACK_ENTRY);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_LEAN_BACK_EXIT);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_SIT_GROUND);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_SIT_GROUND_ENTRY);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_SIT_GROUND_EXIT);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_SIT_CHAIR);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_SIT_CHAIR_ENTRY);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_SIT_CHAIR_EXIT);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_STAND);
	ADD_ACTIVITY_TO_SR(ACT_BUSY_QUEUE);

	// Dodge animations
	ADD_ACTIVITY_TO_SR(ACT_DUCK_DODGE);

	// For NPCs being lifted/eaten by barnacles:
	// being swallowed by a barnacle
	ADD_ACTIVITY_TO_SR(ACT_DIE_BARNACLE_SWALLOW);
	// being lifted by a barnacle
	ADD_ACTIVITY_TO_SR(ACT_GESTURE_BARNACLE_STRANGLE);

	ADD_ACTIVITY_TO_SR(ACT_PHYSCANNON_DETACH);	// An activity to be played if we're picking this up with the physcannon
	ADD_ACTIVITY_TO_SR(ACT_PHYSCANNON_ANIMATE); // An activity to be played by an object being picked up with the physcannon ); but has different behavior to DETACH
	ADD_ACTIVITY_TO_SR(ACT_PHYSCANNON_ANIMATE_PRE);	// An activity to be played by an object being picked up with the physcannon ); before playing the ADD_ACTIVITY_TO_SR( ACT_PHYSCANNON_ANIMATE
	ADD_ACTIVITY_TO_SR(ACT_PHYSCANNON_ANIMATE_POST);// An activity to be played by an object being picked up with the physcannon ); after playing the ADD_ACTIVITY_TO_SR( ACT_PHYSCANNON_ANIMATE

	ADD_ACTIVITY_TO_SR(ACT_DIE_FRONTSIDE);
	ADD_ACTIVITY_TO_SR(ACT_DIE_RIGHTSIDE);
	ADD_ACTIVITY_TO_SR(ACT_DIE_BACKSIDE);
	ADD_ACTIVITY_TO_SR(ACT_DIE_LEFTSIDE);

	ADD_ACTIVITY_TO_SR(ACT_OPEN_DOOR);

	// Dynamic interactions
	ADD_ACTIVITY_TO_SR(ACT_DI_ALYX_ZOMBIE_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_DI_ALYX_ZOMBIE_TORSO_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_DI_ALYX_HEADCRAB_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_DI_ALYX_ANTLION);

	ADD_ACTIVITY_TO_SR(ACT_DI_ALYX_ZOMBIE_SHOTGUN64);
	ADD_ACTIVITY_TO_SR(ACT_DI_ALYX_ZOMBIE_SHOTGUN26);

	ADD_ACTIVITY_TO_SR(ACT_READINESS_RELAXED_TO_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_READINESS_RELAXED_TO_STIMULATED_WALK);
	ADD_ACTIVITY_TO_SR(ACT_READINESS_AGITATED_TO_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_READINESS_STIMULATED_TO_RELAXED);

	ADD_ACTIVITY_TO_SR(ACT_READINESS_PISTOL_RELAXED_TO_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_READINESS_PISTOL_RELAXED_TO_STIMULATED_WALK);
	ADD_ACTIVITY_TO_SR(ACT_READINESS_PISTOL_AGITATED_TO_STIMULATED);
	ADD_ACTIVITY_TO_SR(ACT_READINESS_PISTOL_STIMULATED_TO_RELAXED);

	ADD_ACTIVITY_TO_SR(ACT_IDLE_CARRY);
	ADD_ACTIVITY_TO_SR(ACT_WALK_CARRY);

	//===========================
	// TF2 Specific Activities
	//===========================
	ADD_ACTIVITY_TO_SR(ACT_STARTDYING);
	ADD_ACTIVITY_TO_SR(ACT_DYINGLOOP);
	ADD_ACTIVITY_TO_SR(ACT_DYINGTODEAD);

	ADD_ACTIVITY_TO_SR(ACT_RIDE_MANNED_GUN);

	// All viewmodels
	ADD_ACTIVITY_TO_SR(ACT_VM_SPRINT_ENTER);
	ADD_ACTIVITY_TO_SR(ACT_VM_SPRINT_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_VM_SPRINT_LEAVE);

	// Looping weapon firing
	ADD_ACTIVITY_TO_SR(ACT_FIRE_START);
	ADD_ACTIVITY_TO_SR(ACT_FIRE_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_FIRE_END);

	ADD_ACTIVITY_TO_SR(ACT_CROUCHING_GRENADEIDLE);
	ADD_ACTIVITY_TO_SR(ACT_CROUCHING_GRENADEREADY);
	ADD_ACTIVITY_TO_SR(ACT_CROUCHING_PRIMARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_OVERLAY_GRENADEIDLE);
	ADD_ACTIVITY_TO_SR(ACT_OVERLAY_GRENADEREADY);
	ADD_ACTIVITY_TO_SR(ACT_OVERLAY_PRIMARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_OVERLAY_SHIELD_UP);
	ADD_ACTIVITY_TO_SR(ACT_OVERLAY_SHIELD_DOWN);
	ADD_ACTIVITY_TO_SR(ACT_OVERLAY_SHIELD_UP_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_OVERLAY_SHIELD_ATTACK);
	ADD_ACTIVITY_TO_SR(ACT_OVERLAY_SHIELD_KNOCKBACK);
	ADD_ACTIVITY_TO_SR(ACT_SHIELD_UP);
	ADD_ACTIVITY_TO_SR(ACT_SHIELD_DOWN);
	ADD_ACTIVITY_TO_SR(ACT_SHIELD_UP_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_SHIELD_ATTACK);
	ADD_ACTIVITY_TO_SR(ACT_SHIELD_KNOCKBACK);
	ADD_ACTIVITY_TO_SR(ACT_CROUCHING_SHIELD_UP);
	ADD_ACTIVITY_TO_SR(ACT_CROUCHING_SHIELD_DOWN);
	ADD_ACTIVITY_TO_SR(ACT_CROUCHING_SHIELD_UP_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_CROUCHING_SHIELD_ATTACK);
	ADD_ACTIVITY_TO_SR(ACT_CROUCHING_SHIELD_KNOCKBACK);

	// turning in place
	ADD_ACTIVITY_TO_SR(ACT_TURNRIGHT45);
	ADD_ACTIVITY_TO_SR(ACT_TURNLEFT45);

	ADD_ACTIVITY_TO_SR(ACT_TURN);

	ADD_ACTIVITY_TO_SR(ACT_OBJ_ASSEMBLING);
	ADD_ACTIVITY_TO_SR(ACT_OBJ_DISMANTLING);
	ADD_ACTIVITY_TO_SR(ACT_OBJ_STARTUP);
	ADD_ACTIVITY_TO_SR(ACT_OBJ_RUNNING);
	ADD_ACTIVITY_TO_SR(ACT_OBJ_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_OBJ_PLACING);
	ADD_ACTIVITY_TO_SR(ACT_OBJ_DETERIORATING);
	ADD_ACTIVITY_TO_SR(ACT_OBJ_UPGRADING);

	// Deploy
	ADD_ACTIVITY_TO_SR(ACT_DEPLOY);
	ADD_ACTIVITY_TO_SR(ACT_DEPLOY_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_UNDEPLOY);

	// Crossbow
	ADD_ACTIVITY_TO_SR(ACT_CROSSBOW_DRAW_UNLOADED);

	// Gauss
	ADD_ACTIVITY_TO_SR(ACT_GAUSS_SPINUP);
	ADD_ACTIVITY_TO_SR(ACT_GAUSS_SPINCYCLE);

	//===========================
	// CSPort Specific Activities
	//===========================

	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_SILENCED);		// fire
	ADD_ACTIVITY_TO_SR(ACT_VM_RELOAD_SILENCED);
	ADD_ACTIVITY_TO_SR(ACT_VM_DRYFIRE_SILENCED);				// fire with no ammo loaded.
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_SILENCED);
	ADD_ACTIVITY_TO_SR(ACT_VM_DRAW_SILENCED);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_EMPTY_LEFT);
	ADD_ACTIVITY_TO_SR(ACT_VM_DRYFIRE_LEFT);

	// new for CS2
	ADD_ACTIVITY_TO_SR(ACT_VM_IS_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_VM_IS_HOLSTER);
	ADD_ACTIVITY_TO_SR(ACT_VM_IS_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_VM_IS_PRIMARYATTACK);

	ADD_ACTIVITY_TO_SR(ACT_PLAYER_IDLE_FIRE);
	ADD_ACTIVITY_TO_SR(ACT_PLAYER_CROUCH_FIRE);
	ADD_ACTIVITY_TO_SR(ACT_PLAYER_CROUCH_WALK_FIRE);
	ADD_ACTIVITY_TO_SR(ACT_PLAYER_WALK_FIRE);
	ADD_ACTIVITY_TO_SR(ACT_PLAYER_RUN_FIRE);

	ADD_ACTIVITY_TO_SR(ACT_IDLETORUN);
	ADD_ACTIVITY_TO_SR(ACT_RUNTOIDLE);

	ADD_ACTIVITY_TO_SR(ACT_VM_DRAW_DEPLOYED);

	//===========================
	// DoD Specific Activities
	//===========================
	ADD_ACTIVITY_TO_SR(ACT_SPRINT);

	ADD_ACTIVITY_TO_SR(ACT_GET_DOWN_STAND);
	ADD_ACTIVITY_TO_SR(ACT_GET_UP_STAND);
	ADD_ACTIVITY_TO_SR(ACT_GET_DOWN_CROUCH);
	ADD_ACTIVITY_TO_SR(ACT_GET_UP_CROUCH);
	ADD_ACTIVITY_TO_SR(ACT_PRONE_FORWARD);
	ADD_ACTIVITY_TO_SR(ACT_PRONE_IDLE);

	ADD_ACTIVITY_TO_SR(ACT_DEEPIDLE1);
	ADD_ACTIVITY_TO_SR(ACT_DEEPIDLE2);
	ADD_ACTIVITY_TO_SR(ACT_DEEPIDLE3);
	ADD_ACTIVITY_TO_SR(ACT_DEEPIDLE4);

	ADD_ACTIVITY_TO_SR(ACT_VM_RELOAD_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_VM_RELOAD_IDLE);

	//Weapon is empty activities
	ADD_ACTIVITY_TO_SR(ACT_VM_DRAW_EMPTY);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_EMPTY);
	ADD_ACTIVITY_TO_SR(ACT_VM_RELOAD_EMPTY);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_EMPTY);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_DEPLOYED_EMPTY);

	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_8);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_7);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_6);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_5);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_4);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_3);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_2);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_1);

	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_DEPLOYED_8);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_DEPLOYED_7);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_DEPLOYED_6);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_DEPLOYED_5);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_DEPLOYED_4);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_DEPLOYED_3);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_DEPLOYED_2);
	ADD_ACTIVITY_TO_SR(ACT_VM_IDLE_DEPLOYED_1);

	// Animation from prone idle to standing/crouch idle. Number designates bullets left
	ADD_ACTIVITY_TO_SR(ACT_VM_UNDEPLOY);
	ADD_ACTIVITY_TO_SR(ACT_VM_UNDEPLOY_8);
	ADD_ACTIVITY_TO_SR(ACT_VM_UNDEPLOY_7);
	ADD_ACTIVITY_TO_SR(ACT_VM_UNDEPLOY_6);
	ADD_ACTIVITY_TO_SR(ACT_VM_UNDEPLOY_5);
	ADD_ACTIVITY_TO_SR(ACT_VM_UNDEPLOY_4);
	ADD_ACTIVITY_TO_SR(ACT_VM_UNDEPLOY_3);
	ADD_ACTIVITY_TO_SR(ACT_VM_UNDEPLOY_2);
	ADD_ACTIVITY_TO_SR(ACT_VM_UNDEPLOY_1);
	ADD_ACTIVITY_TO_SR(ACT_VM_UNDEPLOY_EMPTY);

	// Animation from standing/crouch idle to prone idle. Number designates bullets left
	ADD_ACTIVITY_TO_SR(ACT_VM_DEPLOY);
	ADD_ACTIVITY_TO_SR(ACT_VM_DEPLOY_8);
	ADD_ACTIVITY_TO_SR(ACT_VM_DEPLOY_7);
	ADD_ACTIVITY_TO_SR(ACT_VM_DEPLOY_6);
	ADD_ACTIVITY_TO_SR(ACT_VM_DEPLOY_5);
	ADD_ACTIVITY_TO_SR(ACT_VM_DEPLOY_4);
	ADD_ACTIVITY_TO_SR(ACT_VM_DEPLOY_3);
	ADD_ACTIVITY_TO_SR(ACT_VM_DEPLOY_2);
	ADD_ACTIVITY_TO_SR(ACT_VM_DEPLOY_1);
	ADD_ACTIVITY_TO_SR(ACT_VM_DEPLOY_EMPTY);

	// Shooting animations for standing/crouch position.  Number designates bullets left at START of animation
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_8);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_7);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_6);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_5);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_4);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_3);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_2);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_1);

	// Shooting animations for prone position. Number designates bullets left at START of animation
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_DEPLOYED_8);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_DEPLOYED_7);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_DEPLOYED_6);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_DEPLOYED_5);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_DEPLOYED_4);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_DEPLOYED_3);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_DEPLOYED_2);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_DEPLOYED_1);
	ADD_ACTIVITY_TO_SR(ACT_VM_PRIMARYATTACK_DEPLOYED_EMPTY);

	// Player anim ACTs
	ADD_ACTIVITY_TO_SR(ACT_DOD_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_IDLE_ZOOMED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_ZOOMED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_ZOOMED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_ZOOMED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_ZOOMED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_FORWARD_ZOOMED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_PRONE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM);

	// Positions
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_PISTOL);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_C96);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_RIFLE);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_BOLT);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_TOMMY);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_MP40);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_MP44);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_GREASE);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_MG);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_30CAL);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_GREN_FRAG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_GREN_FRAG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_GREN_FRAG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_GREN_FRAG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_GREN_FRAG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_GREN_FRAG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_AIM_GREN_FRAG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_AIM_GREN_FRAG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_GREN_STICK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_GREN_STICK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_GREN_STICK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_GREN_STICK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_GREN_STICK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_GREN_STICK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_AIM_GREN_STICK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_AIM_GREN_STICK);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_KNIFE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_KNIFE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_KNIFE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_KNIFE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_KNIFE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_KNIFE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_AIM_KNIFE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_AIM_KNIFE);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_SPADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_SPADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_SPADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_SPADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_SPADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_SPADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_AIM_SPADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_AIM_SPADE);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_BAZOOKA);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_PSCHRECK);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_AIM_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_AIM_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_AIM_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_AIM_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_AIM_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_AIM_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_BAR);

	// Zoomed aims
	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_ZOOM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_ZOOM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_ZOOM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_ZOOM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_ZOOM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_ZOOM_RIFLE);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_ZOOM_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_ZOOM_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_ZOOM_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_ZOOM_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_ZOOM_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_ZOOM_BOLT);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_ZOOM_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_ZOOM_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_ZOOM_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_ZOOM_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_ZOOM_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_ZOOM_BAZOOKA);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_ZOOM_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_ZOOM_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_ZOOM_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_ZOOM_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_ZOOM_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_ZOOM_PSCHRECK);

	// Deployed Aim
	ADD_ACTIVITY_TO_SR(ACT_DOD_DEPLOY_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_DEPLOY_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_DEPLOY_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_DEPLOY_30CAL);

	// Prone Deployed Aim
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_DEPLOY_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_DEPLOY_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_DEPLOY_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_DEPLOY_30CAL);

	// Attacks

	// Rifle
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_PRONE_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_DEPLOYED_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_DEPLOYED_RIFLE);

	// Bolt
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_PRONE_BOLT);

	// Tommy
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_PRONE_TOMMY);

	// MP40
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_PRONE_MP40);

	// MP44
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_MP44);

	// Greasegun
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_GREASE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_GREASE);

	// Pistols (Colt ); Luger)
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_C96);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_C96);

	// Mgs (mg42 ); mg34)
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_DEPLOYED_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_DEPLOYED_MG);

	// 30cal
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_DEPLOYED_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_DEPLOYED_30CAL);

	// Grenades
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_GREN_FRAG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_GREN_FRAG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_GREN_STICK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_GREN_STICK);

	// Knife
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_KNIFE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_KNIFE);

	// Spade
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_SPADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_SPADE);

	// Bazooka
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_BAZOOKA);

	// Pschreck
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_PSCHRECK);

	// Bar
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_PRONE_BAR);

	// Reloads
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_GARAND);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_K43);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_M1CARBINE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_GREASEGUN);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_FG42);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_RIFLEGRENADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_C96);

	// Crouch
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_RIFLEGRENADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_M1CARBINE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_CROUCH_C96);

	// Bazookas
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_ZOOMLOAD_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_ZOOMLOAD_PSCHRECK);

	// Deployed
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_DEPLOYED_FG42);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_DEPLOYED_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_DEPLOYED_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_DEPLOYED_MG34);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_DEPLOYED_BAR);

	// Prone
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_GARAND);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_M1CARBINE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_K43);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_MP40);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_GREASEGUN);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_FG42);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_RIFLEGRENADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_C96);

	// Prone bazooka
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_ZOOMLOAD_PRONE_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_ZOOMLOAD_PRONE_PSCHRECK);

	// Prone deployed
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_DEPLOYED_BAR);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_DEPLOYED_FG42);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_DEPLOYED_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_DEPLOYED_MG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RELOAD_PRONE_DEPLOYED_MG34);

	// Prone zoomed aim
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_ZOOM_FORWARD_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_ZOOM_FORWARD_BOLT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_ZOOM_FORWARD_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONE_ZOOM_FORWARD_PSCHRECK);

	// Crouch attack
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_CROUCH);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_CROUCH_SPADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_CROUCH_KNIFE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_CROUCH_GREN_FRAG);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRIMARYATTACK_CROUCH_GREN_STICK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_CROUCH);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_CROUCH_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SECONDARYATTACK_CROUCH_MP40);

	// Hand Signals
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_CROUCH);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_IDLE_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_IDLE_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_IDLE_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_IDLE_KNIFE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_IDLE_MG42);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_IDLE_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_IDLE_STICKGRENADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_IDLE_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_IDLE_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_IDLE_K98);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_CROUCH_30CAL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_CROUCH_BAZOOKA);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_CROUCH_PSCHRECK);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_CROUCH_KNIFE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_CROUCH_MG42);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_CROUCH_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_CROUCH_STICKGRENADE);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_CROUCH_TOMMY);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_CROUCH_MP44);
	ADD_ACTIVITY_TO_SR(ACT_DOD_HS_CROUCH_K98);

	ADD_ACTIVITY_TO_SR(ACT_DOD_STAND_IDLE_TNT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCH_IDLE_TNT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_CROUCHWALK_IDLE_TNT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_WALK_IDLE_TNT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_RUN_IDLE_TNT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_SPRINT_IDLE_TNT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_PRONEWALK_IDLE_TNT);

	ADD_ACTIVITY_TO_SR(ACT_DOD_PLANT_TNT);
	ADD_ACTIVITY_TO_SR(ACT_DOD_DEFUSE_TNT);

	// HL2MP
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_RUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROUCH);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_WALK_CROUCH);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RANGE_ATTACK);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RELOAD);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_JUMP);

	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_RUN_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROUCH_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_WALK_CROUCH_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RANGE_ATTACK_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RELOAD_PISTOL);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_JUMP_PISTOL);

	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_RUN_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROUCH_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_WALK_CROUCH_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RANGE_ATTACK_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RELOAD_SMG1);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_JUMP_SMG1);

	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_AR2);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_RUN_AR2);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROUCH_AR2);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_WALK_CROUCH_AR2);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RANGE_ATTACK_AR2);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RELOAD_AR2);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_JUMP_AR2);

	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_RUN_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROUCH_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_WALK_CROUCH_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RANGE_ATTACK_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RELOAD_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_JUMP_SHOTGUN);

	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_RPG);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_RUN_RPG);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROUCH_RPG);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_WALK_CROUCH_RPG);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RANGE_ATTACK_RPG);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RELOAD_RPG);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_JUMP_RPG);

	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_RUN_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROUCH_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_WALK_CROUCH_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RANGE_ATTACK_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RELOAD_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_JUMP_GRENADE);

	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_PHYSGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_RUN_PHYSGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROUCH_PHYSGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_WALK_CROUCH_PHYSGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RANGE_ATTACK_PHYSGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RELOAD_PHYSGUN);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_JUMP_PHYSGUN);

	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROSSBOW);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_RUN_CROSSBOW);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROUCH_CROSSBOW);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_WALK_CROUCH_CROSSBOW);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RANGE_ATTACK_CROSSBOW);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RELOAD_CROSSBOW);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_JUMP_CROSSBOW);

	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_RUN_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROUCH_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_WALK_CROUCH_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RANGE_ATTACK_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RELOAD_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_JUMP_MELEE);

	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_SLAM);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_RUN_SLAM);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_IDLE_CROUCH_SLAM);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_WALK_CROUCH_SLAM);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RANGE_ATTACK_SLAM);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_GESTURE_RELOAD_SLAM);
	ADD_ACTIVITY_TO_SR(ACT_HL2MP_JUMP_SLAM);


	// Portal!
	ADD_ACTIVITY_TO_SR(ACT_VM_FIZZLE);

	// Multiplayer
	ADD_ACTIVITY_TO_SR(ACT_MP_STAND_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCH_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCH_DEPLOYED_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCH_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_MP_DEPLOYED_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_RUN);
	ADD_ACTIVITY_TO_SR(ACT_MP_WALK);
	ADD_ACTIVITY_TO_SR(ACT_MP_AIRWALK);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCHWALK);
	ADD_ACTIVITY_TO_SR(ACT_MP_SPRINT);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_START);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_FLOAT);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_LAND);
	ADD_ACTIVITY_TO_SR(ACT_MP_DOUBLEJUMP);
	ADD_ACTIVITY_TO_SR(ACT_MP_SWIM);
	ADD_ACTIVITY_TO_SR(ACT_MP_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_MP_SWIM_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_MP_VCD);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_PRIMARYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_PRIMARYFIRE_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_SECONDARYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_PRIMARYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_PRIMARYFIRE_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_SECONDARYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_PRIMARYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_SECONDARYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_PRIMARYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_SECONDARYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_GRENADE);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_STAND);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_STAND_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_STAND_END);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_CROUCH);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_CROUCH_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_CROUCH_END);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_SWIM);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_SWIM_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_SWIM_END);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_AIRWALK);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_AIRWALK_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_AIRWALK_END);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_PREFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_POSTFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_STARTFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_PREFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_POSTFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_PREFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_POSTFIRE);

	// Multiplayer - Primary
	ADD_ACTIVITY_TO_SR(ACT_MP_STAND_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCH_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_RUN_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_WALK_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_AIRWALK_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCHWALK_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_START_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_FLOAT_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_LAND_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_SWIM_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_DEPLOYED_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_SWIM_DEPLOYED_PRIMARY);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_PRIMARY);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_PRIMARY_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_PRIMARY);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_PRIMARY_DEPLOYED);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_PRIMARY);

	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_STAND_PRIMARY);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_STAND_PRIMARY_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_STAND_PRIMARY_END);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_CROUCH_PRIMARY);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_CROUCH_PRIMARY_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_CROUCH_PRIMARY_END);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_SWIM_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_SWIM_PRIMARY_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_SWIM_PRIMARY_END);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_AIRWALK_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_AIRWALK_PRIMARY_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_AIRWALK_PRIMARY_END);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_GRENADE_PRIMARY);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_GRENADE_PRIMARY);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_GRENADE_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_GRENADE_PRIMARY);

	// Secondary
	ADD_ACTIVITY_TO_SR(ACT_MP_STAND_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCH_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_RUN_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_WALK_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_AIRWALK_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCHWALK_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_START_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_FLOAT_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_LAND_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_SWIM_SECONDARY);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_SECONDARY);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_SECONDARY);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_SECONDARY);

	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_STAND_SECONDARY);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_STAND_SECONDARY_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_STAND_SECONDARY_END);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_CROUCH_SECONDARY);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_CROUCH_SECONDARY_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_CROUCH_SECONDARY_END);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_SWIM_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_SWIM_SECONDARY_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_SWIM_SECONDARY_END);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_AIRWALK_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_AIRWALK_SECONDARY_LOOP);
	ADD_ACTIVITY_TO_SR(ACT_MP_RELOAD_AIRWALK_SECONDARY_END);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_GRENADE_SECONDARY);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_GRENADE_SECONDARY);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_GRENADE_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_GRENADE_SECONDARY);

	// Melee
	ADD_ACTIVITY_TO_SR(ACT_MP_STAND_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCH_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_RUN_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_WALK_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_AIRWALK_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCHWALK_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_START_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_FLOAT_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_LAND_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_SWIM_MELEE);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_MELEE);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_MELEE_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_MELEE);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_MELEE_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_MELEE);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_GRENADE_MELEE);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_GRENADE_MELEE);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_GRENADE_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_GRENADE_MELEE);

	// Item1
	ADD_ACTIVITY_TO_SR(ACT_MP_STAND_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCH_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_RUN_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_WALK_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_AIRWALK_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCHWALK_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_START_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_FLOAT_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_LAND_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_SWIM_ITEM1);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_ITEM1);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_ITEM1_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_ITEM1);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_ITEM1_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_ITEM1);

	// Item2
	ADD_ACTIVITY_TO_SR(ACT_MP_STAND_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCH_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_RUN_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_WALK_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_AIRWALK_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCHWALK_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_START_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_FLOAT_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_LAND_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_SWIM_ITEM2);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_ITEM2);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_ITEM2_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_ITEM2);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_ITEM2_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_ITEM2);

	// Flinches
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_ITEM2);

	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_HEAD);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_CHEST);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_STOMACH);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_LEFTARM);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_RIGHTARM);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_LEFTLEG);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_FLINCH_RIGHTLEG);

	// Team Fortress specific - medic heal ); medic infect ); etc.....
	ADD_ACTIVITY_TO_SR(ACT_MP_GRENADE1_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_GRENADE1_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_GRENADE1_ATTACK);
	ADD_ACTIVITY_TO_SR(ACT_MP_GRENADE2_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_GRENADE2_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_GRENADE2_ATTACK);

	ADD_ACTIVITY_TO_SR(ACT_MP_PRIMARY_GRENADE1_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_PRIMARY_GRENADE1_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_PRIMARY_GRENADE1_ATTACK);
	ADD_ACTIVITY_TO_SR(ACT_MP_PRIMARY_GRENADE2_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_PRIMARY_GRENADE2_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_PRIMARY_GRENADE2_ATTACK);

	ADD_ACTIVITY_TO_SR(ACT_MP_SECONDARY_GRENADE1_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_SECONDARY_GRENADE1_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_SECONDARY_GRENADE1_ATTACK);
	ADD_ACTIVITY_TO_SR(ACT_MP_SECONDARY_GRENADE2_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_SECONDARY_GRENADE2_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_SECONDARY_GRENADE2_ATTACK);

	ADD_ACTIVITY_TO_SR(ACT_MP_MELEE_GRENADE1_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_MELEE_GRENADE1_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_MELEE_GRENADE1_ATTACK);
	ADD_ACTIVITY_TO_SR(ACT_MP_MELEE_GRENADE2_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_MELEE_GRENADE2_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_MELEE_GRENADE2_ATTACK);

	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM1_GRENADE1_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM1_GRENADE1_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM1_GRENADE1_ATTACK);
	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM1_GRENADE2_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM1_GRENADE2_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM1_GRENADE2_ATTACK);

	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM2_GRENADE1_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM2_GRENADE1_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM2_GRENADE1_ATTACK);
	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM2_GRENADE2_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM2_GRENADE2_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MP_ITEM2_GRENADE2_ATTACK);

	// Building
	ADD_ACTIVITY_TO_SR(ACT_MP_STAND_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCH_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_RUN_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_WALK_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_AIRWALK_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCHWALK_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_START_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_FLOAT_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_LAND_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_SWIM_BUILDING);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_BUILDING);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_BUILDING);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_BUILDING);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_GRENADE_BUILDING);		// RUN ); WALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_CROUCH_GRENADE_BUILDING);		// CROUCHWALK
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_GRENADE_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_AIRWALK_GRENADE_BUILDING);

	ADD_ACTIVITY_TO_SR(ACT_MP_STAND_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCH_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_RUN_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_WALK_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_AIRWALK_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_CROUCHWALK_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_START_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_FLOAT_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_JUMP_LAND_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_SWIM_PDA);

	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_STAND_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_ATTACK_SWIM_PDA);

	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_HANDMOUTH);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FINGERPOINT);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FISTPUMP);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_THUMBSUP);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODYES);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODNO);

	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_HANDMOUTH_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FINGERPOINT_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FISTPUMP_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_THUMBSUP_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODYES_PRIMARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODNO_PRIMARY);

	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_HANDMOUTH_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FINGERPOINT_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FISTPUMP_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_THUMBSUP_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODYES_SECONDARY);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODNO_SECONDARY);

	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_HANDMOUTH_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FINGERPOINT_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FISTPUMP_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_THUMBSUP_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODYES_MELEE);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODNO_MELEE);

	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_HANDMOUTH_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FINGERPOINT_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FISTPUMP_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_THUMBSUP_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODYES_ITEM1);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODNO_ITEM1);

	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_HANDMOUTH_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FINGERPOINT_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FISTPUMP_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_THUMBSUP_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODYES_ITEM2);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODNO_ITEM2);

	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_HANDMOUTH_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FINGERPOINT_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FISTPUMP_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_THUMBSUP_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODYES_BUILDING);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODNO_BUILDING);

	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_HANDMOUTH_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FINGERPOINT_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_FISTPUMP_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_THUMBSUP_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODYES_PDA);
	ADD_ACTIVITY_TO_SR(ACT_MP_GESTURE_VC_NODNO_PDA);


	ADD_ACTIVITY_TO_SR(ACT_VM_UNUSABLE);
	ADD_ACTIVITY_TO_SR(ACT_VM_UNUSABLE_TO_USABLE);
	ADD_ACTIVITY_TO_SR(ACT_VM_USABLE_TO_UNUSABLE);

	// Specific viewmodel activities for weapon roles
	ADD_ACTIVITY_TO_SR(ACT_PRIMARY_VM_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_PRIMARY_VM_HOLSTER);
	ADD_ACTIVITY_TO_SR(ACT_PRIMARY_VM_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_PRIMARY_VM_PULLBACK);
	ADD_ACTIVITY_TO_SR(ACT_PRIMARY_VM_PRIMARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_PRIMARY_VM_SECONDARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_PRIMARY_VM_RELOAD);
	ADD_ACTIVITY_TO_SR(ACT_PRIMARY_VM_DRYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_PRIMARY_VM_IDLE_TO_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_PRIMARY_VM_IDLE_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_PRIMARY_VM_LOWERED_TO_IDLE);

	ADD_ACTIVITY_TO_SR(ACT_SECONDARY_VM_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_SECONDARY_VM_HOLSTER);
	ADD_ACTIVITY_TO_SR(ACT_SECONDARY_VM_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_SECONDARY_VM_PULLBACK);
	ADD_ACTIVITY_TO_SR(ACT_SECONDARY_VM_PRIMARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_SECONDARY_VM_SECONDARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_SECONDARY_VM_RELOAD);
	ADD_ACTIVITY_TO_SR(ACT_SECONDARY_VM_DRYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_SECONDARY_VM_IDLE_TO_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_SECONDARY_VM_IDLE_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_SECONDARY_VM_LOWERED_TO_IDLE);

	ADD_ACTIVITY_TO_SR(ACT_MELEE_VM_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_VM_HOLSTER);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_VM_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_VM_PULLBACK);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_VM_PRIMARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_VM_SECONDARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_VM_RELOAD);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_VM_DRYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_VM_IDLE_TO_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_VM_IDLE_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_MELEE_VM_LOWERED_TO_IDLE);

	ADD_ACTIVITY_TO_SR(ACT_PDA_VM_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_PDA_VM_HOLSTER);
	ADD_ACTIVITY_TO_SR(ACT_PDA_VM_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_PDA_VM_PULLBACK);
	ADD_ACTIVITY_TO_SR(ACT_PDA_VM_PRIMARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_PDA_VM_SECONDARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_PDA_VM_RELOAD);
	ADD_ACTIVITY_TO_SR(ACT_PDA_VM_DRYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_PDA_VM_IDLE_TO_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_PDA_VM_IDLE_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_PDA_VM_LOWERED_TO_IDLE);

	ADD_ACTIVITY_TO_SR(ACT_ITEM1_VM_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_ITEM1_VM_HOLSTER);
	ADD_ACTIVITY_TO_SR(ACT_ITEM1_VM_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_ITEM1_VM_PULLBACK);
	ADD_ACTIVITY_TO_SR(ACT_ITEM1_VM_PRIMARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_ITEM1_VM_SECONDARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_ITEM1_VM_RELOAD);
	ADD_ACTIVITY_TO_SR(ACT_ITEM1_VM_DRYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_ITEM1_VM_IDLE_TO_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_ITEM1_VM_IDLE_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_ITEM1_VM_LOWERED_TO_IDLE);

	ADD_ACTIVITY_TO_SR(ACT_ITEM2_VM_DRAW);
	ADD_ACTIVITY_TO_SR(ACT_ITEM2_VM_HOLSTER);
	ADD_ACTIVITY_TO_SR(ACT_ITEM2_VM_IDLE);
	ADD_ACTIVITY_TO_SR(ACT_ITEM2_VM_PULLBACK);
	ADD_ACTIVITY_TO_SR(ACT_ITEM2_VM_PRIMARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_ITEM2_VM_SECONDARYATTACK);
	ADD_ACTIVITY_TO_SR(ACT_ITEM2_VM_RELOAD);
	ADD_ACTIVITY_TO_SR(ACT_ITEM2_VM_DRYFIRE);
	ADD_ACTIVITY_TO_SR(ACT_ITEM2_VM_IDLE_TO_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_ITEM2_VM_IDLE_LOWERED);
	ADD_ACTIVITY_TO_SR(ACT_ITEM2_VM_LOWERED_TO_IDLE);

	// Infested activities
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_SUCCEED);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_FAIL);
	// Autogun
	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM_AUTOGUN);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM_AUTOGUN);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_AUTOGUN);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_AIM_AUTOGUN);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_AUTOGUN);
	ADD_ACTIVITY_TO_SR(ACT_CROUCH_IDLE_AUTOGUN);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_AUTOGUN);
	ADD_ACTIVITY_TO_SR(ACT_JUMP_AUTOGUN);
	// Pistol
	ADD_ACTIVITY_TO_SR(ACT_IDLE_AIM_PISTOL);
	// PDW
	ADD_ACTIVITY_TO_SR(ACT_WALK_AIM_DUAL);
	ADD_ACTIVITY_TO_SR(ACT_RUN_AIM_DUAL);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_DUAL);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_AIM_DUAL);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_DUAL);
	ADD_ACTIVITY_TO_SR(ACT_CROUCH_IDLE_DUAL);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_DUAL);
	ADD_ACTIVITY_TO_SR(ACT_JUMP_DUAL);
	// Shotgun
	ADD_ACTIVITY_TO_SR(ACT_IDLE_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_IDLE_AIM_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_CROUCH_IDLE_SHOTGUN);
	ADD_ACTIVITY_TO_SR(ACT_JUMP_SHOTGUN);
	// Rifle
	ADD_ACTIVITY_TO_SR(ACT_IDLE_AIM_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_RELOAD_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_CROUCH_IDLE_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_RANGE_ATTACK_RIFLE);
	ADD_ACTIVITY_TO_SR(ACT_JUMP_RIFLE);
	// Infested General AI
	ADD_ACTIVITY_TO_SR(ACT_SLEEP);
	ADD_ACTIVITY_TO_SR(ACT_WAKE);

	// Shield Bug
	ADD_ACTIVITY_TO_SR(ACT_FLICK_LEFT);
	ADD_ACTIVITY_TO_SR(ACT_FLICK_LEFT_MIDDLE);
	ADD_ACTIVITY_TO_SR(ACT_FLICK_RIGHT_MIDDLE);
	ADD_ACTIVITY_TO_SR(ACT_FLICK_RIGHT);
	ADD_ACTIVITY_TO_SR(ACT_SPINAROUND);

	// Mortar Bug
	ADD_ACTIVITY_TO_SR(ACT_PREP_TO_FIRE);
	ADD_ACTIVITY_TO_SR(ACT_FIRE);
	ADD_ACTIVITY_TO_SR(ACT_FIRE_RECOVER);

	// Shaman
	ADD_ACTIVITY_TO_SR(ACT_SPRAY);

	// Boomer
	ADD_ACTIVITY_TO_SR(ACT_PREP_EXPLODE);
	ADD_ACTIVITY_TO_SR(ACT_EXPLODE);

	AssertMsg( m_iNumActivities == LAST_SHARED_ACTIVITY + 1, "Not all activities from ai_activity.h registered in ai_activity.cpp" ); 
}
