---
UID: NN:iads.IADsContainer
title: IADsContainer (iads.h)
description: The IADsContainer interface enables an ADSI container object to create, delete, and manage contained ADSI objects. Container objects represent hierarchical directory trees, such as in a file system, and to organize the directory hierarchy.
old-location: adsi\iadscontainer.htm
tech.root: adsi
ms.assetid: 6c1d6c7c-e003-47f9-adfa-4a753fb3e9b2
ms.date: 12/05/2018
ms.keywords: IADsContainer, IADsContainer interface [ADSI], IADsContainer interface [ADSI],described, _ds_iadscontainer, adsi.iadscontainer, iads/IADsContainer
f1_keywords:
- iads/IADsContainer
dev_langs:
- c++
req.header: iads.h
req.include-header: 
req.target-type: Windows
req.target-min-winverclnt: Windows Vista
req.target-min-winversvr: Windows Server 2008
req.kmdf-ver: 
req.umdf-ver: 
req.ddi-compliance: 
req.unicode-ansi: 
req.idl: 
req.max-support: 
req.namespace: 
req.assembly: 
req.type-library: 
req.lib: 
req.dll: Activeds.dll
req.irql: 
topic_type:
- APIRef
- kbSyntax
api_type:
- COM
api_location:
- Activeds.dll
api_name:
- IADsContainer
targetos: Windows
req.typenames: 
req.redist: 
ms.custom: 19H1
---

# IADsContainer interface


## -description


The <b>IADsContainer</b> interface enables an ADSI container 
    object to create, delete, and manage contained ADSI objects. Container objects represent hierarchical directory 
    trees, such as in a file system, and to organize the directory hierarchy.

You can use the <b>IADsContainer</b> interface to either 
    enumerate contained objects or manage their lifecycle. An example would be to recursively navigate a directory 
    tree. By querying the <b>IADsContainer</b> interface on an ADSI 
    object, you can determine if the object has any children. If the interface is not supported, the object is a leaf. 
    Otherwise, it is a container. You can continue this process for the newly found container objects. To create, 
    copy, or delete an object, send the request to the container object to perform the task.


## -inheritance

The <b xmlns:loc="http://microsoft.com/wdcml/l10n">IADsContainer</b> interface inherits from the <a href="https://docs.microsoft.com/previous-versions/windows/desktop/api/oaidl/nn-oaidl-idispatch">IDispatch</a> interface. <b>IADsContainer</b> also has these types of members:
<ul>
<li><a href="https://docs.microsoft.com/">Methods</a></li>
<li><a href="https://docs.microsoft.com/">Properties</a></li>
</ul>

## -members

The <b>IADsContainer</b> interface has these methods.
<table class="members" id="memberListMethods">
<tr>
<th align="left" width="37%">Method</th>
<th align="left" width="63%">Description</th>
</tr>
<tr data="declared;">
<td align="left" width="37%">
<a href="https://docs.microsoft.com/windows/desktop/api/iads/nf-iads-iadscontainer-copyhere">CopyHere</a>
</td>
<td align="left" width="63%">
Copies an object to the container.

</td>
</tr>
<tr data="declared;">
<td align="left" width="37%">
<a href="https://docs.microsoft.com/windows/desktop/api/iads/nf-iads-iadscontainer-create">Create</a>
</td>
<td align="left" width="63%">
Creates an object in the container.

</td>
</tr>
<tr data="declared;">
<td align="left" width="37%">
<a href="https://docs.microsoft.com/windows/desktop/api/iads/nf-iads-iadscontainer-delete">Delete</a>
</td>
<td align="left" width="63%">
Deletes an object in the container.

</td>
</tr>
<tr data="declared;">
<td align="left" width="37%">
<a href="https://docs.microsoft.com/windows/desktop/api/iads/nf-iads-iadscontainer-get__newenum">get__NewEnum</a>
</td>
<td align="left" width="63%">
Retrieves an enumerator object for the container.

</td>
</tr>
<tr data="declared;">
<td align="left" width="37%">
<a href="https://docs.microsoft.com/windows/desktop/api/iads/nf-iads-iadscontainer-getobject">GetObject</a>
</td>
<td align="left" width="63%">
Retrieves an interface for a directory object in the container.

</td>
</tr>
<tr data="declared;">
<td align="left" width="37%">
<a href="https://docs.microsoft.com/windows/desktop/api/iads/nf-iads-iadscontainer-movehere">MoveHere</a>
</td>
<td align="left" width="63%">
Moves an object to the container.

</td>
</tr>
</table> 
<h3><a id="properties"></a>Properties</h3>The <b xmlns:loc="http://microsoft.com/wdcml/l10n">IADsContainer</b> interface has these properties.
<table class="members" id="memberListProperties">
<tr>
<th align="left" width="27%">Property</th>
<th align="left" width="10%">Access type</th>
<th align="left" width="63%">Description</th>
</tr>
<tr data="declared;">
<td align="left" width="27%" xml:space="preserve">

<a href="https://docs.microsoft.com/windows/desktop/ADSI/iadscontainer-property-methods">Count</a>


</td>
<td align="left" width="10%">
Read-only

</td>
<td align="left" width="63%">
Contains the number of directory objects in the container.

</td>
</tr>
<tr data="declared;">
<td align="left" width="27%" xml:space="preserve">

<a href="https://docs.microsoft.com/windows/desktop/ADSI/iadscontainer-property-methods">Filter</a>


</td>
<td align="left" width="10%">
Read/write

</td>
<td align="left" width="63%">
Contains the filter on the schema classes to use for an enumeration.

</td>
</tr>
<tr data="declared;">
<td align="left" width="27%" xml:space="preserve">

<a href="https://docs.microsoft.com/windows/desktop/ADSI/iadscontainer-property-methods">Hints</a>


</td>
<td align="left" width="10%">
Read/write

</td>
<td align="left" width="63%">
Contains the properties to retrieve for each object that is enumerated by the container.

</td>
</tr>
</table> 


## -remarks



To determine if an object is a container, use the <a href="https://docs.microsoft.com/windows/desktop/ADSI/iadsclass-property-methods">IADsClass.Container</a> property of the object.

When you bind to a container object using its GUID (or SID), you can only perform specific operations on the container object. These operations include examination of the object attributes and enumeration of the object's immediate children. These operations are shown in the following code example.


```vb
Dim con As IADsContainer
Dim obj As IADs
Set con = GetObject("LDAP://svr01/<GUID=xxxx>")
con.Filter = Array("user")
For Each item In con
    debug.print item.Name " &  " of " & item.Class
Next
```


All other operations, that is, <b>GetObject</b>, <b>Create</b>, <b>Delete</b>, <b>CopyHere</b>, and <b>MoveHere</b> are not supported in the container's GUID representation. For example, the last line of the following code example will result in an error.


```vb
Dim con As IADsContainer
Dim obj As IADs
Set con = GetObject("LDAP://svr01/<GUID=xxxx>")
Set obj = con.GetObject("user", "CN=Jeff Smith")
```


Binding, using GUID (or SID), is intended for low overhead and, thus, fast binds, which are often used for object introspection.

To call these methods of the container bound with its GUID (or SID), rebind to the object using its distinguished name.


```vb
Dim conGUID, conDN As IADsContainer
Dim obj As IADs
Set conGUID = GetObject("LDAP://svr/<GUID=xxxx>")
Set conDN=GetObject("LDAP://svr/" & conGUID.Get("distinguishedName"))
Set obj = conDN.GetObject("user", "CN=Jeff Smith")
```


For more information about object GUID representation, see <a href="https://docs.microsoft.com/windows/desktop/ADSI/iads-property-methods">IADs.GUID</a>.


#### Examples

The following code example determines if an ADSI object is a container.


```vb
Dim obj As IADs
Dim cls As IADsClass
On Error GoTo Cleanup

Set obj = GetObject("WinNT://myComputer,computer")
Set cls = GetObject(obj.Schema)
If (cls.Container = TRUE) Then
    MsgBox "The object is a container."
Else
    MsgBox "The object is a leaf."
End If

Cleanup:
    If (Err.Number<>0) Then
        MsgBox("An error has occurred. " & Err.Number)
    End If
    Set obj = Nothing
    Set cls = Nothing
```


The following code example determines if an ADSI object is a container.


```cpp
IADs *pADs = NULL;
IADsClass *pCls = NULL;
HRESULT hr = S_OK;
BSTR bstr;

hr = ADsGetObject(L"WinNT://myComputer,computer", IID_IADs, (void**)&pADs);
if(FAILED(hr)){return;}

pADs->get_Schema(&bstr);
hr = ADsGetObject(bstr, IID_IADsClass, (void**)&pCls);
pADs->Release();
SysFreeString(bstr);

if(FAILED(hr)){return;}

VARIANT_BOOL isContainer;
pCls->get_Container(&isContainer);

if(isContainer) 
    printf("Object is a container.\n");
else
    printf("Object is not a container.\n");

pCls->Release();

```





## -see-also




<a href="https://docs.microsoft.com/windows/desktop/ADSI/creating-and-deleting-objects">Creating and Deleting Objects</a>



<a href="https://docs.microsoft.com/windows/desktop/ADSI/iads-property-methods">IADs::get_GUID</a>



<a href="https://docs.microsoft.com/windows/desktop/ADSI/iadsclass-property-methods">IADsClass::get_Container</a>



<a href="https://docs.microsoft.com/windows/desktop/api/iads/nn-iads-iadsnamespaces">IADsNamespaces</a>



<a href="https://docs.microsoft.com/previous-versions/windows/desktop/api/oaidl/nn-oaidl-idispatch">IDispatch</a>
 

 

