---
UID: NC:winbio_adapter.PIBIO_ENGINE_COMMIT_ENROLLMENT_FN
title: PIBIO_ENGINE_COMMIT_ENROLLMENT_FN (winbio_adapter.h)
description: Finalizes the enrollment object, converts it to a template, and saves the template in the database.
old-location: secbiomet\engineadaptercommitenrollment.htm
tech.root: SecBioMet
ms.assetid: f8eb3dd9-b993-4b45-b7f4-e1925c233a80
ms.date: 12/05/2018
ms.keywords: EngineAdapterCommitEnrollment, EngineAdapterCommitEnrollment callback function [Windows Biometric Framework API], PIBIO_ENGINE_COMMIT_ENROLLMENT_FN, PIBIO_ENGINE_COMMIT_ENROLLMENT_FN callback, secbiomet.engineadaptercommitenrollment, winbio_adapter/EngineAdapterCommitEnrollment
f1_keywords:
- winbio_adapter/EngineAdapterCommitEnrollment
dev_langs:
- c++
req.header: winbio_adapter.h
req.include-header: Winbio_adapter.h
req.target-type: Windows
req.target-min-winverclnt: Windows 7 [desktop apps only]
req.target-min-winversvr: Windows Server 2008 R2 [desktop apps only]
req.kmdf-ver: 
req.umdf-ver: 
req.ddi-compliance: 
req.unicode-ansi: 
req.idl: 
req.max-support: 
req.namespace: 
req.assembly: 
req.type-library: 
req.lib: 
req.dll: 
req.irql: 
topic_type:
- APIRef
- kbSyntax
api_type:
- UserDefined
api_location:
- Winbio_adapter.h
api_name:
- EngineAdapterCommitEnrollment
targetos: Windows
req.typenames: 
req.redist: 
ms.custom: 19H1
---

# PIBIO_ENGINE_COMMIT_ENROLLMENT_FN callback function


## -description


Called by the Windows Biometric Framework to finalize the enrollment object, convert it to a template, and save the template in the database.


## -parameters




### -param Pipeline [in, out]

Pointer to a <a href="https://docs.microsoft.com/windows/desktop/api/winbio_adapter/ns-winbio_adapter-winbio_pipeline">WINBIO_PIPELINE</a> structure associated with the biometric unit performing the operation.


### -param Identity [in]

Pointer to a  <a href="https://docs.microsoft.com/windows/desktop/SecBioMet/winbio-identity">WINBIO_IDENTITY</a> structure that contains the GUID or SID of the template to be stored in the database.


### -param SubFactor [in]

A <b>WINBIO_BIOMETRIC_SUBTYPE</b> value that specifies the sub-factor associated with the template to be stored in the database.


### -param PayloadBlob [in, optional]

An optional pointer to an array of bytes that contain a verification signature generated by the Windows Biometric Framework.



### -param PayloadBlobSize [in]

The size, in bytes, of the character array pointed to by the <i>PayloadBlob</i> parameter. This value must be zero if the <i>PayloadBlob</i> parameter is <b>NULL</b>.


## -returns



If the function succeeds, it returns S_OK. If the function fails, it must return one of the following <b>HRESULT</b> values or any value returned by the storage adapter.

<table>
<tr>
<th>Return code</th>
<th>Description</th>
</tr>
<tr>
<td width="40%">
<dl>
<dt><b>E_POINTER</b></dt>
</dl>
</td>
<td width="60%">
A mandatory pointer argument is <b>NULL</b>.

</td>
</tr>
<tr>
<td width="40%">
<dl>
<dt><b><b> E_INVALIDARG</b></b></dt>
</dl>
</td>
<td width="60%">
The value specified by the <i>Identity</i> parameter or the <i>SubFactor</i> parameter is not valid.

</td>
</tr>
<tr>
<td width="40%">
<dl>
<dt><b><b>WINBIO_E_DUPLICATE_ENROLLMENT</b></b></dt>
</dl>
</td>
<td width="60%">
The template specified by the <i>Identity</i> and <i>SubFactor</i> parameters is already saved in the database.

</td>
</tr>
<tr>
<td width="40%">
<dl>
<dt><b>WINBIO_E_INVALID_DEVICE_STATE</b></dt>
</dl>
</td>
<td width="60%">
There is no template attached to the pipeline.

</td>
</tr>
</table>
 




## -remarks



If this function succeeds, it should flush the enrollment template from the pipeline. The result of this action should be equivalent to calling <a href="https://docs.microsoft.com/windows/desktop/api/winbio_adapter/nc-winbio_adapter-pibio_engine_clear_context_fn">EngineAdapterClearContext</a>.

If this function fails, it should not change the state of the engine context. In particular, if there is a completed template attached to the pipeline, it should be possible to call this function again (after reasons for any failure have been addressed) to commit the template to the database.

Engine adapters that support pre-boot authentication must commit the enrollment not only to the storage adapter attached to the pipeline, but also to the pre-boot storage area. The details of how this is to be accomplished are left to the vendor.

<div class="alert"><b>Important</b>  <p class="note">Do not attempt to validate the value supplied for the <i>SubFactor</i> parameter. The Windows Biometrics Service will validate the supplied value before passing it through to your implementation. If the value is <b>WINBIO_SUBTYPE_NO_INFORMATION</b> or <b>WINBIO_SUBTYPE_ANY</b>, then validate where appropriate.

</div>
<div> </div>

#### Examples

The following pseudocode shows one possible implementation of this function. The example does not compile. You must adapt it to suit your purpose.


```cpp
//////////////////////////////////////////////////////////////////////////////////////////
//
// EngineAdapterCommitEnrollment
//
// Purpose:
//      Finalizes the enrollment object, converts it to a template, and saves 
//      the template in the database.
//
// Parameters:
//      Pipeline        - Pointer to a WINBIO_PIPELINE structure associated 
//                        with the biometric unit performing the operation
//      Identity        - GUID or SID of the template to be stored in the 
//                        database
//      SubFactor       - Sub-factor associated with the template to be stored
//                        in the database
//      PayloadBlob     - Optional pointer to an array of bytes that contain a 
//                        verification signature generated by the Windows Biometric
//                        Framework
//      PayloadBlobSize - Size, in bytes, of the character array pointed to by 
//                        the PayloadBlob parameter
//

static HRESULT
WINAPI
EngineAdapterCommitEnrollment(
    __inout PWINBIO_PIPELINE Pipeline,
    __in PWINBIO_IDENTITY Identity,
    __in WINBIO_BIOMETRIC_SUBTYPE SubFactor,
    __in PUCHAR PayloadBlob,
    __in SIZE_T PayloadBlobSize
    )
{
    HRESULT hr = S_OK;
    DWORD indexVector[NUMBER_OF_TEMPLATE_BINS] = {0};
    WINBIO_REJECT_DETAIL rejectDetail = 0;
    WINBIO_STORAGE_RECORD newTemplate = {0};

    // Verify that pointer arguments are not NULL.
    if (!ARGUMENT_PRESENT(Pipeline) ||
        !ARGUMENT_PRESENT(Identity))
    {
        hr = E_POINTER;
        goto cleanup;
    }

    if (ARGUMENT_PRESENT(PayloadBlob) && PayloadBlobSize == 0)
    {
        hr = E_INVALIDARG;
        goto cleanup;
    }
    
    if (!ARGUMENT_PRESENT(PayloadBlob) && PayloadBlobSize > 0)
    {
        hr = E_INVALIDARG;
        goto cleanup;
    }

    // TODO: Verify that the SubFactor and Identity arguments are valid.

    // Retrieve the context from the pipeline.
    PWINIBIO_ENGINE_CONTEXT context = 
        (PWINIBIO_ENGINE_CONTEXT)Pipeline->EngineContext;

    // Return if an enrollment is not in progress. This example assumes that 
    // an enrollment object is part of your engine context structure.
    if (context->Enrollment.InProgress != TRUE)
    {
        hr = WINBIO_E_INVALID_DEVICE_STATE;
        goto cleanup;
    }

    // If your adapter supports index vectors to place templates into buckets,
    // call a custom function (_AdapterCreateIndexVector) to create an index 
    // vector from the template data in the enrollment object.
    hr = _AdapterCreateIndexVector(
                context, 
                context->Enrollment.Template, 
                context->Enrollment.TemplateSize,
                indexVector, 
                NUMBER_OF_TEMPLATE_BINS, 
                &rejectDetail
                );
    if (FAILED(hr))
    {
        goto cleanup;
    }

    newTemplate.Identity = Identity;
    newTemplate.SubFactor = SubFactor;
    newTemplate.IndexVector = indexVector;
    newTemplate.IndexElementCount = NUMBER_OF_TEMPLATE_BINS;
    newTemplate.TemplateBlob = context->Enrollment.Template;
    newTemplate.TemplateBlobSize = context->Enrollment.TemplateSize;
    newTemplate.PayloadBlob = PayloadBlob;
    newTemplate.PayloadBlobSize = PayloadBlobSize;

    hr = WbioStorageAddRecord(
                Pipeline,
                &newTemplate
                );

    if (FAILED(hr))
    {
        goto cleanup;
    }

    // Call a custom function (_AdapterDestroyEnrollmentTemplate) to release
    // any resources held by the enrollment object.
    _AdapterDestroyEnrollmentTemplate(
        context,
        &context->Enrollment
        );

    // Specify that the enrollment process has been completed.
    context->Enrollment.InProgress = FALSE;

cleanup:

    return hr;
}

```





## -see-also




<a href="https://docs.microsoft.com/windows/desktop/api/winbio_adapter/nc-winbio_adapter-pibio_engine_clear_context_fn">EngineAdapterClearContext</a>



<a href="https://docs.microsoft.com/windows/desktop/SecBioMet/plug-in-functions">Plug-in Functions</a>
 

 

