---
UID: NC:winbio_adapter.PIBIO_STORAGE_CLOSE_DATABASE_FN
title: PIBIO_STORAGE_CLOSE_DATABASE_FN (winbio_adapter.h)
description: Closes the database associated with the pipeline and frees all related resources.
old-location: secbiomet\storageadapterclosedatabase.htm
tech.root: SecBioMet
ms.assetid: ddb8d0b8-e975-4ee2-bb8c-423b1304c467
ms.date: 12/05/2018
ms.keywords: PIBIO_STORAGE_CLOSE_DATABASE_FN, PIBIO_STORAGE_CLOSE_DATABASE_FN callback, StorageAdapterCloseDatabase, StorageAdapterCloseDatabase callback function [Windows Biometric Framework API], secbiomet.storageadapterclosedatabase, winbio_adapter/StorageAdapterCloseDatabase
f1_keywords:
- winbio_adapter/StorageAdapterCloseDatabase
dev_langs:
- c++
req.header: winbio_adapter.h
req.include-header: Winbio_adapter.h
req.target-type: Windows
req.target-min-winverclnt: Windows 7 [desktop apps only]
req.target-min-winversvr: Windows Server 2008 R2 [desktop apps only]
req.kmdf-ver: 
req.umdf-ver: 
req.ddi-compliance: 
req.unicode-ansi: 
req.idl: 
req.max-support: 
req.namespace: 
req.assembly: 
req.type-library: 
req.lib: 
req.dll: 
req.irql: 
topic_type:
- APIRef
- kbSyntax
api_type:
- UserDefined
api_location:
- Winbio_adapter.h
api_name:
- StorageAdapterCloseDatabase
targetos: Windows
req.typenames: 
req.redist: 
ms.custom: 19H1
---

# PIBIO_STORAGE_CLOSE_DATABASE_FN callback function


## -description


Called by the Windows Biometric Framework to close the database associated with the pipeline and free all related resources.


## -parameters




### -param Pipeline [in, out]

Pointer to a <a href="https://docs.microsoft.com/windows/desktop/api/winbio_adapter/ns-winbio_adapter-winbio_pipeline">WINBIO_PIPELINE</a> structure associated with the biometric unit performing the operation.



## -returns



If the function succeeds, it returns S_OK. If the function fails, it must return one of the following <b>HRESULT</b> values to indicate the error.

<table>
<tr>
<th>Return code</th>
<th>Description</th>
</tr>
<tr>
<td width="40%">
<dl>
<dt><b>E_POINTER</b></dt>
</dl>
</td>
<td width="60%">
The <i>Pipeline</i> argument cannot be <b>NULL</b>.

</td>
</tr>
<tr>
<td width="40%">
<dl>
<dt><b><b>WINBIO_E_DATABASE_CANT_CLOSE</b></b></dt>
</dl>
</td>
<td width="60%">
An unspecified problem has caused the request to fail.

</td>
</tr>
</table>
 




## -remarks



The Windows Biometric Framework does not mandate a particular caching policy, but if the database maintains an in-memory cache of records, this function should flush unwritten records to storage.


This function must invalidate any result sets generated by previous database query operations.



#### Examples

The following pseudocode shows one possible implementation of this function. The example does not compile. You must adapt it to suit your purpose.


```cpp
/////////////////////////////////////////////////////////////////////////////////////////
//
// StorageAdapterCloseDatabase
//
// Purpose:
//      Close the database associated with the pipeline and free all 
//      related resources.
//
// Parameters:
//      Pipeline -  Pointer to a WINBIO_PIPELINE structure associated with 
//                  the biometric unit performing the operation.
//
static HRESULT
WINAPI
StorageAdapterCloseDatabase(
    __inout PWINBIO_PIPELINE Pipeline
    )
{
    HRESULT hr = S_OK;

    // Verify that the Pipeline parameter is not NULL.
    if (!ARGUMENT_PRESENT(Pipeline))
    {
        hr = E_POINTER;
        goto cleanup;
    }

    // Retrieve the context from the pipeline.
    PWINBIO_STORAGE_CONTEXT storageContext = 
           (PWINBIO_STORAGE_CONTEXT)Pipeline->StorageContext;

    // Verify the pipeline state.
    if (storageContext == NULL ||
        Pipeline->StorageHandle == INVALID_HANDLE_VALUE)
    {
        hr = WINBIO_E_INVALID_DEVICE_STATE;
        goto cleanup;
    }

    // Remove any data structures attached to the context and remove the
    // context from the pipeline.
    _CleanupCryptoContext(&storageContext->CryptoContext);
    StorageAdapterClearContext(Pipeline);

    // Close the database file handle.
    CloseHandle( Pipeline->StorageHandle );
    Pipeline->StorageHandle = INVALID_HANDLE_VALUE;

    // Call a custom function (_PurgeDeletedRecords) to remove deleted records
    // from the database file.
    _PurgeDeletedRecords( 
        &storageContext->DatabaseId, 
        (LPCWSTR)&storageContext->FilePath
        );

    // Overwrite the database ID and path.
    SecureZeroMemory(&storageContext->DatabaseId, sizeof(WINBIO_UUID));
    SecureZeroMemory(storageContext->FilePath, (MAX_PATH+1)*sizeof(WCHAR));

cleanup:

    return hr;
}

```





## -see-also




<a href="https://docs.microsoft.com/windows/desktop/SecBioMet/plug-in-functions">Plug-in Functions</a>



<a href="https://docs.microsoft.com/windows/desktop/api/winbio_adapter/nc-winbio_adapter-pibio_storage_create_database_fn">StorageAdapterCreateDatabase</a>



<a href="https://docs.microsoft.com/windows/desktop/api/winbio_adapter/nc-winbio_adapter-pibio_storage_erase_database_fn">StorageAdapterEraseDatabase</a>



<a href="https://docs.microsoft.com/windows/desktop/api/winbio_adapter/nc-winbio_adapter-pibio_storage_open_database_fn">StorageAdapterOpenDatabase</a>
 

 

