---
UID: NF:wincrypt.CryptDuplicateHash
title: CryptDuplicateHash function (wincrypt.h)
description: Makes an exact copy of a hash to the point when the duplication is done.
old-location: security\cryptduplicatehash.htm
tech.root: SecCrypto
ms.assetid: 527fce4d-8d42-437b-9692-42583092efbb
ms.date: 12/05/2018
ms.keywords: CryptDuplicateHash, CryptDuplicateHash function [Security], _crypto2_cryptduplicatehash, security.cryptduplicatehash, wincrypt/CryptDuplicateHash
f1_keywords:
- wincrypt/CryptDuplicateHash
dev_langs:
- c++
req.header: wincrypt.h
req.include-header: 
req.target-type: Windows
req.target-min-winverclnt: Windows XP [desktop apps only]
req.target-min-winversvr: Windows Server 2003 [desktop apps only]
req.kmdf-ver: 
req.umdf-ver: 
req.ddi-compliance: 
req.unicode-ansi: 
req.idl: 
req.max-support: 
req.namespace: 
req.assembly: 
req.type-library: 
req.lib: Advapi32.lib
req.dll: Advapi32.dll
req.irql: 
topic_type:
- APIRef
- kbSyntax
api_type:
- DllExport
api_location:
- Advapi32.dll
- API-MS-Win-Security-cryptoapi-l1-1-0.dll
- cryptsp.dll
api_name:
- CryptDuplicateHash
targetos: Windows
req.typenames: 
req.redist: 
ms.custom: 19H1
---

# CryptDuplicateHash function


## -description


The <b>CryptDuplicateHash</b> function makes an exact copy of a <a href="https://docs.microsoft.com/windows/desktop/SecGloss/h-gly">hash</a> to the point when the duplication is done. The duplicate hash includes the <a href="https://docs.microsoft.com/windows/desktop/SecGloss/s-gly">state</a> of the hash.

A hash can be created in a piece-by-piece way. The <b>CryptDuplicateHash</b> function can be used to create separate hashes of two different contents that begin with the same content.


## -parameters




### -param hHash [in]

Handle of the hash to be duplicated.


### -param pdwReserved [in]

Reserved for future use and must be zero.


### -param dwFlags [in]

Reserved for future use and must be zero.


### -param phHash [out]

Address of the handle of the duplicated hash. When you have finished using the hash, release the handle by calling the <a href="https://docs.microsoft.com/windows/desktop/api/wincrypt/nf-wincrypt-cryptdestroyhash">CryptDestroyHash</a> function.


## -returns



If the function succeeds, the function returns <b>TRUE</b>.

If the function fails, it returns <b>FALSE</b>. For extended error information, call 
<a href="https://docs.microsoft.com/windows/desktop/api/errhandlingapi/nf-errhandlingapi-getlasterror">GetLastError</a>.

The error code prefaced by "NTE" is generated by the particular <a href="https://docs.microsoft.com/windows/desktop/SecGloss/c-gly">cryptographic service provider</a> (CSP) that you are using. Some possible error codes follow.

<table>
<tr>
<th>Return code</th>
<th>Description</th>
</tr>
<tr>
<td width="40%">
<dl>
<dt><b>ERROR_CALL_NOT_IMPLEMENTED</b></dt>
</dl>
</td>
<td width="60%">
Because this is a new function, existing CSPs cannot implement it. This error is returned if the CSP does not support this function.

</td>
</tr>
<tr>
<td width="40%">
<dl>
<dt><b>ERROR_INVALID_PARAMETER</b></dt>
</dl>
</td>
<td width="60%">
One of the parameters contains a value that is not valid. This is most often a pointer that is not valid.

</td>
</tr>
<tr>
<td width="40%">
<dl>
<dt><b>NTE_BAD_HASH</b></dt>
</dl>
</td>
<td width="60%">
A handle to the original hash is not valid.

</td>
</tr>
</table>
 




## -remarks



<b>CryptDuplicateHash</b> makes a copy of a <a href="https://docs.microsoft.com/windows/desktop/SecGloss/h-gly">hash</a> and the exact <a href="https://docs.microsoft.com/windows/desktop/SecGloss/s-gly">state</a> of the hash. This function might be used if a calling application needed to generate two hashes but both hashes had to start with some common data hashed. For example, a hash might be created, the common data hashed, a duplicate made with the <b>CryptDuplicateHash</b> function, and then the data unique to each hash would be added.

The <a href="https://docs.microsoft.com/windows/desktop/api/wincrypt/nf-wincrypt-cryptdestroyhash">CryptDestroyHash</a> function must be called to destroy any hashes that are created with <b>CryptDuplicateHash</b>. Destroying the original hash does not cause the duplicate hash to be destroyed. After a duplicate hash is made, it is separate from the original hash. There is no shared <a href="https://docs.microsoft.com/windows/desktop/SecGloss/s-gly">state</a> between the two hashes.


#### Examples

The following example shows making an exact copy of a hash. For an example that includes the complete context for this example, see <a href="https://docs.microsoft.com/windows/desktop/SecCrypto/example-c-program-duplicating-a-hash">Example C Program: Duplicating a Hash</a>.


```cpp
//-------------------------------------------------------------------
//  Declare and initialize variables.

HCRYPTPROV   hCryptProv = NULL;
HCRYPTHASH   hOriginalHash = NULL;
HCRYPTHASH   hDuplicateHash = NULL;

//-------------------------------------------------------------------
// Acquire a CSP context.

if(CryptAcquireContext(
   &hCryptProv, 
   NULL, 
   NULL, 
   PROV_RSA_FULL, 
   0)) 
{
    printf("CryptAcquireContext succeeded. \n");
}
else
{
    printf("Error during CryptAcquireContext.\n");
    exit(1);
}
//-------------------------------------------------------------------
// Create a hash.

if (CryptCreateHash(
    hCryptProv, 
    CALG_SHA1, 
    0, 
    0,
    &hOriginalHash))
{
   printf("An empty hash object has been created. \n");
}
else
{
   printf("Error during CryptCreateHash.\n");
   exit(1);
}
//-------------------------------------------------------------------
// Hash a BYTE string.

if (CryptHashData(
    hOriginalHash, 
    (BYTE*)"Some Common Data", 
    sizeof("Some Common Data"), 0))
{
   printf("An original hash has been created. \n");
}
else
{
   printf("Error during CryptHashData.\n");
   exit(1);
}
//-------------------------------------------------------------------
// Duplicate the hash.

if (CryptDuplicateHash(
   hOriginalHash, 
   NULL, 
   0, 
   &hDuplicateHash))
{
   printf("The hash has been duplicated. \n");
}
else
{
   printf("Error during CryptDuplicateHash.\n");
   exit(1);
}
//-------------------------------------------------------------------
// At this point, the two hash objects are exactly the same.
// The two hash objects can be handled separately.
// When all processing on the hash object is completed, 
// both objects should be destroyed, and the cryptographic
// context should be released.

//-------------------------------------------------------------------
// Destroy the original hash.

if(CryptDestroyHash(hOriginalHash))
{
   printf("The original hash has been destroyed. \n");
}
else
{
   printf("Error during CryptDestroyHash on the original "
       "hash object.\n");
   exit(1);
}
//-------------------------------------------------------------------
// Destroy the duplicate hash.

if (CryptDestroyHash(hDuplicateHash))
{
   printf("The duplicate hash has been destroyed. \n");
}
else
{
   printf("Error during CryptDestroyHash on the duplicated hash object.\n");
   exit(1);
}

//-------------------------------------------------------------------
// Release the CSP.

if(hCryptProv) 
   CryptReleaseContext(hCryptProv,0);

```





## -see-also




<a href="https://docs.microsoft.com/windows/desktop/api/wincrypt/nf-wincrypt-cryptdestroyhash">CryptDestroyHash</a>



<a href="https://docs.microsoft.com/windows/desktop/SecCrypto/cryptography-functions">Hash and Digital Signature Functions</a>
 

 

