---
UID: NF:winuser.RegisterHotKey
title: RegisterHotKey function (winuser.h)
description: Defines a system-wide hot key.
old-location: inputdev\registerhotkey.htm
tech.root: inputdev
ms.assetid: VS|winui|~\winui\windowsuserinterface\userinput\keyboardinput\keyboardinputreference\keyboardinputfunctions\registerhotkey.htm
ms.date: 12/05/2018
ms.keywords: MOD_ALT, MOD_CONTROL, MOD_NOREPEAT, MOD_SHIFT, MOD_WIN, RegisterHotKey, RegisterHotKey function [Keyboard and Mouse Input], _win32_RegisterHotKey, _win32_registerhotkey_cpp, inputdev.registerhotkey, winui._win32_registerhotkey, winuser/RegisterHotKey
f1_keywords:
- winuser/RegisterHotKey
dev_langs:
- c++
req.header: winuser.h
req.include-header: Windows.h
req.target-type: Windows
req.target-min-winverclnt: Windows Vista [desktop apps only]
req.target-min-winversvr: Windows Server 2003 [desktop apps only]
req.kmdf-ver: 
req.umdf-ver: 
req.ddi-compliance: 
req.unicode-ansi: 
req.idl: 
req.max-support: 
req.namespace: 
req.assembly: 
req.type-library: 
req.lib: User32.lib
req.dll: User32.dll
req.irql: 
topic_type:
- APIRef
- kbSyntax
api_type:
- DllExport
api_location:
- User32.dll
- Ext-MS-Win-NTUser-Keyboard-l1-1-0.dll
- Ext-MS-Win-NTUser-Keyboard-l1-1-1.dll
- Ext-MS-Win-RTCore-NTUser-iam-l1-1-0.dll
- ext-ms-win-ntuser-keyboard-l1-1-2.dll
- Ext-MS-Win-NTUser-Keyboard-L1-2-0.dll
- Ext-MS-Win-NTUser-Keyboard-L1-2-1.dll
- Ext-MS-Win-RTCore-NTUser-Iam-L1-1-1.dll
- Ext-MS-Win-NTUser-Keyboard-L1-3-0.dll
api_name:
- RegisterHotKey
targetos: Windows
req.typenames: 
req.redist: 
ms.custom: 19H1
---

# RegisterHotKey function


## -description


Defines a system-wide hot key.


## -parameters




### -param hWnd [in, optional]

Type: <b>HWND</b>

A handle to the window that will receive <a href="https://docs.microsoft.com/windows/desktop/inputdev/wm-hotkey">WM_HOTKEY</a> messages generated by the hot key. If this parameter is <b>NULL</b>, <b>WM_HOTKEY</b> messages are posted to the message queue of the calling thread and must be processed in the message loop.


### -param id [in]

Type: <b>int</b>

The identifier of the hot key.  If the <i>hWnd</i> parameter is NULL, then the hot key is associated with the current thread rather than with a particular window. If a hot key already exists with the same <i>hWnd</i> and <i>id</i> parameters, see Remarks for the action taken.


### -param fsModifiers [in]

Type: <b>UINT</b>

The keys that must be pressed in combination with the key specified by the 
     <i>uVirtKey</i> parameter in order to generate the <a href="https://docs.microsoft.com/windows/desktop/inputdev/wm-hotkey">WM_HOTKEY</a> message. The <i>fsModifiers</i> parameter can be a combination of the following values.

<table>
<tr>
<th>Value</th>
<th>Meaning</th>
</tr>
<tr>
<td width="40%"><a id="MOD_ALT"></a><a id="mod_alt"></a><dl>
<dt><b>MOD_ALT</b></dt>
<dt>0x0001</dt>
</dl>
</td>
<td width="60%">
Either ALT key must be held down.

</td>
</tr>
<tr>
<td width="40%"><a id="MOD_CONTROL"></a><a id="mod_control"></a><dl>
<dt><b>MOD_CONTROL</b></dt>
<dt>0x0002</dt>
</dl>
</td>
<td width="60%">
Either CTRL key must be held down.

</td>
</tr>
<tr>
<td width="40%"><a id="MOD_NOREPEAT"></a><a id="mod_norepeat"></a><dl>
<dt><b>MOD_NOREPEAT</b></dt>
<dt>0x4000</dt>
</dl>
</td>
<td width="60%">
Changes the hotkey behavior so that the keyboard auto-repeat does not yield multiple hotkey notifications.

<b>Windows Vista:  </b>This flag is not supported.

</td>
</tr>
<tr>
<td width="40%"><a id="MOD_SHIFT"></a><a id="mod_shift"></a><dl>
<dt><b>MOD_SHIFT</b></dt>
<dt>0x0004</dt>
</dl>
</td>
<td width="60%">
Either SHIFT key must be held down.

</td>
</tr>
<tr>
<td width="40%"><a id="MOD_WIN"></a><a id="mod_win"></a><dl>
<dt><b>MOD_WIN</b></dt>
<dt>0x0008</dt>
</dl>
</td>
<td width="60%">
Either WINDOWS key was held down. These keys are labeled with the Windows logo. Keyboard shortcuts that involve the WINDOWS key are reserved for use by the operating system.

</td>
</tr>
</table>
 


### -param vk [in]

Type: <b>UINT</b>

The virtual-key code of the hot key. See <a href="https://docs.microsoft.com/windows/desktop/inputdev/virtual-key-codes">Virtual Key Codes</a>.


## -returns



Type: <b>BOOL</b>

If the function succeeds, the return value is nonzero.

If the function fails, the return value is zero. To get extended error information, call <a href="https://docs.microsoft.com/windows/desktop/api/errhandlingapi/nf-errhandlingapi-getlasterror">GetLastError</a>.




## -remarks



When a key is pressed, the system looks for a match against all hot keys. Upon finding a match, the system posts the <a href="https://docs.microsoft.com/windows/desktop/inputdev/wm-hotkey">WM_HOTKEY</a> message to the message queue of the window with which the hot key is associated. If the hot key is not associated with a window, then the <b>WM_HOTKEY</b> message is posted to the thread associated with the hot key.

This function cannot associate a hot key with a window created by another thread.

<b>RegisterHotKey</b> fails if the keystrokes specified for the hot key have already been registered by another hot key.

If a hot key already exists with the same <i>hWnd</i> and <i>id</i> parameters, it is maintained along with the new hot key.  The application must explicitly call <a href="https://docs.microsoft.com/windows/desktop/api/winuser/nf-winuser-unregisterhotkey">UnregisterHotKey</a> to unregister the old hot key.

<b>Windows Server 2003:  </b>If a hot key already exists with the same <i>hWnd</i> and <i>id</i> parameters, it is replaced by the new hot key.

The F12 key is reserved for use by the debugger at all times, so it should not be registered as a hot key. Even when you are not debugging an application, F12 is reserved in case a kernel-mode debugger or a just-in-time debugger is resident.

An application must specify an id value in the range 0x0000 through 0xBFFF. A shared DLL must specify a value in the range 0xC000 through 0xFFFF (the range returned by the <a href="https://docs.microsoft.com/windows/desktop/api/winbase/nf-winbase-globaladdatoma">GlobalAddAtom</a> function). To avoid conflicts with hot-key identifiers defined by other shared DLLs, a DLL should use the <b>GlobalAddAtom</b> function to obtain the hot-key identifier.


#### Examples

The following example shows how to use the <b>RegisterHotKey</b> function with the <b>MOD_NOREPEAT</b> flag.
     In this example, the hotkey 'ALT+b' is registered for the main thread. When the hotkey is pressed, 
     the thread will receive a <a href="https://docs.microsoft.com/windows/desktop/inputdev/wm-hotkey">WM_HOTKEY</a> message, which will get picked up in the <a href="https://docs.microsoft.com/windows/desktop/api/winuser/nf-winuser-getmessage">GetMessage</a> call. 
     Because this example uses <b>MOD_ALT</b> with the <b>MOD_NOREPEAT</b> value for <i>fsModifiers</i>, the thread will only receive another <b>WM_HOTKEY</b> message when the 'b' key is released and then pressed again while the 'ALT' key is being pressed down.


```cpp

#include "stdafx.h"

int _cdecl _tmain (
    int argc, 
    TCHAR *argv[])
{           
    if (RegisterHotKey(
        NULL,
        1,
        MOD_ALT | MOD_NOREPEAT,
        0x42))  //0x42 is 'b'
    {
        _tprintf(_T("Hotkey 'ALT+b' registered, using MOD_NOREPEAT flag\n"));
    }
 
    MSG msg = {0};
    while (GetMessage(&msg, NULL, 0, 0) != 0)
    {
        if (msg.message == WM_HOTKEY)
        {
            _tprintf(_T("WM_HOTKEY received\n"));            
        }
    } 
 
    return 0;
}

```





## -see-also




<b>Conceptual</b>



<a href="https://docs.microsoft.com/windows/desktop/api/winbase/nf-winbase-globaladdatoma">GlobalAddAtom</a>



<a href="https://docs.microsoft.com/windows/desktop/inputdev/keyboard-input">Keyboard Input</a>



<b>Reference</b>



<a href="http://go.microsoft.com/fwlink/p/?LinkID=528102">Register hotkey for the current app (CSRegisterHotkey)</a>



<a href="http://go.microsoft.com/fwlink/p/?LinkID=528099">Register hotkey for the current app (CppRegisterHotkey)</a>



<a href="http://go.microsoft.com/fwlink/p/?LinkID=528101">Register hotkey for the current app (VBRegisterHotkey)</a>



<b>Samples</b>



<a href="https://docs.microsoft.com/windows/desktop/api/winuser/nf-winuser-unregisterhotkey">UnregisterHotKey</a>



<a href="https://docs.microsoft.com/windows/desktop/inputdev/wm-hotkey">WM_HOTKEY</a>
 

 

