////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//      JASON - FULLY GRAPHICAL INTERFACE TO HERCULES IBM 360+ EMULATOR       //
//                                                                            //
//                             PUNCHED CARD IMAGE                             //
//                                                                            //
//                             Start: 19.03.2010                              //
//                        Current version: 24.03.2010                         //
//                                                                            //
// Jason is free software; you can redistribute it and/or modify it under     //
// the terms of the GNU General Public License version 3, as published by the //
// Free Software Foundation.                                                  //
//                                                                            //
// Jason is distributed in the hope that it will be useful, but WITHOUT ANY   //
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS  //
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more      //
// details.                                                                   //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program. If not, see <http://www.gnu.org/licenses/>.             //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#pragma hdrstop
#include <stdio.h>
#include <dir.h>
#include <math.h>

#include "Jason.h"

// Hollerith codes (rows YX0123456789) for ASCII codes.
char *hollerith[256] = {
  "Y0189","Y19",  "Y29",  "Y39",  "79",   "0589", "0689", "0789", // 0x00
  "X69",  "Y59",  "059",  "Y389", "Y489", "Y589", "Y689", "Y789", // 0x08
  "YX189","X19",  "X29",  "X39",  "489",  "589",  "29",   "069",  // 0x10
  "X89",  "X189", "789",  "079",  "X489", "X589", "X689", "X789", // 0x18
  "",     "Y78",  "78",   "38",   "X38",  "048",  "12",   "58",   //  !"#$%&'
  "Y58",  "X58",  "X48",  "Y68",  "038",  "X",    "Y38",  "01",   // ()*+,-./
  "0",    "1",    "2",    "3",    "4",    "5",    "6",    "7",    // 01234567
  "8",    "9",    "28",   "X68",  "Y48",  "68",   "068",  "078",  // 89:;<=>?
  "48",   "Y1",   "Y2",   "Y3",   "Y4",   "Y5",   "Y6",   "Y7",   // @ABCDEFG
  "Y8",   "Y9",   "X1",   "X2",   "X3",   "X4",   "X5",   "X6",   // HIJKLMNO
  "X7",   "X8",   "X9",   "02",   "03",   "04",   "05",   "06",   // PQRSTUVW
  "07",   "08",   "09",   "Y28",  "028",  "X28",  "X78",  "058",  // XYZ[\]^_
  "18",   "Y01",  "Y02",  "Y03",  "Y04",  "Y05",  "Y06",  "Y07",  // `abcdefg
  "Y08",  "Y09",  "YX1",  "YX2",  "YX3",  "YX4",  "YX5",  "YX6",  // hijklmno
  "YX7",  "YX8",  "YX9",  "X02",  "X03",  "X04",  "X05",  "X06",  // pqrstuvw
  "X07",  "X08",  "X09",  "Y0",   "YX",   "X0",   "X01",  "Y79",  // xyz{|}~
  "X0189","019",  "029",  "039",  "049",  "X59",  "Y69",  "X79",  // 0x80
  "089",  "0189", "0289", "0389", "0489", "Y189", "Y289", "X389", // 0x88
  "YX0189","19",  "X289", "39",   "49",   "59",   "69",   "Y89",  // 0x90
  "89",   "189",  "289",  "389",  "Y49",  "X49",  "689",  "X019", // 0x98
  "Y019", "Y029", "Y039", "Y049", "Y059", "Y069", "Y079", "Y089", // 0xA0
  "Y18",  "YX19", "YX29", "YX39", "YX49", "YX59", "YX69", "YX79", // 0xA8
  "YX89", "X18",  "X029", "X039", "X049", "X059", "X069", "X079", // 0xB0
  "X089", "018",  "YX0",  "YX019","YX029","YX039","YX049","YX059",// 0xB8
  "YX069","YX079","YX089","Y018", "Y028", "Y038", "Y048", "Y058", // 0xC0
  "Y068", "Y078", "YX18", "YX28", "YX38", "YX48", "YX58", "YX68", // 0xC8
  "YX78", "X018", "X028", "X038", "X048", "X058", "X068", "X078", // 0xD0
  "YX018","YX01", "YX02", "YX03", "YX04", "YX05", "YX06", "YX07", // 0xD8
  "YX08", "YX09", "YX028","YX038","YX048","YX058","YX068","YX078",// 0xE0
  "Y0289","Y0389","Y0489","Y0589","Y0689","Y0789","YX289","YX389",// 0xE8
  "YX489","YX589","YX689","YX789","X0289","X0389","X0489","X0589",// 0xF0
  "X0689","X0789","YX0289","YX0389","YX0489","YX0589","YX0689","YX0789"
};

// Draws punched card in the specified rectange, filling background with the
// given color. Text is expected to have 80 ASCII characters (NULL: empty card).
// Text selection is set to the range sel0..sel1, the last is not included.
// Optional structure cardpos receives data used by Getcardcharrect() to find
// position of the characters in the text. Returns 0 on success and -1 on error.
int Drawcard(HDC dc,RECT *rc,COLORREF bg,int lineno,char *text,
  int sel0,int sel1,t_cardpos *cp) {
  int i,j,c,x,y,dx,dy,sizex,sizey,viewx,viewy,windx,windy,diameter;
  float factorx,factory;
  char s[32];
  COLORREF cardrgb,selrgb,holergb,textrgb,digitrgb;
  POINT poly[3];
  RECT rb,rt;
  TEXTMETRIC tm;
  HDC mc;
  HBITMAP hbmp;
  HPEN bgpen,cardpen,holepen;
  HBRUSH bgbrush,cardbrush,holebrush;
  if (dc==NULL || rc==NULL || rc->right-rc->left<1 || rc->bottom-rc->top<1)
    return -1;                         // Error in input parameters
  // Prepare memory DC.
  dx=11; dy=32;
  sizex=86*dx; sizey=13*dy;
  diameter=(dy*3)/2;
  mc=CreateCompatibleDC(dc);
  if (mc==NULL)
    return -1;                         // Unable to create memory DC
  hbmp=CreateCompatibleBitmap(dc,sizex,sizey);
  if (hbmp==NULL) {
    DeleteDC(mc);                      // Unable to create memory bitmap.
    return -1; };
  SelectObject(mc,hbmp);
  // Determine colours.
  cardrgb=RGB(220,200,120);
  selrgb=RGB(160,140,100);
  holergb=RGB(GetRValue(bg)/2,GetGValue(bg)/2,GetBValue(bg)/2);
  textrgb=RGB(30,20,10);
  digitrgb=RGB(110,100,80);
  // Create controls.
  bgpen=CreatePen(PS_SOLID,0,bg);
  cardpen=CreatePen(PS_SOLID,0,cardrgb);
  holepen=CreatePen(PS_SOLID,0,holergb);
  bgbrush=CreateSolidBrush(bg);
  cardbrush=CreateSolidBrush(cardrgb);
  holebrush=CreateSolidBrush(holergb);
  // Fill corners with background colour.
  SelectObject(mc,bgpen);
  SelectObject(mc,bgbrush);
  Rectangle(mc,0,0,diameter,diameter);
  Rectangle(mc,0,sizey-diameter,diameter,sizey);
  Rectangle(mc,sizex-diameter,0,sizex,diameter);
  Rectangle(mc,sizex-diameter,sizey-diameter,sizex,sizey);
  // Draw paper base.
  SelectObject(mc,cardpen);
  SelectObject(mc,cardbrush);
  RoundRect(mc,0,0,sizex,sizey,diameter,diameter);
  SelectObject(mc,bgpen);
  SelectObject(mc,bgbrush);
  poly[0].x=0; poly[0].y=0;
  poly[1].x=dy; poly[1].y=0;
  poly[2].x=0; poly[2].y=dy*3/2;
  Polygon(mc,poly,3);
  // Draw digits.
  SetTextAlign(mc,TA_BOTTOM|TA_CENTER);
  SetTextColor(mc,digitrgb);
  SetBkMode(mc,TRANSPARENT);
  SelectObject(mc,GetStockObject(SYSTEM_FONT));
  for (j=0,y=dy*3+dy/4; j<10; j++,y+=dy) {
    s[0]=(char)('0'+j);
    for (i=0,x=(dx*7)/2; i<80; i++,x+=dx) {
      if ((j==4 || j==5) && i>=34 && i<46) continue;
      TextOut(mc,x,y,s,1);
    };
  };
  // Draw column numbers.
  SelectObject(mc,smallcardfont.hfont);
  for (i=0,x=(dx*7)/2; i<80; i++,x+=dx) {
    sprintf(s,"%02i",i+1);
    TextOut(mc,x,dy*3+dy/2+3,s,2);
    TextOut(mc,x,dy*11+dy/2+3,s,2); };
  // Draw logo.
  SelectObject(mc,largefont.hfont);
  TextOut(mc,sizex/2,dy*8,"JASON",5);
  // Draw text, holes and line number.
  SelectObject(mc,GetStockObject(OEM_FIXED_FONT));
  GetTextMetrics(mc,&tm);
  if (text!=NULL) {
    SetTextColor(mc,textrgb);
    SetTextAlign(mc,TA_TOP|TA_CENTER);
    SelectObject(mc,holepen);
    SelectObject(mc,holebrush);
    rt.left=dx*3; rt.right=rt.left+dx;
    rt.top=dy/4-1; rt.bottom=rt.top+tm.tmHeight+1;
    for (i=0,x=(dx*7)/2; i<80; i++,x+=dx) {
      if (i>=sel0 && i<sel1)
        SetBkColor(mc,selrgb);
      else
        SetBkColor(mc,cardrgb);
      ExtTextOut(mc,x,dy/4,ETO_CLIPPED|ETO_OPAQUE,&rt,text+i,1,NULL);
      c=(unsigned char)text[i];
      for (j=0; j<6; j++) {          // At most 6 punches!
        if (hollerith[c][j]=='Y') y=1;
        else if (hollerith[c][j]=='X') y=2;
        else if (isdigit(hollerith[c][j])) y=hollerith[c][j]-'0'+3;
        else break;
        y=y*dy;
        Rectangle(mc,x-dx/3-1,y-dy/4-1,x+dx/3+1,y+dy/4+1);
      };
      rt.left=rt.right; rt.right+=dx;
    };
    if (lineno>=0) {
      SetTextAlign(mc,TA_BOTTOM|TA_LEFT);
      SetBkColor(mc,cardrgb);
      j=sprintf(s,"%08i",lineno);
      ExtTextOut(mc,dx*3,sizey-dy/4,ETO_OPAQUE,NULL,s,j,NULL);
    };
  };
  // Get rectangle on the screen where bitmap will be placed and paint the rest
  // with the background color.
  viewx=rc->right-rc->left;
  viewy=rc->bottom-rc->top;
  rb=*rc;
  SelectObject(dc,bgpen);
  SelectObject(dc,bgbrush);
  if (viewx*sizey>viewy*sizex) {
    // Screen rectangle is too wide.
    windx=(viewy*sizex)/sizey; windy=viewy; }
  else {
    // Screen rectangle is too high.
    windx=viewx; windy=(viewx*sizey)/sizex; };
  if (windx>=sizex && windy>=sizey) {
    // Screen is large than card.
    windx=sizex; windy=sizey; };
  rb.left+=(viewx-windx)/2;
  rb.right=rb.left+windx;
  if (rb.left>rc->left)
    Rectangle(dc,rc->left,rc->top,rb.left,rc->bottom);
  if (rb.right<rc->right)
    Rectangle(dc,rb.right,rc->top,rc->right,rc->bottom);
  rb.top+=(viewy-windy)/2;
  rb.bottom=rb.top+windy;
  if (rb.top>rc->top)
    Rectangle(dc,rb.left,rc->top,rb.right,rb.top);
  if (rb.bottom<rc->bottom)
    Rectangle(dc,rb.left,rb.bottom,rb.right,rc->bottom);
  // Copy bitmap from memory to original DC.
  SetStretchBltMode(dc,STRETCH_HALFTONE);
  StretchBlt(dc,rb.left,rb.top,rb.right-rb.left,rb.bottom-rb.top,
    mc,0,0,sizex,sizey,SRCCOPY);
  // Fill card position structure.
  if (cp!=NULL) {
    factorx=(float)(rb.right-rb.left)/(float)sizex;
    factory=(float)(rb.bottom-rb.top)/(float)sizey;
    cp->basex=rb.left+(dx*3)*factorx;
    cp->basey=rb.top+(dy/4)*factory;
    cp->chardx=dx*factorx;
    cp->chardy=tm.tmHeight*factory; };
  // Clean up.
  DeleteObject(holebrush);
  DeleteObject(cardbrush);
  DeleteObject(bgbrush);
  DeleteObject(holepen);
  DeleteObject(cardpen);
  DeleteObject(bgpen);
  DeleteObject(hbmp);
  DeleteDC(mc);
  // Report success.
  return 0;
};

// Given 0-based character index on the card, determines character location in
// client pixels. Returns 0 on success and -1 on error.
int Getcardcharrect(t_cardpos *cp,int index,RECT *rc) {
  if (cp==NULL || index<0 || index>80 || rc==NULL)
    return -1;                         // Error in input parameters
  rc->left=(int)floor(cp->basex+index*cp->chardx+0.5);
  rc->top=(int)floor(cp->basey+0.5);
  rc->right=(int)floor(cp->basex+(index+1)*cp->chardx+0.5);
  rc->bottom=(int)floor(cp->basey+cp->chardy+0.5);
  return 0;
};

// Given window's X client coordinate of the card image, returns 0-based index
// of the corresponding character, or -1 on error.
int Getcardcharindex(t_cardpos *cp,int x) {
  int index;
  if (cp==NULL || cp->chardx<=0.0)
    return -1;                         // Error in input parameters
  index=(int)floor((x-cp->basex+cp->chardx/2)/cp->chardx);
  if (index<0)
    index=0;
  else if (index>80)
    index=80;
  return index;
};

// Creates and sets caret on the scaled card image.
void Setcardcaret(HWND hw,t_cardpos *cp,int cursorx,int overwrite) {
  int dx;
  RECT rc;
  if (hw==NULL || cp==NULL)
    return;                            // Error in input parameters
  if (GetFocus()!=hw)
    return;                            // Window with caret must have focus
  if (Getcardcharrect(cp,cursorx,&rc)!=0)
    return;                            // Unable to calculate caret position
  if (overwrite)
    dx=max(1L,rc.right-rc.left);
  else
    dx=max(1L,(rc.right-rc.left)/4);
  CreateCaret(hw,NULL,dx,rc.bottom-rc.top+2);
  SetCaretPos(rc.left-(overwrite?0:dx/2),rc.top-1);
};

