////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//      JASON - FULLY GRAPHICAL INTERFACE TO HERCULES IBM 360+ EMULATOR       //
//                                                                            //
//                               DEVICE WINDOW                                //
//                                                                            //
//                             Start: 19.02.2010                              //
//                        Current version: 02.10.2010                         //
//                                                                            //
// Jason is free software; you can redistribute it and/or modify it under     //
// the terms of the GNU General Public License version 3, as published by the //
// Free Software Foundation.                                                  //
//                                                                            //
// Jason is distributed in the hope that it will be useful, but WITHOUT ANY   //
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS  //
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more      //
// details.                                                                   //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program. If not, see <http://www.gnu.org/licenses/>.             //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#pragma hdrstop
#include <stdio.h>
#include <dir.h>

#include "Jason.h"

#define NHIST          24              // Length of command history
#define NAUX           8               // Max number of auxiliary controls
#define NBOOK          80              // Max number of books on the bookshelf

// Identifiers of child device controls. Note that bitmap buttons look for
// bitmaps with the same resource ID.
#define DCI_HERCCMD    1001            // ID of Hercules command list
#define DCI_HERSEND    1002            // ID of Hercules send button
#define DCI_BOOKLIST   1003            // ID of book list

#define DCI_SCISSORS   2001            // ID/bitmap of Cut paper button
#define DCI_TRASH      2002            // ID/bitmap of Discard paper button
#define DCI_STARTP     2003            // ID/bitmap of Start printer button
#define DCI_STOPP      2004            // ID/bitmap of Stop printer button

#define DCI_NEWDECK    2010            // ID/bitmap of New deck
#define DCI_OPENDECK   2011            // ID/bitmap of Open deck
#define DCI_SAVEDECK   2012            // ID/bitmap of Save deck
#define DCI_CARDVIEW   2013            // ID/bitmap of Toggle view
#define DCI_SUBMIT     2014            // ID/bitmap of Submit deck to RDR

#define DCI_NEWDISK    2020            // ID/bitmap of Load volume
#define DCI_TAPERO     2021            // ID/bitmap of Load tape read only
#define DCI_TAPERW     2022            // ID/bitmap of Load tape read/write
#define DCI_NOTAPE     2023            // ID/bitmap of Remove tape
#define DCI_NEWTAPE    2024            // ID/bitmap of New tape
#define DCI_ATTN       2025            // ID/bitmap of Attention Interrupt

#define DCI_ADDBOOK    2030            // ID/bitmap of Add book
#define DCI_OPENBOOK   2031            // ID/bitmap of Open book
#define DCI_DELBOOK    2032            // ID/bitmap of Remove book
#define DCI_EARLIER    2033            // ID/bitmap of Move book up
#define DCI_LATER      2034            // ID/bitmap of Move book down

typedef struct t_devsel {              // Descriptor of selected device
  int            devclass;             // Device class, one of DEV_xxx
  int            devindex;             // Index of selected device
  HWND           hlist;                // List device window, if any
  RECT           rlist;                // Position of list device window
  HWND           haux[NAUX];           // Auxiliary controls
  RECT           raux[NAUX];           // Positions of auxiliary controls
  HWND           hfocus;               // One of above windows that gets focus
} t_devsel;

typedef struct t_devimg {              // Image descriptor
  int            devclass;             // Device class, one of DEV_xxx
  int            devtype;              // Preferred device type (NNNN, decimal)
  char           *bitmap;              // Name of bitmap resource
  char           *protmap;             // Name of protected media bitmap
  char           *unprmap;             // Name of unprotected media bitmap
  int            listx0,listy0;        // Top left corner of list in bitmap
  int            listdx,listdy;        // Size of the list in bitmap
  int            statex0,statey0;      // Top left corner of state in bitmap
  int            textx0,texty0;        // Top left corner of text in bitmap
  int            textdx,textdy;        // Size of the text in bitmap
  COLORREF       background;           // Colour of device background
} t_devimg;

static t_devimg  devimg[] = {          // Available device images
  { DEV_SHELF,    0, "BOOKS", NULL,    NULL,     -1, -1,  0,  0,   -1, -1,   -1, -1,  0,  0, RGB(150,150,40) },
  { DEV_CPU,      0, "CPU",   NULL,    NULL,      5, 32,151,140,   -1, -1,   10, 10,141, 17, RGB(200,150,100) },
  { DEV_HERC,     0, "CONS",  NULL,    NULL,     24,  9, 71, 52,   -1, -1,   -1, -1,  0,  0 },
  { DEV_PRT,   1403, "P1403", NULL,    NULL,     45, 90,132, 86,    7, 15,   -1, -1,  0,  0 },
  { DEV_PRT,   3211, "P3211", NULL,    NULL,     77, 84,132,107,   10,  7,   -1, -1,  0,  0 },
  { DEV_CON,   3215, "T3215", NULL,    NULL,     27,  1, 81, 43,   -1, -1,   -1, -1,  0,  0 },
  { DEV_DSP,   3270, "T3270", NULL,    NULL,     24,  9, 71, 52,   -1, -1,   -1, -1,  0,  0, RGB(50,50,50) },
  { DEV_PCH,   3525, "P3525", NULL,    NULL,     -1, -1,  0,  0,   79, 25,   -1, -1,  0,  0 },
  { DEV_RDR,   3505, "R3505", NULL,    NULL,     -1, -1,  0,  0,  135, 25,   -1, -1,  0,  0 },
  { DEV_DASD,  2311, "D2311", "DISC1", "DISC1",   0,  0,129, 87,   21, 74,   43, 22, 43, 11 },
  { DEV_TAPE,  2401, "T2401", "TPROT", "TUNPR",  10, 31,115,155,   39,  7,   75,111, 45, 12 },
  { DEV_NONE,     0, "UNKN",  NULL,    NULL,     -1, -1,  0,  0,   -1, -1,   -1, -1,  0,  0 }
};

static COLORREF  lptcolors[4] = {      // List of colors for printer
  0x00888888,                          // 0  - Gray
  0x00FFFFFF,                          // 1  - White
  0x00DDFFDD,                          // 2  - Whitegreen
  0x00000000 };                        // 3  - Black

static COLORREF  dircolors[3] = {      // List of colors for DASD/tape dir
  0x00FFFFFF,                          // 0  - White
  0x00000000,                          // 1  - Black
  0x00000080 };                        // 2  - Dark red

static t_devsel  devsel;               // Descriptor of selected device
static WNDPROC   oldcomboeditproc;     // Original proc of edit in combobox
static char      hist[NHIST][TEXTLEN]; // Hercules command history
static int       nhist;                // Length of Hercules command history
static char      book[NBOOK][MAXPATH]; // List of books in bookshelf
static int       nbook;                // Number of books in bookshelf

// Callback function of Selectdevice(), deletes all children of device window.
BOOL CALLBACK Enumdevchildren(HWND hw,LPARAM lp) {
  DestroyWindow(hw);
  return 1;
};

// Sets size of device-dependent controls.
void Setdevicecontrolsize(void) {
  int width,height;
  RECT rc;
  GetClientRect(hwdevice,&rc);
  switch (devsel.devclass) {
    case DEV_SHELF:                      // Bookshelf
      devsel.rlist.left=BORDER;
      devsel.rlist.top=BORDER;
      devsel.rlist.right=rc.right-2*BORDER-BUTTONDX;
      devsel.rlist.bottom=rc.bottom-BORDER;
      devsel.raux[0].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[0].top=0*BUTTONDY+1*BORDER;
      devsel.raux[0].right=rc.right-BORDER;
      devsel.raux[0].bottom=1*BUTTONDY+1*BORDER;
      devsel.raux[1].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[1].top=1*BUTTONDY+2*BORDER;
      devsel.raux[1].right=rc.right-BORDER;
      devsel.raux[1].bottom=2*BUTTONDY+2*BORDER;
      devsel.raux[2].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[2].top=2*BUTTONDY+4*BORDER;
      devsel.raux[2].right=rc.right-BORDER;
      devsel.raux[2].bottom=3*BUTTONDY+4*BORDER;
      devsel.raux[3].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[3].top=3*BUTTONDY+6*BORDER;
      devsel.raux[3].right=rc.right-BORDER;
      devsel.raux[3].bottom=4*BUTTONDY+6*BORDER;
      devsel.raux[4].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[4].top=4*BUTTONDY+7*BORDER;
      devsel.raux[4].right=rc.right-BORDER;
      devsel.raux[4].bottom=5*BUTTONDY+7*BORDER;
      break;
    case DEV_HERC:                       // Hercules console
      devsel.rlist.left=BORDER;
      devsel.rlist.top=BORDER;
      devsel.rlist.right=rc.right-BORDER;
      devsel.rlist.bottom=rc.bottom-BUTTONDY-2*BORDER;
      devsel.raux[0].left=BORDER;
      devsel.raux[0].top=rc.bottom-BUTTONDY-BORDER;
      devsel.raux[0].right=rc.right-2*BORDER-BUTTONDX;
      devsel.raux[0].bottom=rc.bottom-BUTTONDY-BORDER+240;
      devsel.raux[1].left=rc.right-BORDER-BUTTONDX;
      devsel.raux[1].top=rc.bottom-BUTTONDY-BORDER;
      devsel.raux[1].right=rc.right-BORDER;
      devsel.raux[1].bottom=rc.bottom-BORDER;
      break;
    case DEV_CPU:                        // CPU
      width=358; height=168;
      Getiplpanelsize(&width,&height);
      devsel.raux[0].left=rc.right-width-BORDER;
      devsel.raux[0].top=BORDER;
      devsel.raux[0].right=rc.right-BORDER;
      devsel.raux[0].bottom=BORDER+height;
      devsel.rlist.left=BORDER;
      devsel.rlist.top=BORDER;
      devsel.rlist.right=rc.right-width-2*BORDER;
      devsel.rlist.bottom=rc.bottom-BORDER;
      break;
    case DEV_RDR:                        // Card reader
      devsel.rlist.left=BORDER;
      devsel.rlist.top=BORDER;
      devsel.rlist.right=rc.right-BUTTONDX-2*BORDER;
      devsel.rlist.bottom=rc.bottom-BORDER;
      devsel.raux[0].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[0].top=0*BUTTONDY+1*BORDER;
      devsel.raux[0].right=rc.right-BORDER;
      devsel.raux[0].bottom=1*BUTTONDY+1*BORDER;
      devsel.raux[1].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[1].top=1*BUTTONDY+2*BORDER;
      devsel.raux[1].right=rc.right-BORDER;
      devsel.raux[1].bottom=2*BUTTONDY+2*BORDER;
      devsel.raux[2].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[2].top=2*BUTTONDY+3*BORDER;
      devsel.raux[2].right=rc.right-BORDER;
      devsel.raux[2].bottom=3*BUTTONDY+3*BORDER;
      devsel.raux[3].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[3].top=3*BUTTONDY+5*BORDER;
      devsel.raux[3].right=rc.right-BORDER;
      devsel.raux[3].bottom=4*BUTTONDY+5*BORDER;
      devsel.raux[4].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[4].top=4*BUTTONDY+7*BORDER;
      devsel.raux[4].right=rc.right-BORDER;
      devsel.raux[4].bottom=5*BUTTONDY+7*BORDER;
      break;
    case DEV_PCH:                        // Card puncher
      devsel.rlist.left=BORDER;
      devsel.rlist.top=BORDER;
      devsel.rlist.right=rc.right-BUTTONDX-2*BORDER;
      devsel.rlist.bottom=rc.bottom-BORDER;
      devsel.raux[0].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[0].top=0*BUTTONDY+1*BORDER;
      devsel.raux[0].right=rc.right-BORDER;
      devsel.raux[0].bottom=1*BUTTONDY+1*BORDER;
      devsel.raux[1].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[1].top=1*BUTTONDY+2*BORDER;
      devsel.raux[1].right=rc.right-BORDER;
      devsel.raux[1].bottom=2*BUTTONDY+2*BORDER;
      devsel.raux[2].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[2].top=2*BUTTONDY+3*BORDER;
      devsel.raux[2].right=rc.right-BORDER;
      devsel.raux[2].bottom=3*BUTTONDY+3*BORDER;
      break;
    case DEV_PRT:                        // Line printer
      width=640; height=480;
      Getlistwindowsize(device[devsel.devindex].list,
        prtfont.hfont,
        LWO_LBORDER|LWO_SUNKEN|LWO_PERFO,&width,&height);
      if (width>rc.right-BUTTONDX-3*BORDER)
        width=rc.right-BUTTONDX-3*BORDER;
      devsel.rlist.left=BORDER;
      devsel.rlist.top=BORDER;
      devsel.rlist.right=BORDER+width;
      devsel.rlist.bottom=rc.bottom-BORDER;
      devsel.raux[0].left=width+2*BORDER;
      devsel.raux[0].top=BORDER;
      devsel.raux[0].right=width+2*BORDER+BUTTONDX;
      devsel.raux[0].bottom=BORDER+BUTTONDY;
      devsel.raux[1].left=width+2*BORDER;
      devsel.raux[1].top=BUTTONDY+2*BORDER;
      devsel.raux[1].right=width+2*BORDER+BUTTONDX;
      devsel.raux[1].bottom=2*BUTTONDY+2*BORDER;
      devsel.raux[2].left=width+2*BORDER;
      devsel.raux[2].top=2*BUTTONDY+3*BORDER;
      devsel.raux[2].right=width+2*BORDER+BUTTONDX;
      devsel.raux[2].bottom=3*BUTTONDY+3*BORDER;
      devsel.raux[3].left=width+2*BORDER;
      devsel.raux[3].top=3*BUTTONDY+4*BORDER;
      devsel.raux[3].right=width+2*BORDER+BUTTONDX;
      devsel.raux[3].bottom=4*BUTTONDY+4*BORDER;
      break;
    case DEV_CON:                        // Console printer
      devsel.rlist.left=BORDER;
      devsel.rlist.top=BORDER;
      devsel.rlist.right=rc.right-2*BORDER-BUTTONDX;
      devsel.rlist.bottom=rc.bottom-BORDER;
      devsel.raux[0].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[0].top=0*BUTTONDY+1*BORDER;
      devsel.raux[0].right=rc.right-BORDER;
      devsel.raux[0].bottom=1*BUTTONDY+1*BORDER;
      devsel.raux[1].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[1].top=1*BUTTONDY+2*BORDER;
      devsel.raux[1].right=rc.right-BORDER;
      devsel.raux[1].bottom=2*BUTTONDY+2*BORDER;
      break;
    case DEV_DSP:                        // Display terminal
      width=640; height=480;
      Getterminalwindowsize(device[devsel.devindex].terminal,
        dspfont.hfont,LWO_LBORDER|LWO_SUNKEN,&width,&height);
      if (height+2*BORDER>rc.bottom) {
        height=rc.bottom-2*BORDER;
        width+=GetSystemMetrics(SM_CXVSCROLL); };
      if (width+2*BORDER>rc.right) width=rc.right-2*BORDER;
      devsel.rlist.left=(rc.right-width)/2;
      devsel.rlist.top=(rc.bottom-height)/2;        ////BORDER;
      devsel.rlist.right=(rc.right-width)/2+width;
      devsel.rlist.bottom=devsel.rlist.top+height;
      break;
    case DEV_DASD:                       // Direct Access Storage
      devsel.rlist.left=BORDER;
      devsel.rlist.top=BORDER;
      devsel.rlist.right=rc.right-2*BORDER-BUTTONDX;
      devsel.rlist.bottom=rc.bottom-BORDER;
      devsel.raux[0].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[0].top=0*BUTTONDY+1*BORDER;
      devsel.raux[0].right=rc.right-BORDER;
      devsel.raux[0].bottom=1*BUTTONDY+1*BORDER;
      devsel.raux[1].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[1].top=1*BUTTONDY+2*BORDER;
      devsel.raux[1].right=rc.right-BORDER;
      devsel.raux[1].bottom=2*BUTTONDY+2*BORDER;
      break;
    case DEV_TAPE:                       // Tape
      devsel.rlist.left=BORDER;
      devsel.rlist.top=BORDER;
      devsel.rlist.right=rc.right-2*BORDER-BUTTONDX;
      devsel.rlist.bottom=rc.bottom-BORDER;
      devsel.raux[0].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[0].top=0*BUTTONDY+1*BORDER;
      devsel.raux[0].right=rc.right-BORDER;
      devsel.raux[0].bottom=1*BUTTONDY+1*BORDER;
      devsel.raux[1].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[1].top=1*BUTTONDY+2*BORDER;
      devsel.raux[1].right=rc.right-BORDER;
      devsel.raux[1].bottom=2*BUTTONDY+2*BORDER;
      devsel.raux[2].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[2].top=2*BUTTONDY+3*BORDER;
      devsel.raux[2].right=rc.right-BORDER;
      devsel.raux[2].bottom=3*BUTTONDY+3*BORDER;
      devsel.raux[3].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[3].top=3*BUTTONDY+5*BORDER;
      devsel.raux[3].right=rc.right-BORDER;
      devsel.raux[3].bottom=4*BUTTONDY+5*BORDER;
      devsel.raux[4].left=rc.right-BUTTONDX-BORDER;
      devsel.raux[4].top=4*BUTTONDY+7*BORDER;
      devsel.raux[4].right=rc.right-BORDER;
      devsel.raux[4].bottom=5*BUTTONDY+7*BORDER;
      break;
    default: break;
  };
};

// Fits device window to the new size of the main Jason window.
void Resizedevicewindow(void) {
  int i,y0,dy;
  RECT rc;
  // Resize device window.
  GetClientRect(hwmain,&rc);
  if (compactview==0)
    y0=FRAMEDY+2*BORDER;
  else
    y0=GetSystemMetrics(SM_CYHSCROLL)+2*GetSystemMetrics(SM_CYBORDER)+
    mediumfont.dy+8+2*BORDER;
  dy=max((long)DEVICEDY,(long)(rc.bottom-y0-BORDER));
  MoveWindow(hwdevice,BORDER,y0,rc.right-2*BORDER,dy,1);
  // Resize contents of device window.
  Setdevicecontrolsize();
  if (devsel.hlist!=NULL) MoveWindow(devsel.hlist,
    devsel.rlist.left,devsel.rlist.top,
    devsel.rlist.right-devsel.rlist.left,
    devsel.rlist.bottom-devsel.rlist.top,1);
  for (i=0; i<NAUX; i++) {
    if (devsel.haux[i]!=NULL) MoveWindow(devsel.haux[i],
      devsel.raux[i].left,devsel.raux[i].top,
      devsel.raux[i].right-devsel.raux[i].left,
      devsel.raux[i].bottom-devsel.raux[i].top,1);
    ;
  };
};

// Service function, updates enable/disable state of the shelf buttons
// according to selection.
static void Updateshelfbuttons(void) {
  int selection;
  if (devsel.devclass!=DEV_SHELF)
    return;                            // Not a shelf window
  if (nbook<=0)
    selection=LB_ERR;
  else
    selection=SendMessage(devsel.hlist,LB_GETCURSEL,0,0);
  // Add book button.
  EnableWindow(devsel.haux[0],(nbook<NBOOK?1:0));
  // Open book button.
  EnableWindow(devsel.haux[1],(selection!=LB_ERR));
  // Remove book button.
  EnableWindow(devsel.haux[2],(selection!=LB_ERR));
  // Move book up.
  EnableWindow(devsel.haux[3],(selection!=LB_ERR && selection>0));
  // Move book down.
  EnableWindow(devsel.haux[4],(selection!=LB_ERR && selection<nbook-1));
};

// Service function, opens selected shelf book.
static void Openselectedbook(void) {
  int selection;
  if (devsel.devclass!=DEV_SHELF)
    return;                            // Not a shelf window
  if (nbook<=0)
    return;                            // Empty shelf
  selection=SendMessage(devsel.hlist,LB_GETCURSEL,0,0);
  if (selection==LB_ERR || selection<0 || selection>=nbook)
    return;                            // No selection
  ShellExecute(hwdevice,"open",book[selection],NULL,".",SW_SHOW);
};

// Service function, converts path to short book name. Returns pointer to short
// book name on success, or to path on error.
static char *Getshortbookname(char *name,char *path) {
  char fil[MAXFILE],ext[MAXEXT];
  if (path==NULL)
    return path;
  fnsplit(path,NULL,NULL,fil,ext);
  fnmerge(name,NULL,NULL,fil,ext);
  return name;
};

// Service function, sends command from the combobox in Hercules console device
// to Hercules and updates command history.
static void Sendandupdate(void) {
  int i,n;
  char s[TEXTLEN];
  HWND hchild;
  hchild=GetDlgItem(hwdevice,DCI_HERCCMD);
  if (hchild==NULL) return;
  if (SendMessage(hchild,WM_GETTEXT,TEXTLEN,(LPARAM)s)<=0) return;
  Sendstdin("%s",s);                   // s may contain '%'!
  SendMessage(hchild,CB_INSERTSTRING,0,(LPARAM)s);
  nhist=0;
  for (i=0; nhist<NHIST; i++) {
    n=SendMessage(hchild,CB_GETLBTEXT,i,(LPARAM)hist[nhist]);
    if (n<=0 || n==CB_ERR) break;
    if (i!=0 && strcmp(hist[nhist],s)==0) continue;
    nhist++;
  };
};

// Procedure that subclasses edit control of combo box. Processes Enter key.
LRESULT CALLBACK Comboeditsubclass(HWND hw,UINT msg,WPARAM wp,LPARAM lp) {
  if (msg==WM_KEYDOWN && wp==VK_RETURN) {
    Sendandupdate();
    return 0; }
  else if (oldcomboeditproc!=NULL)
    return oldcomboeditproc(hw,msg,wp,lp);
  else                                 // Emergency solution
    return DefWindowProc(hw,msg,wp,lp);
  ;
};

// Selects device with given index and sets contents of the device window. Call
// it with select=-1 to clear device window.
void Selectdevice(int select) {
  int i;
  char name[MAXPATH],fil[MAXFILE],ext[MAXEXT];
  t_device *pdev;
  RECT rc;
  HWND hchild;
  // Clear device window and its descriptor.
  EnumChildWindows(hwdevice,(WNDENUMPROC)Enumdevchildren,0);
  memset(&devsel,0,sizeof(devsel));
  if (select<0 || select>=ndevice)
    return;                            // Error in input parameters
  pdev=device+select;
  if (pdev->devclass==DEV_NONE)
    return;                            // Placeholder
  // Create device-dependent controls.
  GetClientRect(hwdevice,&rc);
  devsel.devclass=pdev->devclass;
  devsel.devindex=select;
  Setdevicecontrolsize();
  switch (pdev->devclass) {
    case DEV_SHELF:                    // Bookshelf
      devsel.hlist=
        CreateWindowEx(WS_EX_CLIENTEDGE,"LISTBOX",NULL,
        WS_CHILD|WS_VISIBLE|WS_VSCROLL|LBS_NOINTEGRALHEIGHT|LBS_NOTIFY,
        devsel.rlist.left,devsel.rlist.top,
        devsel.rlist.right-devsel.rlist.left,
        devsel.rlist.bottom-devsel.rlist.top,
        hwdevice,(HMENU)DCI_BOOKLIST,hinst,NULL);
      SendMessage(devsel.hlist,WM_SETFONT,(WPARAM)mediumfont.hfont,1);
      devsel.haux[0]=CreateWindow("BUTTON","Add book",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[0].left,devsel.raux[0].top,
        devsel.raux[0].right-devsel.raux[0].left,
        devsel.raux[0].bottom-devsel.raux[0].top,
        hwdevice,(HMENU)DCI_ADDBOOK,hinst,0);
      devsel.haux[1]=CreateWindow("BUTTON","Open book",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[1].left,devsel.raux[1].top,
        devsel.raux[1].right-devsel.raux[1].left,
        devsel.raux[1].bottom-devsel.raux[1].top,
        hwdevice,(HMENU)DCI_OPENBOOK,hinst,0);
      devsel.haux[2]=CreateWindow("BUTTON","Remove",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[2].left,devsel.raux[2].top,
        devsel.raux[2].right-devsel.raux[2].left,
        devsel.raux[2].bottom-devsel.raux[2].top,
        hwdevice,(HMENU)DCI_DELBOOK,hinst,0);
      devsel.haux[3]=CreateWindow("BUTTON","Move up",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[3].left,devsel.raux[3].top,
        devsel.raux[3].right-devsel.raux[3].left,
        devsel.raux[3].bottom-devsel.raux[3].top,
        hwdevice,(HMENU)DCI_EARLIER,hinst,0);
      devsel.haux[4]=CreateWindow("BUTTON","Move down",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[4].left,devsel.raux[4].top,
        devsel.raux[4].right-devsel.raux[4].left,
        devsel.raux[4].bottom-devsel.raux[4].top,
        hwdevice,(HMENU)DCI_LATER,hinst,0);
      for (i=0; i<nbook; i++) {
        Getshortbookname(name,book[i]);
        SendMessage(devsel.hlist,LB_ADDSTRING,0,(LPARAM)name); };
      if (nbook>0)
        SendMessage(devsel.hlist,LB_SETCURSEL,0,0);
      Updateshelfbuttons();
      break;
    case DEV_HERC:                     // Hercules console
      devsel.hlist=
        Createlistwindow(herclist,NULL,
        devsel.rlist.left,devsel.rlist.top,
        devsel.rlist.right-devsel.rlist.left,
        devsel.rlist.bottom-devsel.rlist.top,
        hwdevice,0,hinst,hercfont.hfont,NULL,
        LWO_KEYSCROLL|LWO_LBORDER|LWO_SNOWFREE|LWO_SUNKEN);
      devsel.hfocus=devsel.haux[0]=CreateWindow("COMBOBOX","",
        WS_CHILD|WS_VSCROLL|WS_VISIBLE|WS_TABSTOP|CBS_DROPDOWN|CBS_AUTOHSCROLL,
        devsel.raux[0].left,devsel.raux[0].top,
        devsel.raux[0].right-devsel.raux[0].left,
        devsel.raux[0].bottom-devsel.raux[0].top,
        hwdevice,(HMENU)DCI_HERCCMD,hinst,0);
      SendMessage(devsel.haux[0],CB_SETEXTENDEDUI,1,0);
      SendMessage(devsel.haux[0],CB_LIMITTEXT,TEXTLEN-1,0);
      // This is the incorrect way to obtain the handle of the edit child, but
      // it works. Official workaround with ChildWindowFromPoint() doesn't!
      hchild=GetWindow(devsel.haux[0],GW_CHILD);
      oldcomboeditproc=(WNDPROC)SetWindowLong(hchild,
        GWL_WNDPROC,(ulong)Comboeditsubclass);
      for (i=0; i<nhist; i++)
        SendMessage(devsel.haux[0],CB_ADDSTRING,0,(LPARAM)hist[i]);
      SendMessage(devsel.haux[0],CB_SETCURSEL,0,0);
      devsel.haux[1]=CreateWindow("BUTTON","Send",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_DEFPUSHBUTTON,
        devsel.raux[1].left,devsel.raux[1].top,
        devsel.raux[1].right-devsel.raux[1].left,
        devsel.raux[1].bottom-devsel.raux[1].top,
        hwdevice,(HMENU)DCI_HERSEND,hinst,0);
      SendMessage(devsel.haux[1],WM_SETFONT,(WPARAM)sysfont.hfont,1);
      break;
    case DEV_CPU:                      // CPU
      devsel.hlist=
        Createlistwindow(cpulist,NULL,
        devsel.rlist.left,devsel.rlist.top,
        devsel.rlist.right-devsel.rlist.left,
        devsel.rlist.bottom-devsel.rlist.top,
        hwdevice,0,hinst,hercfont.hfont,NULL,
        LWO_KEYSCROLL|LWO_LBORDER|LWO_SNOWFREE|LWO_SUNKEN);
      devsel.haux[0]=
        Createiplpanel(hwdevice,devsel.raux[0].left,devsel.raux[0].top);
      break;
    case DEV_RDR:                      // Card reader
      devsel.hfocus=devsel.hlist=
        Createeditwindow(pdev->edit,NULL,
        devsel.rlist.left,devsel.rlist.top,
        devsel.rlist.right-devsel.rlist.left,
        devsel.rlist.bottom-devsel.rlist.top,
        hwdevice,0,hinst,editfont.hfont,
        EDO_SUNKEN|EDO_LBORDER|EDO_SNOWFREE);
      devsel.haux[0]=CreateWindow("BUTTON","New",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[0].left,devsel.raux[0].top,
        devsel.raux[0].right-devsel.raux[0].left,
        devsel.raux[0].bottom-devsel.raux[0].top,
        hwdevice,(HMENU)DCI_NEWDECK,hinst,0);
      devsel.haux[1]=CreateWindow("BUTTON","Load",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[1].left,devsel.raux[1].top,
        devsel.raux[1].right-devsel.raux[1].left,
        devsel.raux[1].bottom-devsel.raux[1].top,
        hwdevice,(HMENU)DCI_OPENDECK,hinst,0);
      devsel.haux[2]=CreateWindow("BUTTON","Save",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[2].left,devsel.raux[2].top,
        devsel.raux[2].right-devsel.raux[2].left,
        devsel.raux[2].bottom-devsel.raux[2].top,
        hwdevice,(HMENU)DCI_SAVEDECK,hinst,0);
      devsel.haux[3]=CreateWindow("BUTTON","Card view",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[3].left,devsel.raux[3].top,
        devsel.raux[3].right-devsel.raux[3].left,
        devsel.raux[3].bottom-devsel.raux[3].top,
        hwdevice,(HMENU)DCI_CARDVIEW,hinst,0);
      devsel.haux[4]=CreateWindow("BUTTON","Submit",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[4].left,devsel.raux[4].top,
        devsel.raux[4].right-devsel.raux[4].left,
        devsel.raux[4].bottom-devsel.raux[4].top,
        hwdevice,(HMENU)DCI_SUBMIT,hinst,0);
      break;
    case DEV_PCH:                      // Card puncher
      devsel.hfocus=devsel.hlist=
        Createeditwindow(pdev->edit,NULL,
        devsel.rlist.left,devsel.rlist.top,
        devsel.rlist.right-devsel.rlist.left,
        devsel.rlist.bottom-devsel.rlist.top,
        hwdevice,0,hinst,editfont.hfont,
        EDO_SUNKEN|EDO_LBORDER|EDO_SNOWFREE);
      devsel.haux[0]=CreateWindow("BUTTON","Discard",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[0].left,devsel.raux[0].top,
        devsel.raux[0].right-devsel.raux[0].left,
        devsel.raux[0].bottom-devsel.raux[0].top,
        hwdevice,(HMENU)DCI_NEWDECK,hinst,0);
      devsel.haux[1]=CreateWindow("BUTTON","Save",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[1].left,devsel.raux[1].top,
        devsel.raux[1].right-devsel.raux[1].left,
        devsel.raux[1].bottom-devsel.raux[1].top,
        hwdevice,(HMENU)DCI_SAVEDECK,hinst,0);
      devsel.haux[2]=CreateWindow("BUTTON","Card view",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[2].left,devsel.raux[2].top,
        devsel.raux[2].right-devsel.raux[2].left,
        devsel.raux[2].bottom-devsel.raux[2].top,
        hwdevice,(HMENU)DCI_CARDVIEW,hinst,0);
      break;
    case DEV_PRT:                      // Line printer
      devsel.hfocus=devsel.hlist=
        Createlistwindow(pdev->list,NULL,
        devsel.rlist.left,devsel.rlist.top,
        devsel.rlist.right-devsel.rlist.left,
        devsel.rlist.bottom-devsel.rlist.top,
        hwdevice,0,hinst,prtfont.hfont,NULL,
        LWO_KEYSCROLL|LWO_LBORDER|LWO_SNOWFREE|LWO_SUNKEN|LWO_PERFO);
      devsel.haux[0]=CreateWindow("BUTTON","Save",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[0].left,devsel.raux[0].top,
        devsel.raux[0].right-devsel.raux[0].left,
        devsel.raux[0].bottom-devsel.raux[0].top,
        hwdevice,(HMENU)DCI_SCISSORS,hinst,0);
      devsel.haux[1]=CreateWindow("BUTTON","Discard",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[1].left,devsel.raux[1].top,
        devsel.raux[1].right-devsel.raux[1].left,
        devsel.raux[1].bottom-devsel.raux[1].top,
        hwdevice,(HMENU)DCI_TRASH,hinst,0);
      if (hidedev==0) devsel.haux[2]=CreateWindow("BUTTON","Start",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[2].left,devsel.raux[2].top,
        devsel.raux[2].right-devsel.raux[2].left,
        devsel.raux[2].bottom-devsel.raux[2].top,
        hwdevice,(HMENU)DCI_STARTP,hinst,0);
      if (hidedev==0) devsel.haux[3]=CreateWindow("BUTTON","Stop",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[3].left,devsel.raux[3].top,
        devsel.raux[3].right-devsel.raux[3].left,
        devsel.raux[3].bottom-devsel.raux[3].top,
        hwdevice,(HMENU)DCI_STOPP,hinst,0);
      break;
    case DEV_CON:                      // Console printer
      devsel.hfocus=devsel.hlist=
        Createconsolewindow(pdev->console,
        devsel.rlist.left,devsel.rlist.top,
        devsel.rlist.right-devsel.rlist.left,
        devsel.rlist.bottom-devsel.rlist.top,
        hwdevice,0,hinst,consfont.hfont,
        LWO_LBORDER|LWO_SNOWFREE|LWO_CURSOR|LWO_SUNKEN);
      devsel.haux[0]=CreateWindow("BUTTON","Save",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[0].left,devsel.raux[0].top,
        devsel.raux[0].right-devsel.raux[0].left,
        devsel.raux[0].bottom-devsel.raux[0].top,
        hwdevice,(HMENU)DCI_SCISSORS,hinst,0);
      devsel.haux[1]=CreateWindow("BUTTON","Discard",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[1].left,devsel.raux[1].top,
        devsel.raux[1].right-devsel.raux[1].left,
        devsel.raux[1].bottom-devsel.raux[1].top,
        hwdevice,(HMENU)DCI_TRASH,hinst,0);
      break;
    case DEV_DSP:                      // Display terminal
      devsel.hfocus=devsel.hlist=
        Createterminalwindow(pdev->terminal,
        devsel.rlist.left,devsel.rlist.top,
        devsel.rlist.right-devsel.rlist.left,
        devsel.rlist.bottom-devsel.rlist.top,
        hwdevice,0,hinst,dspfont.hfont,
        LWO_LBORDER|LWO_SNOWFREE|LWO_CURSOR|LWO_SUNKEN);
      break;
    case DEV_DASD:                     // Direct Access Storage
      devsel.hlist=
        Createlistwindow(pdev->list,NULL,
        devsel.rlist.left,devsel.rlist.top,
        devsel.rlist.right-devsel.rlist.left,
        devsel.rlist.bottom-devsel.rlist.top,
        hwdevice,0,hinst,hercfont.hfont,NULL,
        LWO_KEYSCROLL|LWO_LBORDER|LWO_SNOWFREE|LWO_SUNKEN);
      devsel.haux[0]=CreateWindow("BUTTON","Mount",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[0].left,devsel.raux[0].top,
        devsel.raux[0].right-devsel.raux[0].left,
        devsel.raux[0].bottom-devsel.raux[0].top,
        hwdevice,(HMENU)DCI_NEWDISK,hinst,0);
      if (hidedev==0) devsel.haux[1]=CreateWindow("BUTTON","Attn",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[1].left,devsel.raux[1].top,
        devsel.raux[1].right-devsel.raux[1].left,
        devsel.raux[1].bottom-devsel.raux[1].top,
        hwdevice,(HMENU)DCI_ATTN,hinst,0);
      break;
    case DEV_TAPE:                     // Tape
      devsel.hlist=
        Createlistwindow(pdev->list,NULL,
        devsel.rlist.left,devsel.rlist.top,
        devsel.rlist.right-devsel.rlist.left,
        devsel.rlist.bottom-devsel.rlist.top,
        hwdevice,0,hinst,hercfont.hfont,NULL,
        LWO_KEYSCROLL|LWO_LBORDER|LWO_SNOWFREE|LWO_SUNKEN);
      devsel.haux[0]=CreateWindow("BUTTON","Mount RO",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[0].left,devsel.raux[0].top,
        devsel.raux[0].right-devsel.raux[0].left,
        devsel.raux[0].bottom-devsel.raux[0].top,
        hwdevice,(HMENU)DCI_TAPERO,hinst,0);
      devsel.haux[1]=CreateWindow("BUTTON","Mount RW",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[1].left,devsel.raux[1].top,
        devsel.raux[1].right-devsel.raux[1].left,
        devsel.raux[1].bottom-devsel.raux[1].top,
        hwdevice,(HMENU)DCI_TAPERW,hinst,0);
      devsel.haux[2]=CreateWindow("BUTTON","Remove",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[2].left,devsel.raux[2].top,
        devsel.raux[2].right-devsel.raux[2].left,
        devsel.raux[2].bottom-devsel.raux[2].top,
        hwdevice,(HMENU)DCI_NOTAPE,hinst,0);
      devsel.haux[3]=CreateWindow("BUTTON","New tape",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[3].left,devsel.raux[3].top,
        devsel.raux[3].right-devsel.raux[3].left,
        devsel.raux[3].bottom-devsel.raux[3].top,
        hwdevice,(HMENU)DCI_NEWTAPE,hinst,0);
      if (hidedev==0) devsel.haux[4]=CreateWindow("BUTTON","Attn",
        WS_CHILD|WS_VISIBLE|WS_TABSTOP|BS_OWNERDRAW,
        devsel.raux[4].left,devsel.raux[4].top,
        devsel.raux[4].right-devsel.raux[4].left,
        devsel.raux[4].bottom-devsel.raux[4].top,
        hwdevice,(HMENU)DCI_ATTN,hinst,0);
      break;
    default: break;
  };
  if (devsel.hfocus!=NULL)
    SetFocus(devsel.hfocus);
  // Highlight selected device.
  for (i=0,pdev=device; pdev->devclass!=DEV_NONE; i++,pdev++)
    pdev->selected=(i==select);
  InvalidateRect(hwframe,NULL,FALSE);
  // Redraw background of device window.
  InvalidateRect(hwdevice,NULL,FALSE);
};

// Updates last selected device.
void Reselectdevice(void) {
  Selectdevice(devsel.devindex);
};

// Processes drag-and-drop message, received by main window, and standard load
// device messages. If pt is NULL, message was to the device, otherwise to the
// specified client point of mainframe. Message contains nfile filenames,
// MAXPATH characters each. Flag unprotect is 1 if tape to be mounted with
// protection ring installed.
void Processdraganddrop(POINT *pt,char *filelist,int nfile,int unprotect) {
  int i,j,select,format,result;
  char s[TEXTLEN+MAXPATH],*file;
  t_device *pdev;
  // Get and select device.
  if (pt!=NULL) {
    select=Devicefromclient(pt->x,NULL);
    if (select>=0) Selectdevice(select); };
  if (devsel.devindex<0 || devsel.devclass==DEV_NONE) {
    Message("JASN0013E  Invalid device at drag-and-drop target");
    return; };
  select=devsel.devindex;
  pdev=device+select;
  switch (pdev->devclass) {
    case DEV_SHELF:                    // Bookshelf
      if (nbook>=NBOOK)
        break;                         // Limit of books reached
      for (i=0; i<nfile; i++) {
        file=filelist+i*MAXPATH;
        for (j=0; j<nbook; j++) {
          if (stricmp(book[j],file)==0) break; };
        if (j<nbook)
          continue;                    // Duplicated title
        Strcopy(book[nbook],MAXPATH,file);
        nbook++; };
      if (select>=0)
        Selectdevice(select);
      Updateshelfbuttons();
      break;
    case DEV_RDR:                      // Card reader
      for (i=0; i<nfile; i++) {
        file=filelist+i*MAXPATH;
        if (i==0 && Iseditcontentschanged(pdev->edit)!=0)
          break;
        else
          Loadedit(pdev->edit,file,(i==0?LEO_REPLACE:LEO_ADD));
        Sleep(10);                     // Just to be sure
      };
      break;
    case DEV_DASD:                     // Direct Access Storage
      format=Verifyfileformat(filelist);
      if (format!=FF_CKD && format!=FF_FBA) {
        sprintf (s,"File '%s' is probably not a DASD image in CKD or FBA "
          "format. Load anyway?",filelist);
        result=MessageBox(hwdevice,s,"Confirm DASD image",
          MB_ICONEXCLAMATION|MB_YESNO);
        if (result!=IDYES) break; };
      Sendstdin("devinit %04X \"%s\"",pdev->cuu,filelist);
      break;
    case DEV_TAPE:                     // Tape
      format=Verifyfileformat(filelist);
      if (format!=FF_AWSTAPE && format!=FF_HETTAPE) {
        sprintf (s,"File '%s' is probably not a tape image in AWS or HET "
          "format. Load anyway?",filelist);
        result=MessageBox(hwdevice,s,"Confirm tape image",
          MB_ICONEXCLAMATION|MB_YESNO);
        if (result!=IDYES) break; };
      Sendstdin("devinit %04X \"%s\" %s",pdev->cuu,filelist,
        (unprotect?"RING":"NORING"));
      break;
    default:
      Message("JASN0014E  Device does not support drag-and-drop");
    break;
  };
};

// Windows function of the device window.
LRESULT CALLBACK Devicewp(HWND hw,UINT msg,WPARAM wp,LPARAM lp) {
  int i,n,id,setfocus,result;
  char s[TEXTLEN],path[MAXPATH],parms[TEXTLEN+MAXPATH];
  t_device *pdev;
  RECT rc;
  SIZE size;
  PAINTSTRUCT ps;
  DRAWITEMSTRUCT *di;
  HDC dc;
  HBRUSH hbr;
  switch (msg) {
    case WM_DESTROY:
      hwdevice=NULL;
      break;
    case WM_LBUTTONDOWN:
    case WM_RBUTTONDOWN:
    case WM_MBUTTONDOWN:
    case WM_SETFOCUS:
      if (devsel.hfocus!=NULL)
        SetFocus(devsel.hfocus);
      break;
    case WM_CTLCOLORBTN:
      // Prevent own-drawn buttons from filling background before WM_DRAWITEM.
      return (LRESULT)GetStockObject(NULL_BRUSH);
    case WM_DRAWITEM:
      di=(DRAWITEMSTRUCT *)lp;
      if (di->CtlType==ODT_BUTTON) {
        // Draw custom pushbutton with bitmap.
        rc=di->rcItem;
        dc=di->hDC;
        if (di->itemState & ODS_SELECTED)
          SelectObject(dc,darkpen);
        else
          SelectObject(dc,lightpen);
        MoveToEx(dc,rc.left,rc.bottom-2,NULL);
        LineTo(dc,rc.left,rc.top);
        LineTo(dc,rc.right,rc.top);
        if (di->itemState & ODS_SELECTED)
          SelectObject(dc,lightpen);
        else
          SelectObject(dc,darkpen);
        MoveToEx(dc,rc.left,rc.bottom-1,NULL);
        LineTo(dc,rc.right-1,rc.bottom-1);
        LineTo(dc,rc.right-1,rc.top-1);
        SelectObject(dc,shadowpen);
        MoveToEx(dc,rc.left+1,rc.bottom-2,NULL);
        if (di->itemState & ODS_SELECTED) {
          LineTo(dc,rc.left+1,rc.top+1);
          LineTo(dc,rc.right-1,rc.top+1);
          rc.left+=2; rc.top+=2;
          rc.right-=1; rc.bottom-=1; }
        else {
          LineTo(dc,rc.right-2,rc.bottom-2);
          LineTo(dc,rc.right-2,rc.top);
          rc.left+=1; rc.top+=1;
          rc.right-=2; rc.bottom-=2; };
        SelectObject(dc,btnpen);
        MoveToEx(dc,rc.left,rc.top,NULL);
        LineTo(dc,rc.right-1,rc.top);
        LineTo(dc,rc.right-1,rc.bottom-1);
        LineTo(dc,rc.left,rc.bottom-1);
        LineTo(dc,rc.left,rc.top);
        rc.left+=1; rc.top+=1;
        rc.right-=1; rc.bottom-=1;
        if (Drawbitmap(dc,(char *)wp,&rc,GetSysColor(COLOR_BTNFACE),0)!=0)
          FillRect(dc,&rc,btnbrush);   // Emergency solution
        SelectObject(dc,sysfont.hfont);
        n=GetWindowText(di->hwndItem,s,TEXTLEN);
        if (n>0) {
          GetTextExtentPoint32(dc,s,n,&size);
          SetTextAlign(dc,TA_TOP|TA_RIGHT);
          SetBkMode(dc,TRANSPARENT);
          if (di->itemState & ODS_DISABLED) {
            SetTextColor(dc,GetSysColor(COLOR_BTNHIGHLIGHT));
            ExtTextOut(dc,rc.right-3,
              rc.top+(rc.bottom-rc.top-size.cy)/2+1,
              ETO_CLIPPED,&rc,s,n,NULL);
            SetTextColor(dc,GetSysColor(COLOR_GRAYTEXT)); }
          else
            SetTextColor(dc,GetSysColor(COLOR_BTNTEXT));
          ExtTextOut(dc,rc.right-4,
            rc.top+(rc.bottom-rc.top-size.cy)/2,
            ETO_CLIPPED,&rc,s,n,NULL);
          ;
        };
      };
      return 1;
    case WM_ERASEBKGND:
      return 1;
    case WM_PAINT:
      dc=BeginPaint(hw,&ps);
      GetClientRect(hw,&rc);
      pdev=device+devsel.devindex;
      if (pdev->background==0)
        hbr=NULL;
      else
        hbr=CreateSolidBrush(pdev->background);
      FillRect(dc,&rc,(hbr==NULL?btnbrush:hbr));
      if (hbr!=NULL)
        DeleteObject(hbr);
      EndPaint(hw,&ps);
      break;
    case WM_COMMAND:
      setfocus=0;
      pdev=device+devsel.devindex;
      id=LOWORD(wp);
      switch (id) {
        case DCI_HERSEND:              // Hercules send button activated
          Sendandupdate();
          setfocus=1; break;
        case DCI_SCISSORS:             // Cut paper button
          if (pdev->devclass==DEV_PRT) {
            if (Savelisttofile(pdev->list,FT_TXT)==0)
            Clearlist(pdev->list); }
          else if (pdev->devclass==DEV_CON && pdev->console!=NULL) {
            if (Savelisttofile(pdev->console->list,FT_TXT)==0)
            Clearlist(pdev->console->list); };
          setfocus=1; break;
        case DCI_TRASH:                // Discard paper button
          if (pdev->devclass==DEV_PRT)
            Clearlist(pdev->list);
          else if (pdev->devclass==DEV_CON && pdev->console!=NULL)
            Clearlist(pdev->console->list);
          setfocus=1; break;
        case DCI_STARTP:               // Start printer
          Sendstdin("START %04X",pdev->cuu);
          setfocus=1; break;
        case DCI_STOPP:                // Stop printer
          Sendstdin("STOP %04X",pdev->cuu);
          setfocus=1; break;
        case DCI_NEWDECK:              // Clear contents of editor
          setfocus=1;
          if (Iseditcontentschanged(pdev->edit)!=0)
            break;
          Resetedit(pdev->edit);
          break;
        case DCI_OPENDECK:             // Load deck from file
          Loadeditfile(pdev->edit);
          setfocus=1; break;
        case DCI_SAVEDECK:             // Save deck to file
          Saveeditfile(pdev->edit,1);
          setfocus=1; break;
        case DCI_CARDVIEW:             // Toggle view
          Toggleeditview(pdev->edit);
          setfocus=1; break;
        case DCI_SUBMIT:               // Submit deck to RDR
          Submitedittodevice(pdev->edit,pdev->cuu,pdev->rdrport);
          setfocus=1; break;
        case DCI_NEWDISK:              // Mount DASD volume
          Strcopy(path,MAXPATH,pdev->last);
          result=Browsefilename("Mount new DASD volume",path,FT_DASD,
            hercdir,NULL,0);
          if (result==0)
            Processdraganddrop(NULL,path,1,0);
          setfocus=1; break;
        case DCI_TAPERO:               // Mount tape read only
          Strcopy(path,MAXPATH,pdev->last);
          result=Browsefilename("Mount new tape (read only mode)",path,FT_TAPE,
            hercdir,NULL,0);
          if (result==0)
            Processdraganddrop(NULL,path,1,0);
          setfocus=1; break;
        case DCI_TAPERW:               // Mount tape read/write
          Strcopy(path,MAXPATH,pdev->last);
          result=Browsefilename("Mount new tape (read/write mode)",path,FT_TAPE,
            hercdir,NULL,0);
          if (result==0)
            Processdraganddrop(NULL,path,1,1);
          setfocus=1; break;
        case DCI_NOTAPE:               // Remove tape
          Sendstdin("devinit %04X *",pdev->cuu);
          setfocus=1; break;
        case DCI_NEWTAPE:              // ID/bitmap of New tape
          Strcopy(path,MAXPATH,pdev->last);
          result=Browsefilename("Create new tape (file name is a label)",
            path,FT_AWS,hercdir,NULL,1);
          setfocus=1;
          if (result!=0)
            break;
          fnsplit(path,NULL,NULL,s,NULL);
          s[6]='\0';                   // Tape labels are max 6 characters long
          strupr(s);
          sprintf(parms,"-d \"%s\" %s",path,s);
          Callutility("hetinit.exe",parms,pdev->list,NULL,NULL,0);
          Processdraganddrop(NULL,path,1,1);
          break;
        case DCI_ATTN:                 // Send Attention Interrupt
          Sendstdin("I %04X",pdev->cuu);
          setfocus=1; break;
        case DCI_BOOKLIST:             // ID of book list
          if (HIWORD(wp)==LBN_SELCHANGE)
            Updateshelfbuttons();
          else if (HIWORD(wp)==LBN_DBLCLK)
            Openselectedbook();
          break;
        case DCI_ADDBOOK:              // ID/bitmap of Add book
          path[0]='\0';
          result=Browsefilename("Select new book",path,FT_BOOK,NULL,NULL,0);
          if (result==0)
            Processdraganddrop(NULL,path,1,1);
          break;
        case DCI_OPENBOOK:             // ID/bitmap of Open book
          Openselectedbook();
          break;
        case DCI_DELBOOK:              // ID/bitmap of Remove book
          i=SendMessage(devsel.hlist,LB_GETCURSEL,0,0);
          if (i==LB_ERR || i<0 || i>=nbook)
            break;
          if (i<nbook-1)
            memmove(book+i,book+i+1,sizeof(book[0])*nbook-i-1);
          SendMessage(devsel.hlist,LB_DELETESTRING,i,0);
          nbook--;
          if (i==nbook) i--;
          if (i>=0)
            SendMessage(devsel.hlist,LB_SETCURSEL,i,0);
          Updateshelfbuttons();
          break;
        case DCI_EARLIER:              // ID/bitmap of Move book up
          i=SendMessage(devsel.hlist,LB_GETCURSEL,0,0);
          if (i==LB_ERR || i<1 || i>=nbook)
            break;
          Strcopy(path,MAXPATH,book[i]);
          Strcopy(book[i],MAXPATH,book[i-1]);
          Strcopy(book[i-1],MAXPATH,path);
          SendMessage(devsel.hlist,LB_DELETESTRING,i,0);
          SendMessage(devsel.hlist,LB_DELETESTRING,i-1,0);
          Getshortbookname(path,book[i-1]);
          SendMessage(devsel.hlist,LB_INSERTSTRING,i-1,(LPARAM)path);
          Getshortbookname(path,book[i]);
          SendMessage(devsel.hlist,LB_INSERTSTRING,i,(LPARAM)path);
          SendMessage(devsel.hlist,LB_SETCURSEL,i-1,0);
          Updateshelfbuttons();
          break;
        case DCI_LATER:                // ID/bitmap of Move book down
          i=SendMessage(devsel.hlist,LB_GETCURSEL,0,0);
          if (i==LB_ERR || i<0 || i>=nbook-1)
            break;
          Strcopy(path,MAXPATH,book[i]);
          Strcopy(book[i],MAXPATH,book[i+1]);
          Strcopy(book[i+1],MAXPATH,path);
          SendMessage(devsel.hlist,LB_DELETESTRING,i+1,0);
          SendMessage(devsel.hlist,LB_DELETESTRING,i,0);
          Getshortbookname(path,book[i]);
          SendMessage(devsel.hlist,LB_INSERTSTRING,i,(LPARAM)path);
          Getshortbookname(path,book[i+1]);
          SendMessage(devsel.hlist,LB_INSERTSTRING,i+1,(LPARAM)path);
          SendMessage(devsel.hlist,LB_SETCURSEL,i+1,0);
          Updateshelfbuttons();
          break;
        default: break;
      };
      if (setfocus!=0 && devsel.hfocus!=NULL)
        SetFocus(devsel.hfocus);
      break;
    default: return DefWindowProc(hw,msg,wp,lp);
  };
  return 0L;
};

// Creates device child in the bottom part of the main window which displays
// currently selected device. This function is called once during startup.
HWND Createdevicewindow(void) {
  int i,dy;
  char key[SHORTNAME];
  RECT rc;
  WNDCLASS wc;
  if (hwdevice!=NULL)                  // Window already exists
    return hwdevice;
  // Register class of device child.
  if (GetClassInfo(hinst,DEVICECLASS,&wc)==0) {
    wc.style=CS_OWNDC|CS_HREDRAW;
    wc.lpfnWndProc=Devicewp;
    wc.cbClsExtra=wc.cbWndExtra=0;
    wc.hInstance=hinst;
    wc.hIcon=NULL;
    wc.hCursor=LoadCursor(NULL,IDC_ARROW);
    wc.hbrBackground=NULL;
    wc.lpszMenuName=NULL;
    wc.lpszClassName=DEVICECLASS;
    if (!RegisterClass(&wc)) return NULL; };
  // Create device child.
  GetClientRect(hwmain,&rc);
  dy=max((long)DEVICEDY,(long)(rc.bottom-FRAMEDY-3*BORDER));
  hwdevice=CreateWindowEx(
    WS_EX_CLIENTEDGE,DEVICECLASS,"",
    WS_CHILD|WS_VISIBLE|WS_CLIPCHILDREN,
    BORDER,FRAMEDY+2*BORDER,rc.right-2*BORDER,dy,
    hwmain,(HMENU)0,hinst,NULL);
  // Get command history from the initialization file.
  nhist=0;
  for (i=0; i<NHIST; i++) {
    sprintf(key,"Command[%i]",i);
    if (Stringfromini("Command history",key,hist[nhist],TEXTLEN)>0)
      nhist++;
    ;
  };
  return hwdevice;
};

// Frees resources associated with device.
void Deletedevice(t_device *pdev) {
  char key[SHORTNAME];
  if (pdev==NULL)
    return;                            // Error in input parameters
  if (pdev->listhw!=NULL) {
    DestroyWindow(pdev->listhw);
    pdev->listhw=NULL; };
  if (pdev->statehw!=NULL) {
    DestroyWindow(pdev->statehw);
    pdev->statehw=NULL; };
  if (pdev->pipe!=NULL) {
    Closepipe(pdev->pipe);
    GlobalFree((HGLOBAL)pdev->pipe);
    pdev->pipe=NULL; };
  if ((pdev->devclass==DEV_DASD || pdev->devclass==DEV_TAPE) &&
    pdev->last[0]!='\0'
  ) {
    sprintf(key,"Device[%04X]",pdev->cuu);
    Writetoini("Filenames",key,"%s",pdev->last); }
  else if (pdev->list!=NULL && pdev->list->filename[0]!='\0') {
    sprintf(key,"Device[%04X]",pdev->cuu);
    Writetoini("Filenames",key,"%s",pdev->list->filename); }
  else if (pdev->edit!=NULL && pdev->edit->filename[0]!='\0') {
    sprintf(key,"Device[%04X]",pdev->cuu);
    Writetoini("Filenames",key,"%s",pdev->edit->filename); };
  if (pdev->list!=NULL) {
    Destroylist(pdev->list);
    pdev->list=NULL; };
  if (pdev->edit!=NULL) {
    Destroyedit(pdev->edit);
    pdev->edit=NULL; };
  if (pdev->console!=NULL) {
    Deleteconsole(pdev->console);
    pdev->console=NULL; };
  if (pdev->terminal!=NULL) {
    Deleteterminal(pdev->terminal);
    pdev->terminal=NULL;
  };
};

// Forces redraw of CPU panel. Call it when CPU registers or PSW change, makes
// fun!
void Redrawcpupanel(void) {
  if (device[2].listhw!=NULL) InvalidateRect(device[2].listhw,NULL,FALSE);
};

// Creates, updates or deletes devices to be displayed in the mainframe window.
void Updatedevices(void) {
  int i,j;
  char s[TEXTLEN],*pt,errmsg[TEXTLEN];
  t_device *pdev;
  t_devimg *pimg,*pbest;
  // Check whether some device is removed from the list.
  for (i=NDEVRESERVED,pdev=device+NDEVRESERVED; i<ndevice; i++,pdev++) {
    if (pdev->devclass==DEV_NONE)
      continue;                        // Placeholder
    if (pdev->changed & DCH_DELETED) {
      Deletedevice(pdev);
      if (i<ndevice-1)
        memmove(pdev,pdev+1,(ndevice-i-1)*sizeof(t_device));
      memset(device+ndevice-1,0,sizeof(t_device));
      i--; pdev--; ndevice--;
    };
  };
  // Process new devices.
  for (i=0,pdev=device; i<ndevice; i++,pdev++) {
    if (pdev->devclass==DEV_NONE)
      continue;                        // Placeholder
    if (pdev->bitmap[0]=='\0') {
      // Find appropriate device. The algorithm is simple: I take full class/
      // type match. If absent, I take the last class match. If absent, I take
      // UNKNOWN device image (last in the list).
      pbest=NULL;
      for (j=0,pimg=devimg; j<sizeof(devimg)/sizeof(devimg[0]); j++,pimg++) {
        if (pdev->devclass!=pimg->devclass)
          continue;
        pbest=pimg;
        if (pdev->devtype==pimg->devtype)
          break;
        ; };
      if (pbest==NULL) pbest=pimg-1;
      Strcopy(pdev->bitmap,SHORTNAME,pbest->bitmap);
      if (pbest->protmap!=NULL)
        Strcopy(pdev->protmap,SHORTNAME,pbest->protmap);
      else
        pdev->protmap[0]='\0';
      if (pbest->unprmap!=NULL)
        Strcopy(pdev->unprmap,SHORTNAME,pbest->unprmap);
      else
        pdev->unprmap[0]='\0';
      // Set parameters for device in the mainframe window.
      Getbitmapsize(pdev->bitmap,&(pdev->bitmapsize));
      pdev->listrc.left=pbest->listx0;
      pdev->listrc.top=pbest->listy0;
      pdev->listrc.right=pbest->listx0+pbest->listdx;
      pdev->listrc.bottom=pbest->listy0+pbest->listdy;
      pdev->staterc.left=pbest->statex0;
      pdev->staterc.top=pbest->statey0;
      pdev->staterc.right=pbest->statex0+STATEDX;
      pdev->staterc.bottom=pbest->statey0+STATEDY;
      pdev->textrc.left=pbest->textx0;
      pdev->textrc.top=pbest->texty0;
      pdev->textrc.right=pbest->textx0+pbest->textdx;
      pdev->textrc.bottom=pbest->texty0+pbest->textdy;
      pdev->background=pbest->background;
      // Set device name.
      if (pdev->devname[0]=='\0') {
        switch (pdev->devclass) {
          case DEV_RDR:                // Card reader
            pt="RDR"; break;
          case DEV_PCH:                // Card puncher
            pt="PCH"; break;
          case DEV_PRT:                // Line printer
            pt="PRT"; break;
          case DEV_CON:                // Console printer
            pt="CON"; break;
          case DEV_DSP:                // Display terminal
            pt="DSP"; break;
          case DEV_DASD:               // Direct Access Storage
            pt="DASD"; break;
          case DEV_TAPE:               // Tape
            pt="TAPE"; break;
          default:
            pt=NULL;
          break; };
        if (pt==NULL)
          sprintf(pdev->devname,"%04i at %03X",pdev->devtype,pdev->cuu);
        else if (pdev->bitmapsize.right<150)
          sprintf(pdev->devname,"%s at %03X",pt,pdev->cuu);
        else
          sprintf(pdev->devname,"%s %04i at %03X",pt,pdev->devtype,pdev->cuu);
        ;
      };
    };
    if (pdev->changed & DCH_NEW) {
      // Open connection to device.
      switch (pdev->devclass) {
        case DEV_HERC:                 // Hercules console
          pdev->listhw=Createlistwindow(herclist,NULL,-10,-10,10,10,
            hwframe,0,hinst,smallfixfont.hfont,NULL,
            LWO_LBORDER|LWO_SCALE|LWO_HILITE|LWO_NOSCROLL|LWO_PASSMOUSE);
          break;
        case DEV_CPU:                  // CPU
          pdev->listhw=Createcpufrontpanel(hwframe,-10,-10,10,10);
          break;
        case DEV_RDR:                  // Card reader
          pdev->edit=Createedit(EDM_72COL,80);
          if (pdev->edit!=0) {
            sprintf(s,"Device[%04X]",pdev->cuu);
            Stringfromini("Filenames",s,pdev->edit->filename,MAXPATH); };
          pdev->rdrport=nextrdrport++;
          break;
        case DEV_PCH:                  // Card puncher
          pdev->pipe=(t_pipe *)GlobalAlloc(GPTR,sizeof(t_pipe));
          if (pdev->pipe!=NULL) {
            if (Openpipe(pdev->pipe,NULL,NULL,PIPE_ACCESS_INBOUND,1)!=0) {
              GlobalFree((HGLOBAL)pdev->pipe);
              pdev->pipe=NULL;
            };
          };
          if (pdev->pipe!=NULL) {
            pdev->edit=Createedit(EDM_72COL|EDM_NOEDIT|EDM_CARD,80);
            if (pdev->edit==NULL) {
              Closepipe(pdev->pipe);
              GlobalFree((HGLOBAL)pdev->pipe);
              pdev->pipe=NULL; }
            else {
              sprintf(s,"Device[%04X]",pdev->cuu);
              Stringfromini("Filenames",s,pdev->edit->filename,MAXPATH);
            };
          };
          if (pdev->pipe!=NULL)
            Sendstdin("devinit %04X \"%s\" ascii crlf",
            pdev->cuu,pdev->pipe->pipename);
          break;
        case DEV_PRT:                  // Line printer
          pdev->pipe=(t_pipe *)GlobalAlloc(GPTR,sizeof(t_pipe));
          if (pdev->pipe!=NULL) {
            if (Openpipe(pdev->pipe,NULL,NULL,PIPE_ACCESS_INBOUND,1)!=0) {
              GlobalFree((HGLOBAL)pdev->pipe);
              pdev->pipe=NULL;
            };
          };
          if (pdev->pipe!=NULL) {
            pdev->list=Createlist(0,132,60000,lptcolors,4);
            if (pdev->list==NULL) {
              Closepipe(pdev->pipe);
              GlobalFree((HGLOBAL)pdev->pipe);
              pdev->pipe=NULL; }
            else {
              sprintf(s,"Device[%04X]",pdev->cuu);
              Stringfromini("Filenames",s,pdev->list->filename,MAXPATH);
            };
          };
          if (pdev->pipe!=NULL) {
            Sendstdin("devinit %04X \"%s\" crlf",
              pdev->cuu,pdev->pipe->pipename);
            pdev->linecount=0;
            pdev->listhw=Createlistwindow(pdev->list,NULL,-10,-10,10,10,
              hwframe,0,hinst,smallfixfont.hfont,NULL,
              LWO_LBORDER|LWO_ROTATE|LWO_SCALE|LWO_NOSCROLL|
              LWO_PASSMOUSE|LWO_PERFO);
            ;
          };
          break;
        case DEV_CON:                  // Console printer
          sprintf(s,"%04X",pdev->cuu);
          if (nocprint==0) {
            pdev->console=Createconsole(132,NULL,cnslport,s,errmsg);
            if (pdev->console==NULL) {
              Message("JASN0025E %s",errmsg);
              break; };
            pdev->listhw=Createconsolewindow(pdev->console,-10,-10,10,10,
              hwframe,0,hinst,smallfixfont.hfont,
              LWO_LBORDER|LWO_SCALE|LWO_HILITE|LWO_NOSCROLL|
              LWO_PASSMOUSE|LWO_PERFO);
            ;
          };
          break;
        case DEV_DSP:                  // Display terminal
          sprintf(s,"%04X",pdev->cuu);
          pdev->terminal=Createterminal(4,0,NULL,cnslport,s,errmsg);
          if (pdev->terminal==NULL) {
            Message("JASN0009E %s",errmsg);
            break; };
          pdev->listhw=Createterminalwindow(pdev->terminal,-10,-10,10,10,
            hwframe,0,hinst,smallfixfont.hfont,
            LWO_LBORDER|LWO_SCALE|LWO_HILITE|LWO_NOSCROLL|LWO_PASSMOUSE);
          break;
        case DEV_DASD:                 // Direct Access Storage
          sprintf(s,"Device[%04X]",pdev->cuu);
          Stringfromini("Filenames",s,pdev->last,MAXPATH);
          pdev->list=Createlist(0,TEXTLEN-1,250,dircolors,3);
          break;
        case DEV_TAPE:                 // Tape
          sprintf(s,"Device[%04X]",pdev->cuu);
          Stringfromini("Filenames",s,pdev->last,MAXPATH);
          pdev->list=Createlist(0,TEXTLEN-1,250,dircolors,3);
          break;
        default: break;
      };
      // Create state panel.
      if (pdev->staterc.left>=0 && pdev->staterc.top>=0)
        pdev->statehw=Createstatepanel(hwframe,-10,-10,pdev->state);
      pdev->changed&=~DCH_NEW;
    };
    if (pdev->changed & DCH_STATE) {
      if (pdev->statehw!=NULL)
        Setstatepanelstate(pdev->statehw,pdev->state);
      pdev->changed&=~DCH_STATE;
    };
    if (pdev->changed & DCH_FILE) {
      switch (pdev->devclass) {
        case DEV_DASD:                 // Direct Access Storage
          Clearlist(pdev->list);
          if (nolist==0) {
            Callutility("dasdls.exe",pdev->file,pdev->list,"VOLSER=",s,1);
            Strcopy(pdev->text,SHORTNAME,s); }
          else
            pdev->text[0]='\0';
          if (pdev->file[0]!='\0' && pdev->file[0]!='*')
            Strcopy(pdev->last,MAXPATH,pdev->file);
          break;
        case DEV_TAPE:                 // Tape
          Clearlist(pdev->list);
          if (nolist==0 && pdev->file[0]!='*') {
            Callutility("tapemap.exe",pdev->file,pdev->list,"VOL1",s,1);
            Strcopy(pdev->text,7,s); };
          if (pdev->file[0]!='\0' && pdev->file[0]!='*')
            Strcopy(pdev->last,MAXPATH,pdev->file);
          break;
        default: break;
      };
      pdev->changed&=~DCH_FILE;
    };
  };
  // Redraw mainframe window.
  InvalidateRect(hwframe,NULL,FALSE);
};

// Processes device-related messages, must be caled periodically. Returns 1 if
// some actions were necessary and 0 otherwise.
int Devicestep(void) {
  int i,j,n,nline,ret;
  char s[TEXTLEN];
  t_device *pdev;
  ret=0;
  for (i=0,pdev=device; i<ndevice; i++,pdev++) {
    if (pdev->devclass==DEV_NONE)
      continue;                        // Placeholder
    if (pdev->console!=NULL)
      ret|=Consolestep(pdev->console);
    if (pdev->terminal!=NULL)
      ret|=Terminalstep(pdev->terminal);
    if (pdev->devclass==DEV_PCH && pdev->pipe!=NULL) {
      // Add all punched card to editor.
      while (1) {
        n=Readpipe(pdev->pipe,s,TEXTLEN);
        if (n<0)
          break;
        Addeditline(pdev->edit,s,n);
        ret=1;
      };
    };
    if (pdev->devclass==DEV_PRT && pdev->pipe!=NULL) {
      // Advance paper only several lines at a time. After all, Jason is for
      // fun, not productivity!
      if (pdev->devtype==3211)
        nline=6;                       // This printer is faster!
      else
        nline=3;                       // Assume all other slow
      for (j=0; j<nline; j++) {
        n=Readpipe(pdev->pipe,s,TEXTLEN);
        if (n<0)
          break;
        if (n==1 && s[0]=='\f' && executeff) {
          while (pdev->linecount!=0) {
            Addtolist(pdev->list,"",0,3,1+(pdev->linecount & 1),
              (pdev->linecount%3)==1?TF_PERFO:0);
            pdev->linecount++;
            if (pdev->linecount>=60) pdev->linecount=0;
          }; }
        else {
          Addtolist(pdev->list,s,n,3,1+(pdev->linecount & 1),
            (pdev->linecount%3)==1?TF_PERFO:0);
          pdev->linecount++;
          if (pdev->linecount>=60) pdev->linecount=0; };
        ret=1;
      };
    };
  };
  return ret;
};

// Reads expected order of devices in the mainframe window, allocates device
// list and creates default devices. Returns 0 on success and -1 on error.
int Initdevicelist(void) {
  int i,j,cuu,ncuu,cuulist[MAXDEVICE];
  char key[TEXTLEN],path[MAXPATH];
  t_device *pdev;
  if (device!=NULL)
    return 0;                          // Already allocated
  // Read user-defined sequence of devices.
  memset(cuulist,0,sizeof(cuulist));
  for (i=ncuu=NDEVRESERVED; i<MAXDEVICE; i++) {
    sprintf(key,"CUU[%i]",i);
    cuu=0;
    if (Getfromini("Device order",key,"%04X",&cuu)!=1 || cuu<0 || cuu>0xFFFF)
      continue;
    cuulist[ncuu++]=cuu;
  };
  // Allocate device list.
  maxdevice=NDEVRESERVED+max(ncuu,32); // Sound default
  device=(t_device *)malloc(maxdevice*sizeof(t_device));
  if (device==NULL)
    return -1;                         // Low memory
  // Create obligatory devices. First is bookshelf.
  pdev=device+DEVICE_HELP;
  pdev->devclass=DEV_SHELF;
  strcpy(pdev->devname,"Help");
  pdev->changed=DCH_NEW;
  for (i=nbook=0; i<NAUX; i++) {
    sprintf(key,"Book[%i]",i);
    if (Stringfromini("Books",key,path,MAXPATH)<=0)
      continue;
    for (j=0; j<nbook; j++) {
      if (stricmp(book[j],path)==0) break; };
    if (j<nbook)
      continue;                        // Duplicated title
    Strcopy(book[nbook],MAXPATH,path);
    nbook++;
  };
  // Create new Hercules console.
  pdev=device+DEVICE_CONSOLE;
  pdev->devclass=DEV_HERC;
  strcpy(pdev->devname,"Console");
  pdev->changed=DCH_NEW;
  // Create new CPU.
  pdev=device+DEVICE_CPU;
  pdev->devclass=DEV_CPU;
  if (cpumodel!=0) {
    sprintf(pdev->devname,"CPU %04i",cpumodel);
    sprintf(pdev->text,"%s/%i",archmode,cpumodel); }
  else {
    strcpy(pdev->devname,"CPU");
    strcpy(pdev->text,"CPU"); }
  pdev->changed=DCH_NEW;
  // Preallocate devices from the user-defined sequence.
  for (i=NDEVRESERVED; i<ncuu; i++)
    device[i].cuu=cuulist[i];
  ndevice=ncuu;
  return 0;
};

// Frees resources allocated by all devices. Call this function once when Jason
// quits.
void Deletealldevices(void) {
  int i,j;
  char key[SHORTNAME];
  t_device *pdev;
  // Flush device window.
  Selectdevice(-1);
  // Delete all devices and save their order to initialization file.
  Clearinisection("Device order");
  for (i=0,j=NDEVRESERVED,pdev=device; i<ndevice; i++,pdev++) {
    if (i>=NDEVRESERVED && pdev->devclass!=DEV_NONE && pdev->cuu>0) {
      sprintf(key,"CUU[%i]",j);
      Writetoini("Device order",key,"%04X",pdev->cuu);
      j++; };
    Deletedevice(pdev); };
  // Save book titles back to initialization file.
  Clearinisection("Books");
  for (i=0; i<nbook; i++) {
    sprintf(key,"Book[%i]",i);
    Writetoini("Books",key,"%s",book[i]); };
  // Save Hercules command history back to initialization file.
  Clearinisection("Command history");
  for (i=0; i<nhist; i++) {
    sprintf(key,"Command[%i]",i);
    Writetoini("Command history",key,"%s",hist[i]); };
  // Free device list.
  free(device);
  device=NULL;
  ndevice=maxdevice=0;
};

