////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//      JASON - FULLY GRAPHICAL INTERFACE TO HERCULES IBM 360+ EMULATOR       //
//                                                                            //
//                            PUNCHED DECK EDITOR                             //
//                                                                            //
//                             Start: 03.03.2010                              //
//                        Current version: 30.05.2010                         //
//                                                                            //
// Jason is free software; you can redistribute it and/or modify it under     //
// the terms of the GNU General Public License version 3, as published by the //
// Free Software Foundation.                                                  //
//                                                                            //
// Jason is distributed in the hope that it will be useful, but WITHOUT ANY   //
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS  //
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more      //
// details.                                                                   //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program. If not, see <http://www.gnu.org/licenses/>.             //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////


#include <windows.h>
#pragma hdrstop
#include <stdio.h>
#include <dir.h>
#include <commctrl.h>

#include "Jason.h"

#define MAXEDITCARD    1000000         // Max number of cards in the editor

#define EDITEXTRAX     1               // Additional pixels between characters
#define EDITEXTRAY     2               // Additional pixels between lines

#define NTAB           16              // Number of tab positions

static int       tabdefault[NTAB] = {  // Default tab positions
  8,  16, 24, 32, 40, 48, 56, 64, 72, 80, 88, 96, 104, 112, 120, 128 };
static int       tabjcl[NTAB] = {      // Tab positions in JCL mode
  11, 15, 24, 32, 40, 48, 56, 64, 72, 80, 88, 96, 104, 112, 120, 128 };
static int       tabasm[NTAB] = {      // Tab positions in Assembler mode
  9,  15, 24, 29, 34, 39, 44, 49, 54, 59, 64, 72, 80,  88,  96,  104 };
static int       tabfortran[NTAB] = {  // Tab positions in Fortran mode
  5,  6,  10, 14, 18, 22, 26, 30, 34, 38, 42, 46, 50,  54,  58,  72  };

// Only one search dialog window is allowed at once.
static FINDREPLACE  findreplace;       // Descripror of FindText() dialog


////////////////////////////////////////////////////////////////////////////////
////////////////////////////// SERVICE FUNCTIONS ///////////////////////////////

// Note that service function in this section (except for Addeditundo() and
// Editredo(), of course) do not finalize undo buffer!

// Service function, determines editor language. Returns one of LNG_xxx.
static int Getlanguage(t_edit *pe) {
  int i;
  char *pd,*px,s[TEXTLEN];
  if (pe==NULL)
    return LNG_NONE;                   // Error in input parameters
  if (editlng==LNG_NONE || editlng==LNG_ASM || editlng==LNG_FORTRAN)
    return editlng;                    // Fixed language
  if (editlng!=LNG_AUTO)
    return LNG_NONE;                   // Unknown language, assume none
  // Determine language automatically. I read at most 32 cards. If JCL card
  // contains something like "EXEC FORTxxx", I set language.
  s[pe->ncol-2]='\0';
  for (i=0,pd=pe->data; i<32 && i<pe->ncard; i++,pd+=pe->ncol) {
    if (pd[0]!='/' || pd[1]!='/')
      continue;                        // Not a JCL card
    memcpy(s,pd+2,pe->ncol-2);
    px=strstr(s," EXEC ");
    if (px==NULL)
      continue;                        // Not an EXEC card
    if (px-s>40)
      continue;                        // Most probably, comment
    px+=6;                             // Length of above search pattern
    while (*px==' ') px++;
    if (strncmp(px,"PGM=",4)==0)
      px+=4;
    else if (strncmp(px,"PROC=",5)==0)
      px+=4;
    if (strncmp(px,"IFOX",4)==0 || strncmp(px,"IEUASM",6)==0 ||
      strncmp(px,"ASM",3)==0) return LNG_ASM;
    if (strncmp(px,"FORT",4)==0)
      return LNG_FORTRAN;
    ;
  };
  return LNG_NONE;
};

// Service function, updates status bar.
static void Updatestatus(t_edit *pe) {
  char s[TEXTLEN];
  if (pe==NULL || pe->w.hw==NULL || pe->w.hstatus==NULL)
    return;
  sprintf(s,"%i:%i",pe->cursorx+1,pe->cursory+1);
  SendMessage(pe->w.hstatus,SB_SETTEXT,0,(LPARAM)s);
  if (pe->mode & EDM_NOEDIT) {
    SendMessage(pe->w.hstatus,SB_SETTEXT,1,(LPARAM)"");
    SendMessage(pe->w.hstatus,SB_SETTEXT,2,(LPARAM)""); }
  else {
    SendMessage(pe->w.hstatus,SB_SETTEXT,1,
      (LPARAM)(pe->modified?"Modified":""));
    SendMessage(pe->w.hstatus,SB_SETTEXT,2,
     (LPARAM)(pe->overwrite?"OVR":"INS"));
    ;
  };
};

// Service function, maintains undo buffer. Operation is one of EDU_xxx. Call
// it before undertaking any real changes to the data.
static void Addeditundo(t_edit *pe,int line,int op) {
  int n,itemsize;
  t_undo *prev,*next;
  if (pe->stopundo)
    return;                            // Undo in progress
  if (pe==NULL || pe->undo==NULL)
    return;                            // Error in input parameters
  // Calculate size of undo element.
  itemsize=sizeof(t_undo)+pe->ncol;
  // Get address of the last previous undo action.
  if (pe->nundo==0)
    prev=NULL;
  else
    prev=(t_undo *)(pe->undo+itemsize*(pe->nundo-1));
  // If operation is a finalization request and we are not in the undo phase,
  // finalize previous undo line.
  if (op==EDU_FINALIZE) {
    if (prev!=NULL && pe->nundo==pe->nredo)
      prev->op|=EDU_FINALIZE;
    return; };
  // If both new and old operations are replacements on the same line with
  // autofinalization and previous line is not yet finalized, merge actions
  // together, that is, do nothing.
  if (prev!=NULL && prev->line==line && prev->op==EDU_REPF && op==EDU_REPF) {
    pe->nredo=pe->nundo;
    return; };
  // If previous action was EDU_REPF (autofinalization), finalize it.
  if (prev!=NULL && (prev->op & EDU_OPMASK)==EDU_REPF)
    prev->op|=EDU_FINALIZE;
  // Assure that we have room in the undo buffer.
  if (pe->nundo>=MAXUNDO) {
    n=MAXUNDO/16;
    memmove(pe->undo,pe->undo+itemsize*n,itemsize*(pe->nundo-n));
    pe->nundo-=n; };
  // Add undo action.
  next=(t_undo *)(pe->undo+itemsize*pe->nundo);
  next->line=line;
  next->op=op;
  next->cursorx=pe->cursorx;
  next->cursory=pe->cursory;
  next->selx0=pe->selx0;
  next->sely0=pe->sely0;
  next->selx1=pe->selx1;
  next->sely1=pe->sely1;
  if ((op & EDU_OPMASK)==EDU_DEL || (op & EDU_OPMASK)==EDU_REPL ||
    (op & EDU_OPMASK)==EDU_REPF)
    memcpy(next->data,pe->data+line*pe->ncol,pe->ncol);
  pe->nundo++;
  pe->nredo=pe->nundo;
};

// Service function, extends size of edit buffer so that it can keep at least
// n more lines. In fcat, I at least double its size. Returns 0 on success and
// -1 on error.
static int Extendeditbuffer(t_edit *pe,int n) {
  int newmaxcard;
  char *newdata;
  if (pe==NULL || pe->data==NULL || n>MAXEDITCARD)
    return -1;                         // Error in input parameters
  // Allocate new memory block.
  if (n<pe->maxcard)
    newmaxcard=pe->maxcard*2;
  else
    newmaxcard=pe->maxcard+n;
  if (newmaxcard>MAXEDITCARD)
    newmaxcard=MAXEDITCARD;
  if (newmaxcard-pe->maxcard<n)
    return -1;                         // Not this simple editor!
  newdata=(char *)malloc(pe->ncol*newmaxcard);
  if (newdata==NULL)
    return -1;                         // Low memory
  // Copy data to the new block and free the old one.
  memcpy(newdata,pe->data,pe->ncol*pe->ncard);
  free(pe->data);
  pe->data=newdata;
  pe->maxcard=newmaxcard;
  // Report success.
  return 0;
};

// Service function, sets cursor (correcting its coordinates), creates and sets
// caret, and optionally scrolls the window so that cursor is visible. Returns
// 1 if cursor was moved in the data and 0 if not.
static int Seteditcaretpos(t_edit *pe,int cursorx,int cursory,int scroll) {
  int x,fw,fh,nx,ny,dx,redraw,moved;
  RECT rc;
  if (pe==NULL || pe->w.hw==NULL || pe->w.hedit==NULL)
    return 0;                          // Error in parameters or no window
  // Correct cursor position.
  if (cursorx>pe->ncol) cursorx=pe->ncol;
  if (cursorx<0) cursorx=0;
  if (cursory>pe->ncard) cursory=pe->ncard;
  if (cursory<0) cursory=0;
  // Get window size in characters.
  GetClientRect(pe->w.hedit,&rc);
  fw=max(1,pe->w.fontdx); nx=rc.right/fw;
  if (pe->mode & EDM_CARD)
    ny=1;
  else {
    fh=max(1,pe->w.fontdy); ny=rc.bottom/fh; };
  // Scroll screen to make cursor visible, even if cursor was not moved.
  if (scroll) {
    redraw=0;
    if (cursorx<pe->w.offsetx) {
      pe->w.offsetx=cursorx; redraw=1; }
    else if (cursorx>=pe->w.offsetx+nx) {
      pe->w.offsetx=cursorx-nx+1; redraw=1; };
    if (cursory<pe->w.offsety) {
      pe->w.offsety=cursory; redraw=1; }
    else if (cursory>=pe->w.offsety+ny) {
      pe->w.offsety=cursory-ny+1; redraw=1; };
    if (redraw!=0 && pe->w.invalidated==0) {
      pe->w.invalidated=1;
      InvalidateRect(pe->w.hedit,NULL,FALSE);
    };
  };
  // Check whether cursor was moved.
  moved=(pe->cursorx!=cursorx || pe->cursory!=cursory);
  // Set new cursor.
  pe->cursorx=cursorx; pe->cursory=cursory;
  // Create and set caret.
  if (pe->mode & EDM_CARD)
    Setcardcaret(pe->w.hedit,&(pe->cardpos),cursorx,pe->overwrite);
  else {
    if (pe->overwrite==0)
      dx=max(GetSystemMetrics(SM_CXBORDER),fw/4);
    else
      dx=fw;
    if (GetFocus()==pe->w.hedit)
      CreateCaret(pe->w.hedit,NULL,dx,fh-1-GetSystemMetrics(SM_CYBORDER));
    x=-pe->w.offsetx*fw;
    if (pe->w.options & EDO_LBORDER) x+=fw/2;
    SetCaretPos(x+pe->cursorx*fw-(pe->overwrite?0:dx/2),
      (pe->cursory-pe->w.offsety)*fh+1);
    ;
  };
  if (GetFocus()==pe->w.hedit)
    ShowCaret(pe->w.hedit);
  if (moved) Updatestatus(pe);
  // Report whether cursor was moved.
  return moved;
};

// Service function, positions cursor at the end of the line (ignoring trailing
// spaces and, for the first time, the numeration). On success, sets position
// and returns 0. On error, returns -1.
static int Toendofeditline(t_edit *pe,int *posx,int *posy) {
  int i,x,y,last;
  char *pd;
  if (pe==NULL || pe->data==NULL || posx==NULL || posy==NULL)
    return -1;                         // Error in input parameters
  x=*posx; y=*posy;
  if (x<0 || x>=pe->ncol || y<0 || y>=pe->ncard)
    return -1;                         // Position outside the deck
  // Determine end of data area.
  if ((pe->mode & EDM_72COL)==0 || pe->ncol<20 || x>=pe->ncol-8)
    last=pe->ncol;
  else
    last=pe->ncol-8;
  // Find last nonspace symbol in the data area.
  pd=pe->data+y*pe->ncol;
  for (i=last; i>x; i--) {
    if (pd[i-1]!=' ') break; };
  if (i==x && pe->ncol>last) {
    // Empty data area, check for numeration.
    for (i=pe->ncol; i>last; i--) {
      if (pd[i-1]!=' ') break; };
    if (i==last) i=x; };
  *posx=i;
  return 0;
};

// Service function, inserts n empty lines at the specified position (shifting
// this and following cards down). Returns 0 on success and -1 on error.
// Position of cursor is not corrected, window is not scrolled. Attention,
// does not finalize undo buffer!
static int Inserteditlines(t_edit *pe,int posy,int n) {
  int i;
  if (pe==NULL || pe->data==NULL || posy<0 || posy>pe->ncard || n<=0)
    return -1;                         // Error in input parameters
  // Assure that we have enough free space.
  if (pe->maxcard-pe->ncard<n)
    Extendeditbuffer(pe,pe->maxcard-pe->ncard);
  if (pe->maxcard-pe->ncard<n)
    return -1;                         // Low memory
  // Add this action to undo buffer.
  for (i=0; i<n; i++)
    Addeditundo(pe,posy+i,EDU_INS);
  // Make free space and fill it with... er... spaces.
  if (posy<pe->ncard) memmove(pe->data+(posy+n)*pe->ncol,
    pe->data+posy*pe->ncol,(pe->ncard-posy)*pe->ncol);
  memset(pe->data+posy*pe->ncol,' ',n*pe->ncol);
  pe->ncard+=n;
  // Update selection.
  if (pe->sely0>posy) pe->sely0+=n;
  if (pe->sely1>posy) pe->sely1+=n;
  if (pe->selyb>posy) pe->selyb+=n;
  pe->modified=1;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  Updatestatus(pe);
  // Report success.
  return 0;
};

// Service function, renumerates editor cards in the last 8 columns of the
// deck. Returns 0 on success and -1 on error.
static int Renumerateeditlines(t_edit *pe) {
  int i;
  ulong u,step;
  char s[32],*pd;
  if (pe==NULL || pe->data==NULL || pe->ncol<20)
    return -1;                         // Error in input parameters
  // Determine step.
  if (pe->ncard<9999)
    step=10000;
  else if (pe->ncard<99999)
    step=1000;
  else if (pe->ncard<999999)
    step=100;
  else
    step=10;
  // Renumerate cards.
  pd=pe->data+pe->ncol-8;
  for (i=0,u=step; i<pe->ncard; i++,u+=step,pd+=pe->ncol) {
    sprintf(s,"%08lu",u);
    if (memcmp(pd,s,8)!=0) {
      Addeditundo(pe,i,EDU_REPL);      // Poor undo buffer!...
      memcpy(pd,s,8);
    };
  };
  pe->modified=1;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  Updatestatus(pe);
  // Report success.
  return 0;
};

// Service function, deletes n lines starting at the specified position.
// Returns 0 on success and -1 on error. Position of cursor is not corrected,
// window is not scrolled.
static int Deleteeditlines(t_edit *pe,int posy,int n) {
  int i;
  if (pe==NULL || pe->data==NULL || posy<0 || posy>=pe->ncard || n<=0)
    return -1;                         // Error in input parameters
  // Add this action to undo buffer. As the lines are not yet physically
  // deleted, I process them in the reverse order.
  for (i=n-1; i>=0; i--) {
    if (posy+i>=pe->ncard) continue;
    Addeditundo(pe,posy+i,EDU_DEL); };
  // Delete cards.
  if (posy+n>=pe->ncard)
    pe->ncard=posy;
  else {
    memmove(pe->data+posy*pe->ncol,
      pe->data+(posy+n)*pe->ncol,(pe->ncard-posy-n)*pe->ncol);
    pe->ncard-=n; };
  // Update selection.
  if (pe->sely0>=posy+n)
    pe->sely0-=n;
  else if (pe->sely0>=posy) {
    pe->selx0=0; pe->sely0=posy; };
  if (pe->sely1>=posy+n)
    pe->sely1-=n;
  else if (pe->sely1>=posy) {
    pe->selx1=0; pe->sely1=posy; };
  if (pe->selyb>=posy+n)
    pe->selyb-=n;
  else if (pe->selyb>=posy) {
    pe->selxb=0; pe->selyb=posy; };
  pe->modified=1;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  Updatestatus(pe);
  // Report success.
  return 0;
};

// Service function, splits line at the specified position. Returns 0 on
// success and -1 on error. Position of cursor is not corrected, window is not
// scrolled.
static int Spliteditline(t_edit *pe,int posx,int posy) {
  int n,last;
  char *pd;
  if (pe==NULL || pe->data==NULL)
    return -1;                         // Error in input parameters
  if (posx<0 || posx>pe->ncol || posy<0 || posy>pe->ncard)
    return -1;                         // Position outside the deck
  // Insert empty line below.
  if (Inserteditlines(pe,posy+1,1)!=0)
    return -1;
  // Determine data to move.
  if ((pe->mode & EDM_72COL)==0 || pe->ncol<20 || posx==0 || posx>=pe->ncol-8)
    last=pe->ncol;
  else
    last=pe->ncol-8;
  n=last-posx;
  pd=pe->data+posy*pe->ncol;
  if (n>0) {
    Addeditundo(pe,posy,EDU_REPL);
    Addeditundo(pe,posy+1,EDU_REPL);     // Strictly speaking, unnecessary
    memcpy(pd+pe->ncol,pd+posx,n);
    memset(pd+posx,' ',n); };
  pe->modified=1;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  Updatestatus(pe);
  // Report success.
  return 0;
};

// Service function, merges lines posy-1 and posy. Returns merge position in
// the line posy-1 on success and -1 on error. Position of cursor itself is not
// corrected, window is not scrolled.  Attention, does not finalize undo buffer!
static int Mergeeditlines(t_edit *pe,int posy) {
  int i,last,nprev,ncurr,nmove;
  char *pd;
  if (pe==NULL || pe->data==NULL || posy<1 || posy>pe->ncard)
    return -1;                         // Error in input parameters
  // Determine end of the moveable data on the cards.
  if ((pe->mode & EDM_72COL)==0 || pe->ncol<20)
    last=pe->ncol;
  else
    last=pe->ncol-8;
  pd=pe->data+(posy-1)*pe->ncol;       // Start of previous card
  // Determine end of data on the previous card.
  for (nprev=last; nprev>0; nprev--) {
    if (pd[nprev-1]!=' ') break; };
  nmove=last-nprev;
  if (nmove==0) return -1;             // No room on the previous card
  // Add current state to undo buffer.
  Addeditundo(pe,posy-1,EDU_REPL);
  Addeditundo(pe,posy,EDU_REPL);
  // Determine size of data on the current card.
  for (ncurr=last; ncurr>0; ncurr--) {
    if (pd[pe->ncol+ncurr-1]!=' ') break; };
  if (nmove>ncurr) nmove=ncurr;
  // Move data. Note that rest of the previous card is already filled with
  // spaces.
  if (nmove>0) {
    memcpy(pd+nprev,pd+pe->ncol,nmove);
    if (ncurr>nmove) memmove(pd+pe->ncol,pd+pe->ncol+nmove,ncurr-nmove);
    memset(pd+pe->ncol+ncurr-nmove,' ',nmove); };
  // Update selection.
  if (pe->sely0==posy-1 && pe->sely1==posy-1) {
    if (pe->selx0>nprev) pe->selx0=nprev;
    if (pe->selx1>nprev) pe->selx1=nprev; }
  else if (pe->sely0==posy && pe->sely1==posy) {
    if (pe->selx0<=nmove && pe->selx1<=nmove) {
      pe->selx0+=nprev; pe->sely0--;
      pe->selx1+=nprev; pe->sely1--; }
    else if (pe->selx0>=nmove && pe->selx1>=nmove) {
      pe->selx0-=nmove;
      pe->selx1-=nmove; }
    else {
      pe->selx0=nprev; pe->sely0=posy-1;
      pe->selx1=nprev; pe->sely1=posy-1;
    };
  };
  // If current line is empty, move card number (if none on the previous) and
  // delete current card.
  if (nmove==ncurr) {
    for (i=last; i<pe->ncol; i++) {
      if (pd[i]!=' ') break; };
    if (i==pe->ncol && pe->ncol-last>0)
      memcpy(pd+last,pd+pe->ncol+last,pe->ncol-last);
    Deleteeditlines(pe,posy,1); };
  pe->modified=1;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  Updatestatus(pe);
  // Report success.
  return nprev;
};

// Service function, deletes n characters on a single card, starting at the
// specified position. Returns 0 on success and -1 on error. Does not move
// cursor. Attention, does not finalize undo buffer!
static int Deleteeditchars(t_edit *pe,int n,int posx,int posy) {
  int last;
  char *pd;
  if (pe==NULL || pe->data==NULL || n<=0)
    return -1;                         // Error in input parameters
  if (posx<0 || posx>=pe->ncol || posy<0 || posy>=pe->ncard)
    return -1;                         // Position outside the deck
  // Add current state to undo buffer.
  Addeditundo(pe,posy,EDU_REPL);
  // Determine data to move.
  if ((pe->mode & EDM_72COL)==0 || pe->ncol<20 || posx>=pe->ncol-8)
    last=pe->ncol;
  else
    last=pe->ncol-8;
  if (posx+n>last)
    n=last-posx;
  // Move data.
  pd=pe->data+posy*pe->ncol;
  if (last>posx-n)
    memmove(pd+posx,pd+posx+n,last-posx-n);
  memset(pd+last-n,' ',n);
  // Update selection.
  if (pe->sely0==posy) {
    if (pe->selx0>posx+n && pe->selx0<last) pe->selx0-=n;
    else if (pe->selx0>posx && pe->selx0<=posx+n) pe->selx0=posx; };
  if (pe->sely1==posy) {
    if (pe->selx1>posx+n && pe->selx1<last) pe->selx1-=n;
    else if (pe->selx1>posx && pe->selx1<=posx+n) pe->selx1=posx; };
  if (pe->selyb==posy) {
    if (pe->selxb>posx+n && pe->selxb<last) pe->selxb-=n;
    else if (pe->selxb>posx && pe->selxb<=posx+n) pe->selxb=posx; };
  pe->modified=1;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  Updatestatus(pe);
  // Report success.
  return 0;
};

// Service function, inserts n characters on a single card, taking into account
// the overwrite mode. In insert mode, characters that wrap outside the card
// are discarded. If force80 mode is activated, always overwrites columns 72-
// 80. Does not move cursor. Returns number of really insetred characters on
// success and -1 on error.
static int Inserteditchars(t_edit *pe,char *data,int n,int posx,int posy,
  int overwrite,int force80) {
  int last;
  char *pd;
  if (pe==NULL || pe->data==NULL || data==NULL || n<=0)
    return -1;                         // Error in input parameters
  if (posx<0 || posx>=pe->ncol || posy<0 || posy>pe->ncard)
    return -1;                         // Position outside the deck
  // If position is below the last line, insert line.
  if (posy==pe->ncard && Inserteditlines(pe,posy,1)!=0)
    return -1;
  // Add current state to undo buffer.
  Addeditundo(pe,posy,(n==1?EDU_REPF:EDU_REPL));
  // Determine data to move.
  if ((pe->mode & EDM_72COL)==0 || pe->ncol<20 || posx>=pe->ncol-8 || force80)
    last=pe->ncol;
  else
    last=pe->ncol-8;
  if (posx+n>last)
    n=last-posx;
  // Move data.
  pd=pe->data+posy*pe->ncol;
  if (overwrite==0 && last>posx-n)
    memmove(pd+posx+n,pd+posx,last-posx-n);
  memcpy(pd+posx,data,n);
  // Update selection.
  if (overwrite==0) {   // ???
    if (pe->sely0==posy && pe->selx0>=posx && pe->selx0<last) pe->selx0+=n;
    if (pe->sely1==posy && pe->selx1>=posx && pe->selx1<last) pe->selx1+=n;
    if (pe->selyb==posy && pe->selxb>=posx && pe->selxb<last) pe->selxb+=n; };
  pe->modified=1;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  Updatestatus(pe);
  // Report success.
  return n;
};

// Copies text of size n characters (terminal zero is not included and is not
// necessary) to the clipboard. Returns 0 on success and -1 on error.
static int Texttoclipboard(char *text,int n) {
  char *pt;
  HGLOBAL hmem;
  if (text==NULL || n<=0)
    return -1;                         // Error in input parameters
  if (OpenClipboard(hwmain)==0)
    return -1;                         // This is a bit unexpected
  hmem=GlobalAlloc(GMEM_MOVEABLE|GMEM_DDESHARE,n+1);
  if (hmem==NULL) {
    CloseClipboard();                  // Low memory
    return -1; };
  pt=(char *)GlobalLock(hmem);
  if (pt==NULL) {
    GlobalFree(hmem);                  // Low memory?
    CloseClipboard();
    return -1; };
  EmptyClipboard();
  memcpy(pt,text,n);
  pt[n]='\0';
  GlobalUnlock(hmem);
  SetClipboardData(CF_TEXT,hmem);
  CloseClipboard();
  return 0;
};

// Reads text from clipboard to the memory allocated by malloc(). Returns
// pointer to memory on success and NULL on error. Calling function must free()
// the memory.
static char *Getclipboardtext(void) {
  int n,nmax;
  char *pt,*pm;
  HGLOBAL hmem;
  if (OpenClipboard(hwmain)==0)
    return NULL;                       // This is a bit unexpected
  hmem=GetClipboardData(CF_TEXT);
  if (hmem==NULL) {
    CloseClipboard();                  // Clipboard contains no text
    return NULL; };
  pt=(char *)GlobalLock(hmem);
  if (pt==NULL) {
    CloseClipboard();                  // Low memory?
    return NULL; };
  nmax=GlobalSize(hmem);
  n=Strlength(pt,nmax);
  pm=(char *)malloc(n+1);
  if (pm!=NULL)
    Strcopy(pm,n+1,pt);
  GlobalUnlock(hmem);
  CloseClipboard();
  return pm;
};

// Service function, copies edit selection to clipboard. If one or several
// complete cards are selected, they are stripped of the trailing spaces and
// newline is added at the end of each line. Returns 0 on success and -1 on
// error.
static int Copyedittoclipboard(t_edit *pe) {
  int i,length;
  char *pd,*pm;
  if (pe==NULL || pe->data==NULL)
    return -1;                         // Error in input parameters
  pd=pe->data+pe->sely0*pe->ncol;
  if (pe->sely0==pe->sely1) {
    // Copy piece of text.
    if (pe->selx0==pe->selx1)
      return -1;                       // Nothing to copy
    Texttoclipboard(pd+pe->selx0,pe->selx1-pe->selx0); }
  else if (pe->sely0<pe->sely1) {
    // Copy cards.
    pm=(char *)malloc((pe->sely1-pe->sely0)*(pe->ncol+2));
    if (pm==NULL)
      return -1;                       // Low memory
    length=0;
    for (i=pe->sely0; i<pe->sely1; i++,pd+=pe->ncol) {
      memcpy(pm+length,pd,pe->ncol);
      length+=pe->ncol;
      while (length>0 && pm[length-1]==' ') length--;
      pm[length++]='\r';
      pm[length++]='\n'; };
    Texttoclipboard(pm,length);
    free(pm);
  };
  return 0;
};

// Service function, pastes text from clipboard at the specified position. If
// clipboard contains several lines, text is pasted at posx=0. On success, sets
// selection, updates posx and posy to new cursor coortdinates and returns 0.
// On error, returns -1.
static int Pasteeditfromclipboard(t_edit *pe,int *posx,int *posy) {
  int i,n,newlines,length,ret;
  char *pm,*pd;
  if (pe==NULL || pe->data==NULL || posx==NULL || posy==NULL)
    return -1;                         // Error in input parameters
  if (*posx<0 || *posx>=pe->ncol || *posy<0 || *posy>pe->ncard)
    return -1;                         // Position outside the deck
  // Get clipboard data.
  pm=Getclipboardtext();
  if (pm==NULL)
    return -1;                         // Clipboard contains no text
  length=strlen(pm);
  if (length==0) {
    free(pm); return 0; };             // Clipboard contains empty text
  // Get number of lines (zero means piece of text).
  for (i=0,newlines=0; i<length; i++) {
    if (pm[i]=='\n' && pm[i+1]=='\r') {
      newlines++; i++; }
    else if (pm[i]=='\r' && pm[i+1]=='\n') {
      newlines++; i++; }
    else if (pm[i]=='\r')
      newlines++;
    else if (pm[i]=='\n')
      newlines++;
    ;
  };
  if (newlines>0 && pm[length-1]!='\n' && pm[length-1]!='\r')
    newlines++;
  if (newlines==0) {
    // Paste piece of text.
    n=Inserteditchars(pe,pm,length,*posx,*posy,0,0);
    if (n<0)
      ret=-1;
    else {
      pe->selx0=pe->selxb=*posx; pe->sely0=pe->selyb=*posy;
      pe->selx1=*posx+n; pe->sely1=*posy;
      *posx+=n;
    }; }
  else {
    // Paste cards.
    if (Inserteditlines(pe,*posy,newlines)!=0)
      ret=-1;
    else {
      pe->selx0=pe->selxb=0; pe->sely0=pe->selyb=*posy;
      pe->selx1=0; pe->sely1=*posy+newlines;
      pd=pe->data+(*posy)*pe->ncol; n=0;
      for (i=0; i<length; i++) {
        if (pm[i]!='\n' && pm[i]!='\r') {
          if (n<pe->ncol) pd[n++]=pm[i]; }
        else {
          if ((pm[i]=='\n' && pm[i+1]=='\r') || (pm[i]=='\r' && pm[i+1]=='\n'))
            i++;
          pd+=pe->ncol; n=0;
        };
      };
      ret=0;
    };
  };
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  free(pm);
  return ret;
};

// Service function, deletes selection. On success, saves new cursor position
// to (posx,posy) and returns 0. On error, returns -1.
static int Deleteeditselection(t_edit *pe,int *posx,int *posy) {
  if (pe==NULL || pe->data==NULL)
    return -1;                         // Error in input parameters
  if (pe->sely0==pe->sely1 && pe->selx0<pe->selx1) {
    Deleteeditchars(pe,pe->selx1-pe->selx0,pe->selx0,pe->sely0);
    if (posx!=NULL) *posx=pe->selx0;
    if (posy!=NULL) *posy=pe->sely0;
    return 0; }
  else if (pe->sely0<pe->sely1) {
    Deleteeditlines(pe,pe->sely0,pe->sely1-pe->sely0);
    if (posx!=NULL) *posx=0;
    if (posy!=NULL) *posy=pe->sely0;
    return 0; }
  else return -1;
};

// Service function, removes selection. Data remains unchanged. Returns 0 on
// success and -1 on error.
static int Removeeditselection(t_edit *pe) {
  if (pe==NULL || pe->data==NULL)
    return -1;                         // Error in input parameters
  Addeditundo(pe,0,EDU_REPL);          // Line does not matter
  pe->selx0=pe->selx1=pe->selxb;
  pe->sely0=pe->sely1=pe->selyb;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  // Report success.
  return 0;
};

// Service function, copies (copy=1) or moves (copy=0) selection to the
// specified position. On success, sets selection, updates posx and posy to new
// cursor coortdinates and returns 0. On error, returns -1.
static int Copymoveeditselection(t_edit *pe,int *posx,int *posy,int copy) {
  int n,y;
  char s[TEXTLEN],*pb;
  if (pe==NULL || pe->data==NULL || posx==NULL || posy==NULL)
    return -1;                         // Error in input parameters
  if (*posx<0 || *posx>=pe->ncol || *posy<0 || *posy>pe->ncard)
    return -1;                         // Position outside the deck
  if (pe->sely0==pe->sely1) {
    // Piece of text is selected.
    if (pe->selx0>=pe->selx1)
      return -1;                       // No selection
    if (copy==0 && *posy==pe->sely0 && *posx>=pe->sely0 && *posx<=pe->sely1)
      return 0;                        // Move to itself
    n=pe->selx1-pe->selx0;
    memcpy(s,pe->data+pe->sely0*pe->ncol+pe->selx0,n);
    if (copy==0) {
      if (pe->selx0==*posy) Deleteeditselection(pe,posx,posy);
      else Deleteeditselection(pe,NULL,NULL); };
    Inserteditchars(pe,s,n,*posx,*posy,0,0);
    pe->selx0=pe->selxb=*posx; pe->sely0=pe->selyb=*posy;
    pe->selx1=*posx+n; pe->sely1=*posy;
    *posx+=n; }
  else if (pe->sely0<pe->sely1) {
    // One or several lines are selected.
    if (copy==0 && *posy>=pe->sely0 && *posy<=pe->sely1)
      return 0;                        // Move to itself
    // Copy selection to the temporary buffer.
    n=pe->sely1-pe->sely0;
    pb=(char *)malloc(n*pe->ncol);
    if (pb==NULL)
      return -1;                       // Low memory
    memcpy(pb,pe->data+pe->sely0*pe->ncol,n*pe->ncol);
    // If necessary, delete selection.
    y=*posy;
    if (copy==0) {
      if (y>=pe->sely1)
        y-=pe->sely1-pe->sely0;
      else if (y>pe->sely0)
        y=pe->sely0;
      Deleteeditlines(pe,pe->sely0,n); };
    // Insert text.
    if (Inserteditlines(pe,y,n)==0) {
      memcpy(pe->data+y*pe->ncol,pb,n*pe->ncol);
      pe->selx0=pe->selxb=0; pe->sely0=pe->selyb=y;
      pe->selx1=0; pe->sely1=y+n; };
    *posx=0;
    *posy=pe->sely1;
    free(pb); }
  else return -1;                      // Invalid selection
  pe->modified=1;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  Updatestatus(pe);
  // Report success.
  return 0;
};

// Service function, opens Find Text dialog. Returns 0 on success and -1 on
// error.
static int Getfindedittextpattern(t_edit *pe) {
  if (pe==NULL || pe->data==NULL || pe->w.hw==NULL || pe->w.hedit==NULL)
    return -1;                         // Error in input parameters
  if (findtextmsgid==0)
    return -1;                         // Oops, unregistered FindText() message
  if (hwfindtext!=NULL)
    return 0;
  memset(&findreplace,0,sizeof(findreplace));
  findreplace.lStructSize=sizeof(findreplace);
  findreplace.hwndOwner=pe->w.hedit;
  findreplace.hInstance=hinst;
  findreplace.Flags=pe->searchflags;
  findreplace.lpstrFindWhat=pe->find;
  findreplace.lpstrReplaceWith=NULL;
  findreplace.wFindWhatLen=TEXTLEN;
  findreplace.wReplaceWithLen=0;
  findreplace.lCustData=(LPARAM)pe;
  findreplace.lpfnHook=NULL;
  findreplace.lpTemplateName=NULL;
  hwfindtext=FindText(&findreplace);
  return (hwfindtext==NULL?-1:0);
};

// Service function, searches for the next occurance of the search string.
// Starts at (posx,posy). On success, sets new position and returns 0. On
// error, or if there is no match, returns -1.
static int Findedittext(t_edit *pe,int *posx,int *posy) {
  int n,x,y,dir,matchcase;
  if (pe==NULL || pe->data==NULL || posx==NULL || posy==NULL)
    return -1;                         // Error in input parameters
  if (*posx<0 || *posx>=pe->ncol || *posy<0 || *posy>pe->ncard)
    return -1;                         // Position outside the deck
  n=Strlength(pe->find,TEXTLEN);
  if (n==0)
    Getfindedittextpattern(pe);        // No search pattern, get now
  if (n==0 || n>pe->ncol)
    return -1;                         // No or too long search pattern
  dir=(pe->searchflags & FR_DOWN);     // 0: backward, !0: forward
  matchcase=(pe->searchflags & FR_MATCHCASE);
  x=*posx-n;                           // Assuming cursor after last match
  y=*posy;
  // The search is very ineffective, but I don't care.
  while (1) {
    // Increment or decrement search location.
    if (dir) {
      x++; if (x+n>pe->ncol) {
        x=0; y++; if (y>=pe->ncard) return -1;
      }; }
    else {
      x--; if (x<0) {
        x=pe->ncol-n; y--; if (y<0) return -1;
      };
    };
    if (matchcase!=0 && memcmp(pe->data+y*pe->ncol+x,pe->find,n)==0)
      break;
    else if (matchcase==0 && memicmp(pe->data+y*pe->ncol+x,pe->find,n)==0)
      break;
    ;
  };
  // Text found, set selection and cursor.
  Addeditundo(pe,0,EDU_REPL);          // Line does not matter
  pe->selx0=pe->selxb=x; pe->sely0=pe->selyb=y;
  pe->selx1=x+n; pe->sely1=y;
  *posx=x+n; *posy=y;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  // Report success.
  return 0;
};

// Undoes last modification. Returns 0 on success, 1 if there is nothing to
// undo and -1 on error.
static int Undoedit(t_edit *pe,int *posx,int *posy) {
  int itemsize,firstundo;
  char data[TEXTLEN];
  t_undo *pu;
  if (pe==NULL || pe->undo==NULL)
    return -1;                         // Error in input parameters
  if (pe->nundo==0)
    return 1;                          // Nothing to undo
  // Calculate size of undo element.
  itemsize=sizeof(t_undo)+pe->ncol;
  // Undo blocks are separated by finalization markers. We must stop before the
  // second such marker.
  pe->stopundo=1;
  firstundo=1;
  while (pe->nundo>0) {
    pu=(t_undo *)(pe->undo+itemsize*(pe->nundo-1));
    if (firstundo) {
      if (pe->nundo==pe->nredo)
        pu->op|=EDU_FINALIZE;          // For the possible redo
      firstundo=0; }
    else if (pu->op & EDU_FINALIZE)
      break;                           // Undo block executed
    switch (pu->op & EDU_OPMASK) {
      case EDU_DEL:
        // Line was deleted, insert it back.
        Inserteditlines(pe,pu->line,1);
        memcpy(pe->data+pu->line*pe->ncol,pu->data,pe->ncol);
        break;
      case EDU_INS:
        // Line was inserted, delete it now.
        Deleteeditlines(pe,pu->line,1);
        break;
      case EDU_REPL:
      case EDU_REPF:
        // Line was modified, restore old contents.
        memcpy(data,pe->data+pu->line*pe->ncol,pe->ncol);
        memcpy(pe->data+pu->line*pe->ncol,pu->data,pe->ncol);
        memcpy(pu->data,data,pe->ncol);
        break;
      default: break;
    };
    // Restore cursor and selection.
    if (posx!=0) *posx=pu->cursorx;
    if (posy!=0) *posy=pu->cursory;
    pe->selx0=pu->selx0;
    pe->sely0=pu->sely0;
    pe->selx1=pu->selx1;
    pe->sely1=pu->sely1;
    pe->nundo--;
  };
  pe->stopundo=0;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  return 0;
};

// Redoes last undo. Returns 0 on success, 1 if there is nothing to redo and -1
// on error.
static int Redoedit(t_edit *pe,int *posx,int *posy) {
  int itemsize;
  char data[TEXTLEN];
  t_undo *pu;
  if (pe==NULL || pe->undo==NULL)
    return -1;                         // Error in input parameters
  if (pe->nredo<=pe->nundo)
    return 1;                          // Nothing to redo
  // Calculate size of undo element.
  itemsize=sizeof(t_undo)+pe->ncol;
  // Undo blocks are separated by finalization markers. We must stop after
  // processing first such marker.
  pe->stopundo=1;
  while (pe->nredo>pe->nundo) {
    pu=(t_undo *)(pe->undo+itemsize*pe->nundo);
    switch (pu->op & EDU_OPMASK) {
      case EDU_DEL:
        // Line was originally deleted, delete it again.
        Deleteeditlines(pe,pu->line,1);
        break;
      case EDU_INS:
        // Line was inserted, repeat.
        Inserteditlines(pe,pu->line,1);
        break;
      case EDU_REPL:
      case EDU_REPF:
        // Line was modified, restore new contents.
        memcpy(data,pe->data+pu->line*pe->ncol,pe->ncol);
        memcpy(pe->data+pu->line*pe->ncol,pu->data,pe->ncol);
        memcpy(pu->data,data,pe->ncol);
        break;
      default: break;
    };
    // Restore cursor and selection.
    if (posx!=0) *posx=pu->cursorx;
    if (posy!=0) *posy=pu->cursory;
    pe->selx0=pu->selx0;
    pe->sely0=pu->sely0;
    pe->selx1=pu->selx1;
    pe->sely1=pu->sely1;
    pe->nundo++;
    if (pu->op & EDU_FINALIZE) break;  // Undo block executed
  };
  pe->stopundo=0;
  if (pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  return 0;
};


////////////////////////////////////////////////////////////////////////////////
//////////////////////////// EDIT WINDOW FUNCTIONS /////////////////////////////

// Window procedure of base editor window.
LRESULT CALLBACK Ebasewindowclassproc(HWND hw,UINT msg,WPARAM wp,LPARAM lp) {
  t_edit *pe;
  RECT rc,rs;
  PAINTSTRUCT ps;
  pe=(t_edit *)GetWindowLong(hw,0);
  if (pe==NULL)                        // Alarm! Editor window without edit!
    return DefWindowProc(hw,msg,wp,lp);
  switch (msg) {                                                     
    case WM_DESTROY:                   // About to destroy edit window
      if (hwfindtext!=NULL) {
        DestroyWindow(hwfindtext); hwfindtext=NULL; };
      pe->w.hw=NULL;
      SetWindowLong(hw,0,NULL);
      break;
    case WM_SETFOCUS:
      SetFocus(pe->w.hedit);
      break;
    case WM_LBUTTONDOWN:
    case WM_RBUTTONDOWN:
    case WM_MBUTTONDOWN:
      SetFocus(pe->w.hedit);
      break;
    case WM_SIZE:
      GetClientRect(hw,&rc);
      if (pe->w.hstatus!=NULL) {
        SendMessage(pe->w.hstatus,msg,wp,lp);
        GetWindowRect(pe->w.hstatus,&rs);
        rc.bottom-=rs.bottom-rs.top; };
      SetWindowPos(pe->w.hedit,NULL,
        0,0,rc.right,rc.bottom,SWP_NOMOVE|SWP_NOZORDER);
      break;
    case WM_PAINT:
      BeginPaint(hw,&ps);              // There is no free client area here
      EndPaint(hw,&ps);
      break;
    default:
      return DefWindowProc(hw,msg,wp,lp);
    ;
  };
  return 0;
};

// Window procedure of work editor window.
LRESULT CALLBACK Editwindowclassproc(HWND hedit,UINT msg,WPARAM wp,LPARAM lp) {
  int i,j,x,y,nx,ny,fw,fh,selx0,selx1,snowfree,spaces[TEXTLEN];
  int shiftkey,controlkey,cursorx,cursory,linesize,isjcl,redraw;
  int lng,*ptab;
  char c,s[TEXTLEN],*pd;
  t_edit *pe;
  RECT rf,rl,rm,rc;
  PAINTSTRUCT ps;
  SCROLLINFO sinfo;
  HDC dc,wc;
  HPEN hpen,hlightpen;
  HBRUSH hwhite,hgreen,hselect;
  HBITMAP hbmp;
  pe=(t_edit *)GetWindowLong(hedit,0);
  if (pe==NULL)                        // Alarm! Editor window without edit!
    return DefWindowProc(hedit,msg,wp,lp);
  switch (msg) {
    case WM_DESTROY:                   // About to destroy edit window
      if (hwfindtext!=NULL) {
        DestroyWindow(hwfindtext); hwfindtext=NULL; };
      pe->w.hedit=NULL;
      SetWindowLong(hedit,0,NULL);
      break;
    case WM_SETFOCUS:
      Seteditcaretpos(pe,pe->cursorx,pe->cursory,0);
      pe->ctrlx=0;
      break;
    case WM_KILLFOCUS:
      DestroyCaret();
      break;
    case WM_LBUTTONDOWN:
    case WM_LBUTTONDBLCLK:
      SetFocus(hedit);
      fw=max(1,pe->w.fontdx);
      fh=max(1,pe->w.fontdy);
      Addeditundo(pe,0,EDU_FINALIZE);
      x=LOINT(lp); if ((pe->w.options & EDO_LBORDER)==0) x+=fw/2;
      y=HIINT(lp);
      if (pe->mode & EDM_CARD) {
        cursorx=Getcardcharindex(&pe->cardpos,x);
        cursory=pe->cursory; }
      else {
        cursorx=x/fw+pe->w.offsetx;
        cursory=y/fh+pe->w.offsety; };
      Seteditcaretpos(pe,cursorx,cursory,1);
      // Remove selection.
      pe->selx0=pe->selx1=pe->selxb=pe->cursorx;
      pe->sely0=pe->sely1=pe->selyb=pe->cursory;
      SetCapture(hedit);
      if (pe->w.invalidated==0) {
        pe->w.invalidated=1;
        InvalidateRect(pe->w.hedit,NULL,FALSE); };
      break;
    case WM_MOUSEMOVE:
      if (GetCapture()!=hedit)
        break;
      fw=max(1,pe->w.fontdx);
      fh=max(1,pe->w.fontdy);
      x=LOINT(lp); if ((pe->w.options & EDO_LBORDER)==0) x+=fw/2;
      y=HIINT(lp);
      if (pe->mode & EDM_CARD) {
        cursorx=Getcardcharindex(&pe->cardpos,x);
        cursory=pe->cursory; }
      else {
        cursorx=x/fw+pe->w.offsetx;
        cursory=y/fh+pe->w.offsety; };
      if (Seteditcaretpos(pe,cursorx,cursory,1)!=0) {
        // Change selection.
        if (pe->cursory<pe->selyb ||
          (pe->cursory==pe->selyb && pe->cursorx<=pe->selxb)
        ) {
          pe->selx0=pe->cursorx; pe->sely0=pe->cursory;
          pe->selx1=pe->selxb; pe->sely1=pe->selyb; }
        else {
          pe->selx0=pe->selxb; pe->sely0=pe->selyb;
          pe->selx1=pe->cursorx; pe->sely1=pe->cursory; };
        if (pe->sely0!=pe->sely1) {
          pe->selx0=0; pe->selx1=0; };
        if (pe->w.invalidated==0) {
          pe->w.invalidated=1;
          InvalidateRect(pe->w.hedit,NULL,FALSE);
        };
      };
      break;
    case WM_LBUTTONUP:
      if (GetCapture()==hedit)
        ReleaseCapture();
      break;
    case WM_RBUTTONDOWN:
    case WM_MBUTTONDOWN:
      SetFocus(hedit);
      break;
    case WM_HSCROLL:
      // Recalculate width of the client area into characters.
      GetClientRect(hedit,&rc);
      fw=max(1,pe->w.fontdx);
      nx=rc.right/fw;
      // Process message.
      j=pe->w.offsetx;
      switch (LOWORD(wp)) {
        case SB_LINELEFT: j--; break;
        case SB_LINERIGHT: j++; break;
        case SB_PAGELEFT: j-=max(1,nx-1); break;
        case SB_PAGERIGHT: j+=max(1,nx-1); break;
        case SB_THUMBTRACK:
          sinfo.cbSize=sizeof(sinfo);
          sinfo.fMask=SIF_TRACKPOS;
          GetScrollInfo(hedit,SB_HORZ,&sinfo);
          j=sinfo.nTrackPos;
          break;
        default: break; };
      linesize=pe->ncol;
      if (pe->w.options & EDO_LBORDER)
        linesize++;
      if (j+nx>linesize) j=linesize-nx;
      if (j<0) j=0;
      if (j!=pe->w.offsetx) {
        pe->w.offsetx=j;
        pe->w.invalidated=1;
        InvalidateRect(hedit,NULL,FALSE); };
      break;
    case WM_VSCROLL:
      // Recalculate height of the client area into characters.
      GetClientRect(hedit,&rc);
      if (pe->mode & EDM_CARD)
        ny=1;
      else {
        fh=max(1,pe->w.fontdy); ny=rc.bottom/fh; };
      // Process message.
      j=pe->w.offsety;
      switch (LOWORD(wp)) {
        case SB_LINEUP: j-=1; break;
        case SB_LINEDOWN: j+=1; break;
        case SB_PAGEUP: j-=max(1,ny-1)*1; break;
        case SB_PAGEDOWN: j+=max(1,ny-1)*1; break;
        case SB_THUMBTRACK:
          sinfo.cbSize=sizeof(sinfo);
          sinfo.fMask=SIF_TRACKPOS;
          GetScrollInfo(hedit,SB_VERT,&sinfo);
          j=sinfo.nTrackPos;
          break;
        default: break; };
      if (j+ny>pe->ncard+1) j=pe->ncard+1-ny;
      if (j<0) j=0;
      if (j!=pe->w.offsety) {
        pe->w.offsety=j;
        if (pe->w.invalidated==0) {
          pe->w.invalidated=1;
          InvalidateRect(pe->w.hedit,NULL,FALSE);
        };
      };
      break;
    case WM_USER_WHEEL:
      // Wheel scrolling is translated into the sequence of WM_VSCROLLs.
      i=((signed short)HIWORD(wp))/WHEEL_DELTA;
      if ((pe->mode & EDM_CARD)==0) {
        // Form view.
        for (j=0; j<abs(i) && j<8; j++)
          Editwindowclassproc(hedit,WM_VSCROLL,(i>0?SB_LINEUP:SB_LINEDOWN),0);
        ; }
      else {
        // Card view.
        if (i>0)
          Editwindowclassproc(hedit,WM_KEYDOWN,VK_UP,0);
        else
          Editwindowclassproc(hedit,WM_KEYDOWN,VK_DOWN,0);
        ; };
      break;
    case WM_SYSKEYDOWN:
    case WM_KEYDOWN:
    case WM_CHAR:
      GetClientRect(hedit,&rc);
      fw=max(1,pe->w.fontdx); nx=rc.right/fw;
      if (pe->mode & EDM_CARD)
        ny=1;
      else {
        fh=max(1,pe->w.fontdy); ny=rc.bottom/fh; };
      cursorx=pe->cursorx;
      cursory=pe->cursory;
      shiftkey=GetKeyState(VK_SHIFT) & 0x8000;
      controlkey=GetKeyState(VK_CONTROL) & 0x8000;
      redraw=0;
      // WM_SYSKEYDOWN messages.
      if (msg==WM_SYSKEYDOWN) {
        switch (wp) {
          case VK_BACK:
            pe->ctrlx=0;
            if (pe->mode & EDM_NOEDIT)
              break;
            if (shiftkey==0)
              // Alt-Backspace undoes last operation.
              Undoedit(pe,&cursorx,&cursory);
            else
              // Shift-Alt-Backspace redoes last undo.
              Redoedit(pe,&cursorx,&cursory);
            break;
          default: break;
        }; }
      // WM_KEYDOWN messages.
      else if (msg==WM_KEYDOWN) {
        switch (wp) {
          case VK_SHIFT:
            // Mark start of selection.
            if (cursorx==pe->selx0 && cursory==pe->sely0) {
              pe->selxb=pe->selx1;
              pe->selyb=pe->sely1; }
            else if (cursorx==pe->selx1 && cursory==pe->sely1) {
              pe->selxb=pe->selx0;
              pe->selyb=pe->sely0; }
            else {
              pe->selxb=cursorx;
              pe->selyb=cursory; };
            break;
          case VK_INSERT:
            pe->ctrlx=0;
            if (controlkey) {
              // Ctrl+Ins copies selection to clipboard.
              if ((pe->mode & EDM_NOEDIT)==0)
                Addeditundo(pe,0,EDU_FINALIZE);
              Copyedittoclipboard(pe); }
            else if (shiftkey) {
              // Shift+Ins pastes cipboard to cursor position.
              if (pe->mode & EDM_NOEDIT) break;
              Addeditundo(pe,0,EDU_FINALIZE);
              Pasteeditfromclipboard(pe,&cursorx,&cursory);
              Addeditundo(pe,0,EDU_FINALIZE); }
            else {
              // Ins toggles between insert and overwrite modes.
              if (pe->mode & EDM_NOEDIT) break;
              pe->overwrite=~pe->overwrite;
              Seteditcaretpos(pe,cursorx,cursory,0);
              Updatestatus(pe); };
            break;
          case VK_LEFT:
            Addeditundo(pe,0,EDU_FINALIZE);
            cursorx--; pe->ctrlx=0; break;
          case VK_RIGHT:
            Addeditundo(pe,0,EDU_FINALIZE);
            cursorx++; pe->ctrlx=0; break;
          case VK_UP:
            Addeditundo(pe,0,EDU_FINALIZE);
            cursory--; pe->ctrlx=0; break;
          case VK_DOWN:
            Addeditundo(pe,0,EDU_FINALIZE);
            cursory++; pe->ctrlx=0; break;
          case VK_PRIOR:
            Addeditundo(pe,0,EDU_FINALIZE);
            if (controlkey) cursory=0;
            else cursory-=max(ny-2,1);
            pe->ctrlx=0; break;
          case VK_NEXT:
            Addeditundo(pe,0,EDU_FINALIZE);
            if (controlkey) cursory=pe->ncard;
            else cursory+=max(ny-2,1);
            pe->ctrlx=0; break;
          case VK_HOME:
            Addeditundo(pe,0,EDU_FINALIZE);
            cursorx=0; pe->ctrlx=0; break;
          case VK_END:
            Addeditundo(pe,0,EDU_FINALIZE);
            Toendofeditline(pe,&cursorx,&cursory);
            pe->ctrlx=0; break;
          case VK_TAB:
            pe->ctrlx=0;
            if (pe->mode & EDM_NOEDIT)
              break;
            Addeditundo(pe,0,EDU_FINALIZE);
            if (cursory==pe->ncard && Inserteditlines(pe,cursory,1)!=0)
              break;
            isjcl=(pe->data[cursory*pe->ncol]=='/' &&
              (pe->data[cursory*pe->ncol+1]=='/' ||
              pe->data[cursory*pe->ncol+1]=='*'));
            if (isjcl)
              ptab=tabjcl;
            else {
              lng=Getlanguage(pe);
              if (lng==LNG_ASM)
                ptab=tabasm;
              else if (lng==LNG_FORTRAN)
                ptab=tabfortran;
              else
                ptab=tabdefault;
              ;
            };
            j=cursorx+1;
            for (i=0; i<NTAB; i++) {
              if (ptab[i]>cursorx) {
                j=ptab[i]; break;
              };
            };
            if (pe->overwrite==0) {
              memset(s,' ',j-cursorx);
              Inserteditchars(pe,s,j-cursorx,cursorx,cursory,0,0);
              pe->modified=1;
              Updatestatus(pe);
              if (pe->w.invalidated==0) {
                pe->w.invalidated=1;
                InvalidateRect(pe->w.hedit,NULL,FALSE);
              };
            };
            Addeditundo(pe,0,EDU_FINALIZE);
            cursorx=j;
            break;
          case VK_BACK:
          case VK_DELETE:
            pe->ctrlx=0;
            if (pe->mode & EDM_NOEDIT)
              break;
            if (controlkey!=0 && wp==VK_DELETE) {
              // Ctrl+Del cuts selection to clipboard.
              Addeditundo(pe,0,EDU_FINALIZE);
              Copyedittoclipboard(pe);
              Deleteeditselection(pe,&cursorx,&cursory);
              Addeditundo(pe,0,EDU_FINALIZE); }
            else {
              if (wp==VK_BACK) {
                if (cursorx==0) {
                  // Backspace at the beginning of the line merges cards.
                  if (cursory>0) {
                    cursorx=Mergeeditlines(pe,cursory);
                    if (cursorx<0) cursorx=0;
                    else cursory--; };
                  break; };
                cursorx--; };
              Deleteeditchars(pe,1,cursorx,cursory); };
            break;
          case VK_RETURN:
            pe->ctrlx=0;
            if (pe->mode & EDM_NOEDIT)
              break;
            Addeditundo(pe,0,EDU_FINALIZE);
            if (cursory>=pe->ncard)
              Inserteditlines(pe,cursory,1);
            else
              Spliteditline(pe,cursorx,cursory);
            Addeditundo(pe,0,EDU_FINALIZE);
            cursorx=0;
            cursory++;
            break;
          case VK_F2:
            if (pe->filename[0]!='\0') {
              Addeditundo(pe,0,EDU_FINALIZE);
              Saveeditfile(pe,0); };
            pe->ctrlx=0; break;
          case VK_F3:
            Addeditundo(pe,0,EDU_FINALIZE);
            Findedittext(pe,&cursorx,&cursory);
            pe->ctrlx=0; break;
          case VK_F5:
            // Toggle between form and card modi.
            Addeditundo(pe,0,EDU_FINALIZE);
            pe->mode^=EDM_CARD;
            Removeeditselection(pe);
            redraw=1; break;
          case 'A':
            if (controlkey!=0) {
              // Ctrl+A selects everything.
              Addeditundo(pe,0,EDU_FINALIZE);
              pe->selxb=pe->selx0=0;
              pe->selyb=pe->sely0=0;
              if (pe->ncard==0) {
                pe->selx1=0;
                pe->sely1=0; }
              else {
                pe->selx1=pe->ncol;
                pe->sely1=pe->ncard; };
              redraw=1; pe->ctrlx=0; };
            break;
          case 'C':
            if (controlkey!=0) {
              // Ctrl+C copies selection to clipboard.
              Addeditundo(pe,0,EDU_FINALIZE);
              Copyedittoclipboard(pe);
              pe->ctrlx=0; };
            break;
          case 'F':
            if (controlkey!=0) {
              // Ctrl+F opens text search dialog.
              Addeditundo(pe,0,EDU_FINALIZE);
              Getfindedittextpattern(pe);
              pe->ctrlx=0; };
            break;
          case 'K':
            if (controlkey!=0) {
              // Ctrl+K starts block operations.
              Addeditundo(pe,0,EDU_FINALIZE);
              pe->ctrlx='K'; };
            break;
          case 'L':
            if (controlkey!=0) {
              // Ctrl+L searches for the next occurence of the search pattern.
              Addeditundo(pe,0,EDU_FINALIZE);
              Findedittext(pe,&cursorx,&cursory);
              pe->ctrlx=0; };
            break;
          case 'N':
            if (controlkey!=0) {
              // Ctrl+N renumerates cards in deck.
              pe->ctrlx=0;
              if (pe->mode & EDM_NOEDIT)
                break;
              Addeditundo(pe,0,EDU_FINALIZE);
              Renumerateeditlines(pe);
              Addeditundo(pe,0,EDU_FINALIZE); };
            break;
          case 'O':
            if (controlkey!=0) {
              // Ctrl+O loads file from disk.
              pe->ctrlx=0;
              if (pe->mode & EDM_NOEDIT)
                break;
              Loadeditfile(pe); };
            break;
          case 'Q':
            if (controlkey!=0)
              // Ctrl+Q starts search operations.
              pe->ctrlx='Q';
            break;
          case 'R':
            if (controlkey!=0) {
              // Ctrl+R redoes last undo.
              pe->ctrlx=0;
              if (pe->mode & EDM_NOEDIT)
                break;
              Redoedit(pe,&cursorx,&cursory); };
            break;
          case 'S':
            if (controlkey!=0) {
              // Ctrl+S saves file to disk.
              Addeditundo(pe,0,EDU_FINALIZE);
              Saveeditfile(pe,0);
              pe->ctrlx=0; };
            break;
          case 'V':
            if (controlkey!=0) {
              // Ctrl+V pastes cipboard to cursor position.
              pe->ctrlx=0;
              if (pe->mode & EDM_NOEDIT)
                break;
              Addeditundo(pe,0,EDU_FINALIZE);
              Pasteeditfromclipboard(pe,&cursorx,&cursory);
              Addeditundo(pe,0,EDU_FINALIZE); };
            break;
          case 'X':
            if (controlkey!=0) {
              // Ctrl+X cuts selection to clipboard.
              pe->ctrlx=0;
              if (pe->mode & EDM_NOEDIT)
                break;
              Addeditundo(pe,0,EDU_FINALIZE);
              Copyedittoclipboard(pe);
              Deleteeditselection(pe,&cursorx,&cursory);
              Addeditundo(pe,0,EDU_FINALIZE); };
            break;
          case 'Y':
            if (controlkey!=0) {
              // Ctrl+Y deletes current line.
              pe->ctrlx=0;
              if (pe->mode & EDM_NOEDIT)
                break;
              Addeditundo(pe,0,EDU_FINALIZE);
              Deleteeditlines(pe,cursory,1);
              Addeditundo(pe,0,EDU_FINALIZE); };
            break;
          case 'Z':
            if (controlkey!=0) {
              // Ctrl+Z undoes last modification.
              pe->ctrlx=0;
              if (pe->mode & EDM_NOEDIT)
                break;
              Undoedit(pe,&cursorx,&cursory); };
            break;
          default: break;
        }; }
      // WM_CHAR messages: characters.
      else if (isprint(wp) && controlkey==0 && cursorx<pe->ncol) {
        c=(char)toupper(wp);
        if (pe->ctrlx=='K' && c=='C') {
          // Ctrl+K,C copies selection to cursor position.
          if (pe->mode & EDM_NOEDIT)
            break;
          Addeditundo(pe,0,EDU_FINALIZE);
          Copymoveeditselection(pe,&cursorx,&cursory,1);
          Addeditundo(pe,0,EDU_FINALIZE); }
        else if (pe->ctrlx=='K' && c=='H') {
          // Ctrl+K,H removes selection.
          if (pe->mode & EDM_NOEDIT)
            break;
          Addeditundo(pe,0,EDU_FINALIZE);
          Removeeditselection(pe); }
        else if (pe->ctrlx=='K' && c=='V') {
          // Ctrl+K,V moves selection to cursor position.
          if (pe->mode & EDM_NOEDIT)
            break;
          Addeditundo(pe,0,EDU_FINALIZE);
          Copymoveeditselection(pe,&cursorx,&cursory,0);
          Addeditundo(pe,0,EDU_FINALIZE); }
        else if (pe->ctrlx=='K' && c=='Y') {
          // Ctrl+K,Y deletes selection.
          if (pe->mode & EDM_NOEDIT)
            break;
          Addeditundo(pe,0,EDU_FINALIZE);
          Deleteeditselection(pe,&cursorx,&cursory);
          Addeditundo(pe,0,EDU_FINALIZE); }
        else if (pe->ctrlx=='Q' && c=='F') {
          // Ctrl+Q,F opens text search dialog.
          Addeditundo(pe,0,EDU_FINALIZE);
          Getfindedittextpattern(pe); }
        else if (pe->ctrlx==0) {
          // Insert or replace character.
          if (pe->mode & EDM_NOEDIT)
            break;
          if ((pe->mode & EDM_UPCASE)==0)
            c=(char)wp;
          Inserteditchars(pe,&c,1,cursorx,cursory,pe->overwrite,0);
          cursorx++;
        };
        pe->ctrlx=0;
      };
      // Update cursor position and, if cursor has moved, update selection.
      if (Seteditcaretpos(pe,cursorx,cursory,1)!=0 &&
        shiftkey!=0 && msg==WM_KEYDOWN
      ) {
        if (pe->cursory<pe->selyb ||
          (pe->cursory==pe->selyb && pe->cursorx<=pe->selxb)
        ) {
          pe->selx0=pe->cursorx; pe->sely0=pe->cursory;
          pe->selx1=pe->selxb; pe->sely1=pe->selyb; }
        else {
          pe->selx0=pe->selxb; pe->sely0=pe->selyb;
          pe->selx1=pe->cursorx; pe->sely1=pe->cursory; };
        if (pe->sely0!=pe->sely1) {
          pe->selx0=0; pe->selx1=0; };
        redraw=1;
      };
      if (redraw!=0 && pe->w.invalidated==0) {
        pe->w.invalidated=1;
        InvalidateRect(pe->w.hedit,NULL,FALSE); };
      break;
    case WM_PAINT:
      HideCaret(hedit);
      wc=BeginPaint(hedit,&ps);
      if (pe->mode & EDM_CARD) {
        ShowScrollBar(hedit,SB_VERT,0);
        sinfo.cbSize=sizeof(sinfo);
        sinfo.nPage=80;
        sinfo.nMin=0;
        sinfo.nMax=79;
        sinfo.nPos=0;
        sinfo.fMask=SIF_PAGE|SIF_POS|SIF_RANGE;
        SetScrollInfo(hedit,SB_HORZ,&sinfo,TRUE);
        GetClientRect(hedit,&rc);
        // Show images of the punched card.
        if (pe->cursory<pe->ncard)
          pd=pe->data+pe->ncol*pe->cursory;
        else
          pd=NULL;
        if (pe->cursory>=pe->sely0 && pe->cursory<=pe->sely1) {
          if (pe->cursory==pe->sely0) selx0=pe->selx0;
          else selx0=0;
          if (pe->cursory==pe->sely1) selx1=pe->selx1;
          else selx1=pe->ncol; }
        else
          selx0=selx1=0;
        Drawcard(wc,&rc,0,pe->cursory+1,pd,selx0,selx1,&(pe->cardpos)); }
      else {
        // Show paper form.
        ShowScrollBar(hedit,SB_VERT,1);
        GetClientRect(hedit,&rc);
        fw=max(1,pe->w.fontdx);
        fh=max(1,pe->w.fontdy);
        // Set scroll bars. Vertical bar is always here.
        ny=rc.bottom/fh;
        sinfo.cbSize=sizeof(sinfo);
        sinfo.nPage=max(1,ny);
        sinfo.nMin=0;
        sinfo.nMax=pe->ncard;
        sinfo.nPos=pe->w.offsety;
        sinfo.fMask=SIF_PAGE|SIF_POS|SIF_RANGE|SIF_DISABLENOSCROLL;
        SetScrollInfo(hedit,SB_VERT,&sinfo,TRUE);
        // Now as vertical bar is shown, client rectangle may have changed.
        GetClientRect(hedit,&rc);
        nx=rc.right/fw;
        sinfo.nPage=max(1,nx);
        sinfo.nMax=pe->ncol-1;
        if (pe->w.options & EDO_LBORDER)
          sinfo.nMax++;
        sinfo.nPos=pe->w.offsetx;
        sinfo.fMask=SIF_PAGE|SIF_POS|SIF_RANGE;
        SetScrollInfo(hedit,SB_HORZ,&sinfo,TRUE);
        // If snow-free drawing is requested, draw into the memory DC;
        // otherwise, directly to the screen.
        snowfree=pe->w.options & EDO_SNOWFREE;
        if (snowfree==0) {
          dc=wc; hbmp=NULL; }
        else {
          dc=CreateCompatibleDC(wc);
          if (dc==NULL) {                // Emergency solution
            dc=wc; hbmp=NULL; snowfree=0; }
          else {
            hbmp=CreateCompatibleBitmap(wc,rc.right,fh);
            if (hbmp!=NULL)
              SelectObject(dc,hbmp);
            else {
              DeleteDC(dc);
              dc=wc; hbmp=NULL; snowfree=0;
            };
          };
        };
        // Draw contents of the window. Editor keeps lines padded with spaces
        // to the right.
        GetClientRect(hedit,&rc);
        SetTextAlign(dc,TA_LEFT|TA_BOTTOM);
        SetBkMode(dc,TRANSPARENT);
        SetTextColor(dc,RGB(0,0,0));
        hpen=CreatePen(PS_SOLID,0,RGB(128,220,128));
        hlightpen=CreatePen(PS_SOLID,0,RGB(192,255,192));
        hwhite=CreateSolidBrush(RGB(255,255,255));
        hgreen=CreateSolidBrush(RGB(240,255,240));
        hselect=CreateSolidBrush(RGB(180,180,180));
        SelectObject(dc,pe->w.hfont);
        x=-pe->w.offsetx*fw;
        if (pe->w.options & EDO_LBORDER) x+=fw/2;
        for (i=0; i<pe->ncol; i++)
          spaces[i]=fw;
        rm=rc;
        rm.bottom=fh;
        rl=rc;
        lng=Getlanguage(pe);
        for (j=pe->w.offsety; ; j++) {
          // Draw next visible row.
          if (rl.top>=rc.bottom)
            break;                     // Window is filled
          rl.bottom=rl.top+fh;
          if (snowfree==0)
            rm=rl;
          FillRect(dc,&rm,hwhite);
          // Check whether JCL card.
          isjcl=(j<pe->ncard && pe->data[j*pe->ncol]=='/' &&
            (pe->data[j*pe->ncol+1]=='/' || pe->data[j*pe->ncol+1]=='*'));
          // Draw form background.
          if (j<pe->ncard || whitenempty==0) {
            rf.top=rm.top; rf.bottom=rm.bottom;
            if (isjcl==0 && lng==LNG_FORTRAN) {
              rf.left=x+5*fw+1; rf.right=x+6*fw;
              FillRect(dc,&rf,hgreen); }
            else if (isjcl==0 && lng==LNG_ASM) {
              rf.left=x+8*fw+1; rf.right=x+9*fw;
              FillRect(dc,&rf,hgreen);
              rf.left=x+14*fw+1; rf.right=x+15*fw;
              FillRect(dc,&rf,hgreen); };
            rf.left=x+72*fw+1; rf.right=x+80*fw;
            FillRect(dc,&rf,hgreen);
          };
          // Draw selection.
          if (j>=pe->sely0 && j<=pe->sely1) {
            if (j==pe->sely0) selx0=pe->selx0;
            else selx0=0;
            if (j==pe->sely1) selx1=pe->selx1;
            else selx1=pe->ncol;
            if (selx0<selx1) {
              rf.top=rm.top; rf.bottom=rm.bottom;
              rf.left=x+selx0*fw; rf.right=x+selx1*fw;
              FillRect(dc,&rf,hselect);
            };
          };
          // Draw form grid.
          SelectObject(dc,(j<pe->ncard || whitenempty==0?hpen:hlightpen));
          MoveToEx(dc,x,rm.top,NULL);
          LineTo(dc,x,rm.bottom-1);
          LineTo(dc,x+pe->ncol*fw,rm.bottom-1);
          LineTo(dc,x+pe->ncol*fw,rm.top-1);
          for (i=1; i<pe->ncol; i++) {
            if (i==72 ||
              (isjcl==0 && lng==LNG_FORTRAN && (i==5 || i==6)) ||
              (isjcl==0 && lng==LNG_ASM &&
              (i==8 || i==9 || i==14 || i==15 || i==71)))
              MoveToEx(dc,x+i*fw,rm.top,NULL);
            else
              MoveToEx(dc,x+i*fw,rm.bottom-3,NULL);
            LineTo(dc,x+i*fw,rm.bottom);
          };
          // Draw text.
          if (j<pe->ncard)
            ExtTextOut(dc,x+EDITEXTRAX-EDITEXTRAX/2,rm.bottom-1,0,NULL,
            pe->data+j*pe->ncol,pe->ncol,spaces);
          // If drawing was to memory DC, flush data to the screen.
          if (snowfree)
            BitBlt(wc,0,rl.top,rl.right,fh,dc,0,0,SRCCOPY);
          rl.top=rl.bottom;
        };
        DeleteObject(hselect);
        DeleteObject(hgreen);
        DeleteObject(hwhite);
        DeleteObject(hlightpen);
        DeleteObject(hpen);
        if (snowfree) {
          DeleteDC(dc);
          DeleteObject(hbmp);
        };
      };
      EndPaint(hedit,&ps);
      pe->w.invalidated=0;
      Seteditcaretpos(pe,pe->cursorx,pe->cursory,0);
      break;
    default:
      if (findtextmsgid==0 || msg!=findtextmsgid)
        return DefWindowProc(hedit,msg,wp,lp);
      // Process FindText() message.
      pe->searchflags=((FINDREPLACE *)lp)->Flags;
      if (pe->searchflags & FR_DIALOGTERM)
        hwfindtext=NULL;
      if (pe->searchflags & FR_FINDNEXT)
        Editwindowclassproc(hedit,WM_KEYDOWN,VK_F3,0);
      if (pe->searchflags & FR_REPLACE) {
        // As yet unsupported.
      };
      if (pe->searchflags & FR_REPLACEALL) {
        // As yet unsupported.
      };
    break;
  };
  return 0;
};

// Submits contents of the editor's buffer to the Hercules using sockets.
// Returns 0 on success and -1 on error.
int Submitedittodevice(t_edit *pe,int cuu,int port) {
  int i,k,n,err,nsent,hsock;
  ulong t;
  char hostname[TEXTLEN],card[TEXTLEN+2],*pc;
  struct hostent *he;
  struct sockaddr_in addr;
  if (pe==NULL || pe->data==NULL || pe->ncard==0)
    return -1;                         // Error in input parameters
  // Get host.
  if (gethostname(hostname,TEXTLEN)!=0)
    return -1;                         // Unable to get host name
  he=gethostbyname(hostname);
  if (he==NULL || *(he->h_addr_list)==NULL)
    return -1;                         // Unable to get host
  memcpy(&(addr.sin_addr.s_addr),*(he->h_addr_list),he->h_length);
  addr.sin_family=AF_INET;
  addr.sin_port=htons(port);
  Sendstdin("devinit %04X %i sockdev ascii trunc eof",
    cuu,port);
  Sleep(100);                          // Just for the case
  // Create and connect socket.
  hsock=socket(PF_INET,SOCK_STREAM,0);
  if (hsock==INVALID_SOCKET)
    return -1;
  if (connect(hsock,(struct sockaddr *)&addr,sizeof(addr))==SOCKET_ERROR) {
    closesocket(hsock);
    return -1; };                      // Unable to connect to Hercules
  // Send data, one card at a time.
  for (i=0,pc=pe->data; i<pe->ncard; i++,pc+=pe->ncol) {
    n=pe->ncol;
    while (n>0 && pc[n-1]==' ') n--;
    if (n>0) memcpy(card,pc,n);
    card[n++]='\r';
    card[n++]='\n';
    t=GetTickCount();
    nsent=0;
    while (nsent<n) {
      k=send(hsock,card+nsent,n-nsent,0);
      if (k>=0)
        nsent+=k;
      else if ((err=WSAGetLastError())!=WSAEWOULDBLOCK) {
        Message("JASN0021E  Socket error %i while sending data",err);
        break; }
      else if (GetTickCount()-t>1000) {
        Message("JASN0022E  Socket timeout while sending data");
        break;
      };
    };
    if (nsent<n) break;                // Data is not completed!
  };
  // Close socket.
  shutdown(hsock,SD_BOTH);
  closesocket(hsock);
  return (i<pe->ncard?-1:0);
};

// Destroys editor and associated window.
void Destroyedit(t_edit *pe) {
  if (pe==NULL)
    return;                            // Error in input parameters
  // If there is still open search dialog, close it now.
  if (hwfindtext!=NULL) {
    DestroyWindow(hwfindtext); hwfindtext=NULL; };
  // If there is still open window, close it now.
  if (pe->w.hw!=NULL)
    DestroyWindow(pe->w.hw);
  // Free allocated memory.
  if (pe->data!=NULL)
    free(pe->data);
  if (pe->undo!=NULL)
    free(pe->undo);
  free(pe);
};

// Creates new editor (without associated window). Returns pointer to editor on
// success and NULL on error.
t_edit *Createedit(ulong mode,int ncol) {
  t_edit *pe;
  if (ncol<=0)
    return NULL;                       // Error in input parameters
  if (ncol>TEXTLEN-1)
    ncol=TEXTLEN-1;                    // Correct obvious error
  // Create editor.
  pe=(t_edit *)malloc(sizeof(t_edit));
  if (pe==NULL)
    return NULL;                       // Low memory
  memset(pe,0,sizeof(t_edit));
  pe->ncol=ncol;
  pe->maxcard=1024;                    // Sound default
  pe->searchflags=FR_DOWN|FR_HIDEWHOLEWORD;
  // Allocate data buffer.
  pe->data=(char *)malloc(pe->ncol*pe->maxcard);
  if (pe->data==NULL) {
    free(pe);                          // Low memory
    return NULL; };
  // Allocate undo buffer.
  pe->undo=(uchar *)malloc((sizeof(t_undo)+ncol)*MAXUNDO);
  if (pe->undo==NULL)
    Message("JASN0020W  Low memory, undo is not possible");
  pe->mode=mode;
  return pe;
};

// Attaches window to the editor. If hparent is NULL, window is standalone.
// If hfont is NULL, selects some default font. Returns handle of the window on
// success and NULL on error.
HWND Createeditwindow(t_edit *pe,char *name,int x,int y,int width,int height,
  HWND hparent,int id,HINSTANCE hinst,HFONT hfont,int options) {
  int parts[4];
  ulong style;
  RECT rc,rs;
  WNDCLASS wc;
  TEXTMETRIC tm;
  HWND hw,hstatus,hedit;
  HDC dc;
  if (pe==NULL)
    return NULL;                       // Error in input parameters
  // If some window is already associated with the editor, keep the old one.
  if (pe->w.hw!=NULL)
    return NULL;
  // Assure that editor window classes is registered.
  if (GetClassInfo(hinst,EBASECLASS,&wc)==0) {
    wc.style=CS_OWNDC|CS_HREDRAW|CS_VREDRAW;
    wc.lpfnWndProc=Ebasewindowclassproc;
    wc.cbClsExtra=0;
    wc.cbWndExtra=sizeof(ulong);
    wc.hInstance=hinst;
    wc.hIcon=NULL;
    wc.hCursor=LoadCursor(NULL,IDC_ARROW);
    wc.hbrBackground=NULL;
    wc.lpszMenuName=NULL;
    wc.lpszClassName=EBASECLASS;
    if (RegisterClass(&wc)==0) return NULL; };
  if (GetClassInfo(hinst,EDITCLASS,&wc)==0) {
    wc.style=CS_OWNDC|CS_HREDRAW|CS_VREDRAW;
    wc.lpfnWndProc=Editwindowclassproc;
    wc.cbClsExtra=0;
    wc.cbWndExtra=sizeof(ulong);
    wc.hInstance=hinst;
    wc.hIcon=NULL;
    wc.hCursor=LoadCursor(NULL,IDC_ARROW);
    wc.hbrBackground=NULL;
    wc.lpszMenuName=NULL;
    wc.lpszClassName=EDITCLASS;
    if (RegisterClass(&wc)==0) return NULL; };
  // Set default font.
  if (hfont==NULL)
    hfont=GetStockObject(ANSI_FIXED_FONT);
  // Create base window.
  if (hparent==NULL)
    style=WS_OVERLAPPEDWINDOW|WS_CLIPCHILDREN;
  else
    style=WS_CHILD|WS_CLIPCHILDREN;
  hw=CreateWindow(EBASECLASS,name,style,
    x,y,width,height,hparent,(hparent==NULL?NULL:(HMENU)id),hinst,pe);
  if (hw==NULL)
    return NULL;                       // Unable to create window
  // Create status bar on the bottom.
  hstatus=CreateWindowEx(0,STATUSCLASSNAME,NULL,
    WS_CHILD|WS_VISIBLE,0,0,0,0,hw,(HMENU)0,hinst,NULL);
  if (hstatus!=NULL)
    SendMessage(hstatus,WM_SETFONT,(WPARAM)hfont,0);
  // Create work edit window, where all editing is done.
  GetClientRect(hw,&rc);
  GetWindowRect(hstatus,&rs);
  hedit=CreateWindowEx(
    (options & EDO_SUNKEN?WS_EX_CLIENTEDGE:0),
    EDITCLASS,name,WS_CHILD|WS_VISIBLE,
    0,0,rc.right,rc.bottom-(rs.bottom-rs.top),hw,0,hinst,pe);
  if (hedit==NULL) {
    DestroyWindow(hw);                 // Unable to create window
    return NULL; };
  // Get character size, differs from that in t_font!
  dc=GetDC(hedit);
  SelectObject(dc,hfont);
  GetTextMetrics(dc,&tm);
  if (tm.tmAveCharWidth<1 || tm.tmHeight<1) {
    pe->w.fontdx=6+EDITEXTRAX;
    pe->w.fontdy=10+EDITEXTRAY; }
  else {
    pe->w.fontdx=tm.tmAveCharWidth+EDITEXTRAX;
    pe->w.fontdy=tm.tmHeight+EDITEXTRAY; };
  ReleaseDC(hedit,dc);
  // Divide status bar into parts.
  if (hstatus!=NULL) {
    parts[0]=pe->w.fontdx*9;
    parts[1]=pe->w.fontdx*18;
    parts[2]=pe->w.fontdx*22;
    parts[3]=999999;
    SendMessage(hstatus,SB_SETPARTS,4,(LPARAM)parts); };
  // Save parameters and report success.
  pe->w.hw=hw;
  pe->w.hstatus=hstatus;
  pe->w.hedit=hedit;
  pe->w.options=options;
  pe->w.hfont=hfont;
  pe->w.offsetx=0;
  pe->w.offsety=0;
  pe->w.invalidated=0;
  SetWindowLong(hw,0,(ulong)pe);
  SetWindowLong(hedit,0,(ulong)pe);
  Updatestatus(pe);
  // Show editor and report success.
  ShowWindow(hw,SW_SHOW);
  return hw;
};

// Resets contents of the editor. Returns 0 on success and -1 on error.
// Attention, does not check whether contents is changed!
int Resetedit(t_edit *pe) {
  if (pe==NULL)
    return -1;                         // Error in input parameters
  // Clear contents.
  pe->ncard=0;
  pe->cursorx=0;
  pe->cursory=0;
  pe->selx0=pe->selx1=pe->selxb=0;
  pe->sely0=pe->sely1=pe->selyb=0;
  pe->modified=0;
  pe->ctrlx=0;
  pe->w.offsetx=0;
  pe->w.offsety=0;
  pe->filevalid=0;
  pe->nundo=pe->nredo=0;
  pe->stopundo=0;
  // Update associated window.
  if (pe->w.hedit!=NULL && pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  Updatestatus(pe);
  return 0;
};

// Adds (LEO_ADD) or replaces (LEO_REPLACE) file to the editor. Lines exceeding
// editor width are either truncated (LEO_TRUNCATE) or wrapped (LEO_WRAP) to
// the next card. Returns 0 on success and -1 on error. Attention, does not
// check whether contents is changed!
int Loadedit(t_edit *pe,char *path,int options) {
  int n,length;
  char *buf,*pb,s[TEXTLEN];
  FILE *f;
  if (pe==NULL || path==NULL || path[0]=='\0')
    return -1;                         // Error in input parameters
  if ((options & LEO_ADDMASK)==LEO_REPLACE) {
    if (Resetedit(pe)!=0) return -1; };
  // Open file.
  f=fopen(path,"rb");
  if (f==NULL) {
    Message("JASN0016E  Unable to open file '%s'",path);
    return -1; };
  // I assume that card decks are small in size. In the real world, would you
  // be able to submit a 100000-card job to the RDR?
  fseek(f,0,SEEK_END);
  length=ftell(f);
  fseek(f,0,SEEK_SET);
  buf=(char *)malloc(length+1);
  if (buf==NULL) {
    Message("JASN0017E  Low memory, unable to edit file '%s'",path);
    fclose(f);
    return -1; };
  n=fread(buf,1,length,f);
  fclose(f);
  if (n!=length) {
    Message("JASN0018E  Unable to read file '%s'",path);
    free(buf);
    return -1; };
  buf[n]='\0';

  //  // If file is in EBCDIC format, convert.
  //  if (format==FF_EBCDIC) {
  //    for (n=0; n<length; n++) buf[n]=ebcdic_to_ascii[buf[n]]; };

  // Read all cards, one by one.
  pb=buf;
  while (*pb!='\0') {
    // Read next line.
    n=0;
    while (n<pe->ncol) {
      if (*pb=='\0' || *pb=='\n' || *pb=='\r') break;
      s[n++]=*pb++; };
    // Fill rest of the line with spaces.
    while (n<pe->ncol)
      s[n++]=' ';
    // Add line to the edit buffer.
    if (pe->ncard>=pe->maxcard)
      Extendeditbuffer(pe,1);
    if (pe->ncard>=pe->maxcard) {
      Message("JASN0019E  Low memory, file '%s' incomplete",path);
      break; };
    memcpy(pe->data+pe->ncol*pe->ncard,s,pe->ncol);
    pe->ncard++;
    if ((options & LEO_ADDMASK)==LEO_ADD) pe->modified=1;
    // In the truncate mode, skip rest of the line.
    if ((options & LEO_WRAPMASK)==LEO_TRUNCATE) {
      while (*pb!='\0' && *pb!='\n' && *pb!='\r') pb++; };
    // Skip CR, LF, CR/LF or LF/CR combinations.
    if (pb[0]=='\n' && pb[1]=='\r') pb+=2;
    else if (pb[0]=='\r' && pb[1]=='\n') pb+=2;
    else if (pb[0]=='\n') pb++;
    else if (pb[0]=='\r') pb++; };
  free(buf);
  // Update associated window.
  if (pe->w.hedit!=NULL && pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE); };
  Updatestatus(pe);
  // Save file name.
  if ((options & LEO_ADDMASK)==LEO_REPLACE)
    Strcopy(pe->filename,MAXPATH,path);
  pe->filevalid=1;
  // Report success.
  return 0;
};

// Asks user to load file to the editor. Returns 0 on success, 1 when operator
// cancelled this action and -1 on error.
int Loadeditfile(t_edit *pe) {
  int result;
  if (pe==NULL)
    return -1;                         // Error in input parameters
  result=Iseditcontentschanged(pe);
  if (result!=0)
    return result;
  result=Browsefilename("Select file",pe->filename,FT_PCH,NULL,NULL,0);
  if (result!=0)
    return result;
  return Loadedit(pe,pe->filename,LEO_REPLACE);
};

// Adds line of length n (zero terminated if n is 0) to the end of the editor
// data. Returns 0 on success and -1 on any error.
int Addeditline(t_edit *pe,char *s,int n) {
  if (pe==NULL || s==NULL || n<0)
    return -1;                         // Error in input parameters
  if (Inserteditlines(pe,pe->ncard,1)<0)
    return -1;
  if (n==0)
    n=strlen(s);
  if (n>80)
    n=80;
  return Inserteditchars(pe,s,n,0,pe->ncard-1,0,1);
};

// Asks user to save contents of the editor to ASCII file. Parameter defname
// is the default file name, used when file name in the editor is undefined.
// Returns 0 on success, 1 when operator cancelled this action and -1 on error.
int Saveeditfile(t_edit *pe,int asknewname) {
  int i,j,result;
  char *pd;
  FILE *f;
  if (pe==NULL)
    return -1;                         // Error in input parameters
  if (pe->filename[0]=='\0' || asknewname!=0 || pe->filevalid==0) {
    result=Browsefilename("Save data",pe->filename,FT_PCH,NULL,NULL,1);
    if (result!=0) return result; };
  f=fopen(pe->filename,"wt");
  if (f==NULL) {
    Message("JASN0023E  Unable to create file '%s'",pe->filename);
    return -1; };
  for (i=0,pd=pe->data; i<pe->ncard; i++,pd+=pe->ncol) {
    for (j=pe->ncol; j>0; j--) {
      if (pd[j-1]!=' ') break; };
    if (j==0)
      fprintf(f,"\n");
    else
      fprintf(f,"%*.*s\n",j,j,pd);
    ;
  };
  fclose(f);
  pe->modified=0;
  Updatestatus(pe);
  return 0;
};

// Toggles edit view between form and card.
void Toggleeditview(t_edit *pe) {
  if (pe==NULL)
    return;                            // Error in input parameters
  pe->mode^=EDM_CARD;
  Removeeditselection(pe);
  if (pe->w.hedit!=NULL && pe->w.invalidated==0) {
    pe->w.invalidated=1;
    InvalidateRect(pe->w.hedit,NULL,FALSE);
  };
};

// If contents of the edit window was changed, ask user whether to save data.
// Returns 0 if data is unchanged, was saved or changes to be discarded, 1 if
// operator cancelled action and -1 on error.
int Iseditcontentschanged(t_edit *pe) {
  int result;
  if (pe==NULL)
    return -1;                         // Error in input parameters
  if (pe->modified==0)
    return 0;                          // Data was not modified
  if (pe->mode & EDM_NOEDIT)
    return 0;                          // Non-editable data
  result=MessageBox(hwmain,"Data in editor was changed. Save it to disk?",
    "Data changed",MB_ICONWARNING|MB_YESNOCANCEL|MB_DEFBUTTON1);
  if (result==IDCANCEL)
    return 1;
  if (result==IDNO)
    return 0;
  return Saveeditfile(pe,0);
};

