////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//      JASON - FULLY GRAPHICAL INTERFACE TO HERCULES IBM 360+ EMULATOR       //
//                                                                            //
//                           INTERFACE TO HERCULES                            //
//                                                                            //
//                             Start: 10.02.2010                              //
//                        Current version: 30.05.2010                         //
//                                                                            //
// Jason is free software; you can redistribute it and/or modify it under     //
// the terms of the GNU General Public License version 3, as published by the //
// Free Software Foundation.                                                  //
//                                                                            //
// Jason is distributed in the hope that it will be useful, but WITHOUT ANY   //
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS  //
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more      //
// details.                                                                   //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program. If not, see <http://www.gnu.org/licenses/>.             //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#pragma hdrstop
#include <stdio.h>
#include <dir.h>

#include <shlwapi.h>

#include "Jason.h"


////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////// PIPES /////////////////////////////////////

// Pipes connect Hercules with Jason. There are three general-purpose unnamed
// pipes: stdin, stdout, stderr. Additionally, there are named pipes for every
// printer and puncher in the Hercules. As Hercules writes data by records
// (strings), the pipes accept data also by strings of either variable size
// (recsize=0, CR/LF being end of the record) or fixed size (recsize!=0). If
// file is specified, data read from the pipe is automatically attached 1:1 to
// this file.

// Closes pipe and resets descriptor. Note that data remaining in the pipe is
// discarded.
void Closepipe(t_pipe *pp) {
  if (pp==NULL)
    return;                            // Error in input data
  // Close handles.
  if (pp->hfile!=NULL)
    CloseHandle(pp->hfile);
  if (pp->hpipe!=NULL)
    CloseHandle(pp->hpipe);
  // Clear decsriptor.
  ZeroMemory(pp,sizeof(t_pipe));
};

// Opens or reopens the named pipe. Pipe name must have form \\.\pipe\xxx. If
// name is NULL or empty, creates unique name. Mode is one of PIPE_ACCESS_xxx.
// If mode is 0, Readpipe() reads in binary record mode (i.e exactly as many
// characters as specified or none), mode 1 - text mode (till the next CR, LF
// or FF, skipping CR/LF and returning FF as a separate line), mode 2 - binary
// variable (as many data as available). Returns 0 on success and -1 on error.
int Openpipe(t_pipe *pp,char *pipename,char *filename,int pipemode,int mode) {
  if (pp==NULL)
    return -1;                         // Error in input data
  if (pipemode!=PIPE_ACCESS_DUPLEX && pipemode!=PIPE_ACCESS_OUTBOUND &&
    pipemode!=PIPE_ACCESS_INBOUND)
    return -1;                         // Invalid pipe mode
  if (mode<0 || mode>2)
    return -1;                         // Invalid mode
  // Disconnect if pipe is already open. Call to Closepipe() zeroes the whole
  // descriptor (but leaves system buffers intact).
  Closepipe(pp);
  // On request, create unique pipe name.
  if (pipename==NULL || pipename[0]=='\0')
    sprintf(pp->pipename,"\\\\.\\pipe\\JSN%08X_%08X_%02i",
    GetCurrentProcessId(),GetTickCount(),uniquenumber++);
  else
    strcpy(pp->pipename,pipename);
  // Open pipe.
  pp->hpipe=CreateNamedPipe(pp->pipename,pipemode,
    PIPE_TYPE_BYTE|PIPE_READMODE_BYTE|PIPE_NOWAIT,1,65536,65536,1,NULL);
  if (pp->hpipe==NULL || pp->hpipe==INVALID_HANDLE_VALUE) {
    pp->hpipe=NULL;                    // Unable to create named pipe
    Message("JASN0002E Unable to create pipe '%s'",pp->pipename);
    return -1; };
  pp->mode=mode;
  // Open or create output file, if such file is specified. Inability to open
  // file is not a fatal error.
  if (filename!=NULL && filename[0]!='\0') {
    pp->hfile=CreateFile(filename,GENERIC_WRITE,0,
      NULL,OPEN_ALWAYS,FILE_ATTRIBUTE_NORMAL,NULL);
    if (pp->hfile==NULL || pp->hfile==INVALID_HANDLE_VALUE) {
      pp->hfile=NULL;                  // Unable to create output file
      Message("JASN0003E Unable to create file '%s'",filename); }
    else
      strcpy(pp->filename,filename);
    ;
  };
  // Report success.
  return 0;
};

// Attempts to get data from the pipe according to the mode specified in call
// to Openpipe(). If mode is 0, Readpipe() reads in binary record mode (i.e
// exactly n characters or none), mode 1 - text mode (till the next CR, LF or
// FF, skipping CR/LF and treating FF as a separate line) but not more than n
// characters including terminal zero, mode 2 - binary variable (as many data
// as available, up to n). Returns length of data on success (may be 0) and -1
// on any error or if there is no data in the text mode, to distinguish from
// the empty line.
int Readpipe(t_pipe *pp,char *s,int n) {
  int i,j,ndata,nskip,complete;
  ulong maxin,nin,nout;
  char *pbuf;
  if (pp==NULL || pp->hpipe==NULL || s==NULL || n<1)
    return -1;                         // Error in input parameters
  if (pp->mode==1 && n<2)
    return -1;                         // Buffer is too small
  if (n>NBUF)
    n=NBUF;
  // Check whether we have new data and read it. Note that PeekNamedPipe()
  // works with unnamed pipes, too.
  if (PeekNamedPipe(pp->hpipe,NULL,0,NULL,&maxin,NULL)!=0) {
    if ((int)maxin>NBUF-pp->nbuf)
      maxin=NBUF-pp->nbuf;
    if (maxin>0) {
      if (ReadFile(pp->hpipe,pp->buf+pp->nbuf,maxin,&nin,NULL)!=0) {
        if (pp->hfile!=NULL)
          WriteFile(pp->hfile,pp->buf+pp->nbuf,nin,&nout,NULL);
        pp->nbuf+=nin;
      };
    };
  };
  // Check whether we have complete record.
  pbuf=pp->buf;
  if (pp->mode==0) {
    // Fixed binary records.
    if (pp->nbuf<n) return 0;
    ndata=nskip=n;
    memcpy(s,pbuf,ndata); }
  else if (pp->mode==2) {
    // Variable binary records.
    if (pp->nbuf==0) return 0;
    ndata=nskip=(pp->nbuf>=n?n:pp->nbuf);
    memcpy(s,pbuf,ndata); }
  else {
    // Text mode, till next CR, LF or FF. IBM printers were able to simulate
    // bold characters and emulate APL symbols by printing over the same line
    // several times, but my lists are not as good. The processing is not very
    // effective, but usually data does not stay long in buffer.
    ndata=j=0;
    complete=0;
    // In this loop, ndata is the length of string and j is the current
    // character position.
    for (i=0; i<pp->nbuf; i++) {
      if (pbuf[i]=='\r')
        j=0;                           // CR
      else if (pbuf[i]=='\n') {
        i++; complete=1;               // Skip LF and finish line
        break; }
      else if (pbuf[i]=='\f') {
        if (ndata==0) {
          s[0]='\f'; ndata=1; i++; };  // Report LF as a separate string
        complete=1;
        break; }
      else if (j<ndata) {
        if (s[j]==' ')
          s[j]=pbuf[i];                // Overwrite space from previous pass
        else if (s[j]!=pbuf[i])
          s[j]=0xDB;                   // Black square
        j++; }
      else {
        if (j>=n-1) {
          complete=1;                  // Line is too long, split
          break; };
        s[j]=pbuf[i];                  // Next character
        j++; ndata=j;
      };
    };
    if (complete==0)
      return -1;                       // Incomplete line, wait longer
    s[ndata]='\0';
    nskip=i;
  };
  // Update buffer.
  if (nskip<pp->nbuf)
    memmove(pp->buf,pp->buf+nskip,pp->nbuf-nskip);
  pp->nbuf-=nskip;
  return ndata;
};


////////////////////////////////////////////////////////////////////////////////
//////////////////////////// INTERFACE TO HERCULES /////////////////////////////

static HANDLE    hprocess;             // Handle of Hercules process
static HANDLE    hthread;              // Handle of main Hercules thread
static ulong     hercprocid=0;         // ID of Hercules process

// Hercules's stdin. Here Jasons sends commands to Hercules.
static HANDLE    stdinread;            // Read end of stdin pipe
static HANDLE    stdinwrite;           // Write end of stdin pipe
// Hercules's stdout (Hercules console).
static HANDLE    stdoutread;           // Read end of stdout pipe
static HANDLE    stdoutwrite;          // Write end of stdout pipe
// Hercules's stderr, passes data to external GUI (in our case, Jason).
static HANDLE    stderrread;           // Read end of stderr pipe
static HANDLE    stderrwrite;          // Write end of stderr pipe

// Extracts relevant parameters from the configuration file. On error, sets
// defaults.
void Extractcnfdata(char *path) {
  int n;
  char s[TEXTLEN],*ps;
  FILE *f;
  cpumodel=0;
  archmode[0]='\0';
  cnslport=3270;
  if (path==NULL || path[0]=='\0')
    return;                            // Error in input parameters
  f=fopen(path,"rt");
  if (f==NULL)
    return;                            // Wrong name of the configuration file
  while (1) {
    if (fgets(s,TEXTLEN,f)==NULL)
      break;                           // Reached end of file
    // Skip leading spaces.
    ps=s;
    while (*ps==' ' || *ps=='\t') ps++;
    if (strnicmp(ps,"CPUMODEL ",9)==0 || strnicmp(ps,"CPUMODEL=",9)==0) {
      ps+=9;
      while (*ps!='\0' && !isdigit(*ps)) ps++;
      while (*ps=='0') ps++;           // Don't interprete as octal!
      sscanf(ps,"%i",&cpumodel); }
    else if (strnicmp(ps,"ARCHMODE ",9)==0 || strnicmp(ps,"ARCHMODE=",9)==0) {
      ps+=9;
      while (*ps==' ' || *ps=='\t') ps++;
      if ((*ps=='s' || *ps=='S') && ps[1]=='/')
        ps+=2;
      for (n=0; n<SHORTNAME-8; n++,ps++) {
        if (isgraph(*ps)==0) break;
        archmode[n]=*ps; };
      archmode[n]='\0'; }
    else if (strnicmp(ps,"CNSLPORT ",9)==0 || strnicmp(ps,"CNSLPORT=",9)==0) {
      ps+=9;
      while (*ps!='\0' && !isdigit(*ps)) ps++;
      sscanf(ps,"%i",&cnslport);
      if (cnslport<1 || cnslport>65535) cnslport=3270;
    };
  };
  fclose(f);
};

// Creates new environment for the Hercules and utilities, adding their path to
// PATH. In this way, the whole installation is moveable. On success, returns
// pointer to new environment that must be freed by call to GlobalFree(). On
// error, returns NULL.
char *Createnewenvironment(void) {
  int k,l,n,nenv;
  char *penv,*newenv;
  penv=GetEnvironmentStrings();
  if (penv==NULL)
    return NULL;
  // Calculate memory size necessary for new environment.
  nenv=0;
  while (1) {
    n=strlen(penv+nenv);
    if (n==0) break;
    nenv+=n+1; };
  nenv+=MAXPATH+32;
  newenv=(char *)GlobalAlloc(GMEM_FIXED,nenv);
  if (newenv!=NULL) {
  // Walk old environment and copy strings to new.
    nenv=k=0;
    while (1) {
      n=strlen(penv+nenv);
      if (n==0) break;
      memcpy(newenv+k,penv+nenv,n); k+=n;
      if (strnicmp(penv+nenv,"PATH=",5)==0) {
        if (newenv[k-1]!=';') newenv[k++]=';';
        l=strlen(hercdir);
        if (l>1 && hercdir[l-1]=='\\') l--;
        memcpy(newenv+k,hercdir,l); k+=l; };
      newenv[k++]='\0';
      nenv+=n+1; };
    newenv[k]='\0';
  };
  FreeEnvironmentStrings(penv);
  return newenv;
};

// Starts Hercules. On success, returns 0. If Hercules is already running,
// returns 1. If Hercules is not available or does not start, returns -1. This
// error is not critical, simply repeat attempt later.
int Starthercules(void) {
  int result;
  char cmdline[MAXPATH+TEXTLEN],drv[MAXDRIVE],dir[MAXDIR];
  char *newenv;
  SECURITY_ATTRIBUTES sa;
  STARTUPINFO si;
  PROCESS_INFORMATION pi;
  // Check whether Hercules is already running.
  if (hercprocid!=0)
    return 1;                          // Yep, already here
  // Create pipes that replace stdin, stdout and stderr for Hercules, if not
  // yet done on the previous call.
  sa.nLength=sizeof(sa);
  sa.lpSecurityDescriptor=NULL;
  sa.bInheritHandle=TRUE;
  if (stdinwrite==NULL)
    CreatePipe(&stdinread,&stdinwrite,&sa,4096);
  if (stdoutread==NULL)
    CreatePipe(&stdoutread,&stdoutwrite,&sa,65536);
  if (stderrread==NULL)
    CreatePipe(&stderrread,&stderrwrite,&sa,65536);
  if (stdinwrite==NULL || stdoutread==NULL || stderrread==NULL) {
    Message("JASN0006E Unable to create standard pipes");
    return -1; };
  // Prepare for call. Note that Hercules doesn't like incomplete paths.
  if (fileexists(hercpath)==0) {
    Message("JASN0004E Hercules executable absent: '%s'",hercpath);
    return -1; };
  if (fileexists(herccfg)==0) {
    Message("JASN0005E Hercules configuration file absent: '%s'",herccfg);
    return -1; };
  // Get path to Hercules.
  fnsplit(hercpath,drv,dir,NULL,NULL);
  fnmerge(hercdir,drv,dir,NULL,NULL);
  // Create new environment for the Hercules.
  newenv=Createnewenvironment();
  // Prepare Hercules launch.
  sprintf(cmdline,"%s -f\"%s\" EXTERNALGUI",hercpath,herccfg);
  memset(&si,0,sizeof(si));
  si.cb=sizeof(si);
  si.dwFlags=STARTF_USESTDHANDLES;
  si.hStdInput=stdinread;
  si.hStdOutput=stdoutwrite;
  si.hStdError=stderrwrite;
  // Try to start Hercules.
  result=CreateProcess(NULL,cmdline,NULL,NULL,TRUE,
    CREATE_NO_WINDOW,newenv,hercdir,&si,&pi);
  if (newenv!=NULL)
    GlobalFree((HGLOBAL)newenv);
  if (result==0) {                     // Oops, process does not start!
    Message("JASN0007E Unable to start Hercules");
    return -1; };
  hprocess=pi.hProcess;
  hthread=pi.hThread;
  hercprocid=pi.dwProcessId;
  hercrunning=1; Redrawcpupanel();
  // Initialize t_pipe structures associeted with stdout and stderr from
  // Hercules.
  Closepipe(&outpipe);
  strcpy(outpipe.pipename,"stdout");   // For identification only
  outpipe.hpipe=stdoutread;
  outpipe.mode=1;
  Closepipe(&errpipe);
  strcpy(errpipe.pipename,"stderr");   // For identification only
  errpipe.hpipe=stderrread;
  errpipe.mode=1;
  // Close handles passed to the Hercules.
  if (stdinread!=NULL) {
    CloseHandle(stdinread); stdinread=NULL; };
  if (stdoutwrite!=NULL) {
    CloseHandle(stdoutwrite); stdoutwrite=NULL; };
  if (stderrwrite!=NULL) {
    CloseHandle(stderrwrite); stderrwrite=NULL; };
  // Request full register status for more fun. Jason is all for fun, isn't it?
  Sendstdin("]GREGS=1");
  Sendstdin("]CREGS=1");
  Sendstdin("]AREGS=1");
  Sendstdin("]FREGS=1");
  // Report success.
  return 0;
};

// Sends command to Hercules and follows it by line feed. Returns 0 on success
// and -1 on error.
int Sendstdin(char *format,...) {
  char s[2*TEXTLEN+MAXPATH];
  ulong l,n;
  va_list ap;
  if (format==NULL)
    return -1;                         // Error in input parameters
  if (stdinwrite==NULL)
    return -1;                         // No connection to Hercules
  va_start(ap,format);
  l=vsprintf(s,format,ap);
  if (WriteFile(stdinwrite,s,l,&n,NULL)==0)
    return -1;
  WriteFile(stdinwrite,"\n",1,&n,NULL);
  va_end(ap);
  return 0;
};

// Stops Hercules, then kills it and releases all handles. Call this
// unconditionally at the end of the session.
void Stophercules(void) {
  ulong t;
  // Stop Hercules and wait at most 10 seconds till it terminates.
  Sendstdin("EXIT");
  t=GetTickCount();
  while (hercrunning!=0) {
    Interface();
    Devicestep();
    if (GetTickCount()-t>10000) break; };
  if (hprocess!=NULL) {
    TerminateProcess(hprocess,-1);
    CloseHandle(hprocess); hprocess=NULL; };
  if (hthread!=NULL) {
    CloseHandle(hthread); hthread=NULL; };
  hercprocid=0;
  if (stdinread!=NULL) {
    CloseHandle(stdinread); stdinread=NULL; };
  if (stdinwrite!=NULL) {
    CloseHandle(stdinwrite); stdinwrite=NULL; };
  if (stdoutread!=NULL) {
    CloseHandle(stdoutread); outpipe.hpipe=stdoutread=NULL; };
  if (stdoutwrite!=NULL) {
    CloseHandle(stdoutwrite); stdoutwrite=NULL; };
  if (stderrread!=NULL) {
    CloseHandle(stderrread); errpipe.hpipe=stderrread=NULL; };
  if (stderrwrite!=NULL) {
    CloseHandle(stderrwrite); stderrwrite=NULL; };
  hercrunning=0;
};


////////////////////////////////////////////////////////////////////////////////
/////////////////////// INTERFACE TO HERCULES UTILITIES ////////////////////////

// Executes utility, passing stdout to the list. If filter is presented and
// some stdout line contains its value as a case-sensitive substring, fills
// out with the rest of the line. Any following occurance is ignored. If flag
// nostderr is set, data written to stderr is ignored. This makes call faster.
// Returns 0 on success and -1 on error. Note similarities to Starthercules().
int Callutility(char *utility,char *parms,t_list *pl,char *filter,char *out,
  int nostderr) {
  int i,nso,nse,err,skipo,skipe;
  ulong t,status,nread,maxread;
  char path[MAXPATH],drv[MAXDRIVE],dir[MAXDIR],nam[MAXFILE],ext[MAXEXT];
  char cmdline[MAXPATH+TEXTLEN],*newenv,*pf;
  char so[BUFLEN],se[BUFLEN];
  HANDLE inread,inwrite,outread,outwrite,errread,errwrite;
  SECURITY_ATTRIBUTES sa;
  STARTUPINFO si;
  PROCESS_INFORMATION pi;
  if (utility==NULL || utility[0]=='\0' || pl==NULL)
    return -1;                         // Error in input parameters
  // Initialize variables.
  inread=inwrite=outread=outwrite=errread=errwrite=NULL;
  newenv=NULL;
  pi.hProcess=pi.hThread=INVALID_HANDLE_VALUE;
  status=STILL_ACTIVE;
  if (out==NULL)
    filter=NULL;
  else
    out[0]='\0';
  err=-1;
  // Create pipes that replace stdin, stdout and stderr for the utility.
  sa.nLength=sizeof(sa);
  sa.lpSecurityDescriptor=NULL;
  sa.bInheritHandle=TRUE;
  CreatePipe(&inread,&inwrite,&sa,4096);
  CreatePipe(&outread,&outwrite,&sa,65536);
  CreatePipe(&errread,&errwrite,&sa,65536);
  if (inwrite==NULL || outread==NULL || errread==NULL)
    goto finish;
  // Get utility path.
  fnsplit(utility,drv,dir,nam,ext);
  if (drv[0]=='\0' && dir[0]=='\0')
    fnsplit(hercpath,drv,dir,NULL,NULL);
  fnmerge(path,drv,dir,nam,ext);
  // Create new environment for the utility.
  newenv=Createnewenvironment();
  // Prepare launch.
  if (parms==NULL || parms[0]=='\0')
    Strcopy(cmdline,MAXPATH,path);
  else
    sprintf(cmdline,"%s %s",path,parms);
  memset(&si,0,sizeof(si));
  si.cb=sizeof(si);
  si.dwFlags=STARTF_USESTDHANDLES;
  si.hStdInput=inread;
  si.hStdOutput=outwrite;
  si.hStdError=errwrite;
  // Start utility.
  if (CreateProcess(NULL,cmdline,NULL,NULL,TRUE,
    CREATE_NO_WINDOW,newenv,hercdir,&si,&pi)==0)
    goto finish;                       // Oops, utility does not start!
  // Give utility time to init. Some utilities log their titles to stderr, and
  // without pause it's possible that titles will come after data.
  if (nostderr==0)
    Sleep(100);
  // For at most 15 seconds, read data and pass it to the list.
  nso=nse=0; skipo=skipe=0;
  t=GetTickCount();
  while (1) {
    if (GetExitCodeProcess(pi.hProcess,&status)==0)
      status=STILL_ACTIVE;
    // Get as much data from stderr as possible.
    while (nse<BUFLEN) {
      if (PeekNamedPipe(errread,NULL,0,NULL,&maxread,NULL)==0)
        break;
      if (maxread<=0)
        break;
      if (ReadFile(errread,se+nse,maxread,&nread,NULL)==0)
        break;
      if (nread==0)
        break;
      nse+=nread; };
    if (nse==BUFLEN)
      status=STILL_ACTIVE;
    // Extract as many lines as possible from the received data.
    if (nostderr)
      nse=0;
    else {
      while (nse>0) {
        if (se[0]==skipe)
          i=1;                         // Skip rest of separator
        else {
          for (i=0; i<nse; i++) {
            if (se[i]=='\n' || se[i]=='\r') break; };
          if (i>=nse)
            break;
          skipe=(se[i]=='\n'?'\r':'\n');
          Addtolist(pl,se,i,2,0,0);
          i++; };
        if (i<nse) memmove(se,se+i,nse-i);
        nse-=i;
      };
    };
    // Get as much data from stdout as possible.
    while (nso<BUFLEN) {
      if (PeekNamedPipe(outread,NULL,0,NULL,&maxread,NULL)==0)
        break;
      if (maxread<=0)
        break;
      if (ReadFile(outread,so+nso,maxread,&nread,NULL)==0)
        break;
      if (nread==0)
        break;
      nso+=nread; };
    if (nso==BUFLEN)
      status=STILL_ACTIVE;
    // Extract as many lines as possible from the received data.
    while (nso>0) {
      if (so[0]==skipo)
        i=1;                           // Skip rest of separator
      else {
        for (i=0; i<nso; i++) {
          if (so[i]=='\n' || so[i]=='\r') break; };
        if (i>=nso)
          break;
        skipo=(so[i]=='\n'?'\r':'\n');
        // Check for filter.
        if (filter!=NULL && filter[0]!='\0' && out[0]=='\0') {
          so[i]='\0';
          pf=strstr(so,filter);
          if (pf!=NULL) {
            pf+=strlen(filter);
            while (*pf==' ') pf++;
            Strcopy(out,TEXTLEN,pf);
          };
        };
        Addtolist(pl,so,i,1,0,0);
        i++; };
      if (i<nso) memmove(so,so+i,nso-i);
      nso-=i; };
    // Exit if utility terminated or on timeout.
    if (status!=STILL_ACTIVE || GetTickCount()-t>=15000) break;
  };
  // Kill process if it's still here.
  if (status==STILL_ACTIVE)
    TerminateProcess(pi.hProcess,0);
  // Report success.
  err=0;
finish:
  // Clean up and exit.
  if (pi.hProcess!=INVALID_HANDLE_VALUE) CloseHandle(pi.hProcess);
  if (pi.hThread!=INVALID_HANDLE_VALUE) CloseHandle(pi.hThread);
  if (newenv!=NULL) GlobalFree((HGLOBAL)newenv);
  if (inread!=NULL) CloseHandle(inread);
  if (inwrite!=NULL) CloseHandle(inwrite);
  if (outread!=NULL) CloseHandle(outread);
  if (outwrite!=NULL) CloseHandle(outwrite);
  if (errread!=NULL) CloseHandle(errread);
  if (errwrite!=NULL) CloseHandle(errwrite);
  return err;
};


////////////////////////////////////////////////////////////////////////////////
////////////////////////////// MESSAGE PROCESSING //////////////////////////////

static int       devchanged;           // List of devices is changed, DCH_xxx

// Checks whether string s, at least TEXTLEN bytes long, is a device status
// notification from the Hercules (dyngui) and processes it. Returns 1 if
// message is processed and 0 otherwise. Old-style notifications are not
// supported.
static int Processdevicestatus(char *s) {
  int i,n,slot,cuu,devtype,devclass;
  char devonline,devbusy,devpending,devopen,state;
  char devclassname[8],*ps,*pa;
  t_device *pdev;
  if (s==NULL)                         // Error in input parameters
    return 0;
  // Check whether Hercules terminated.
  if (memcmp(s,"HHCIN099I",9)==0) {    // Hercules terminated
    hercrunning=0;
    Redrawcpupanel(); }
  // Device configuration messages from Hercules come in packs "DEVA=..." ...
  // "DEVC=..." followed by end-of-list marker "DEVX=".
  else if (memcmp(s,"DEVX=",5)==0) {   // End of device list
    if (devchanged!=0) {
      Updatedevices();
      devchanged=0;
    }; }
  // Check whether some device was added or changed.
  else if (memcmp(s,"DEVA=",5)==0 || memcmp(s,"DEVC=",5)==0) {
    n=sscanf(s+5,"%4X %4i %4s %c%c%c%c",
      &cuu,&devtype,devclassname,&devonline,&devbusy,&devpending,&devopen);
    if (n!=7 || cuu<0 || cuu>65535 || devtype<0 || devtype>9999 ||
      (devonline!='0' && devonline!='1') ||
      (devbusy!='0' && devbusy!='1') ||
      (devpending!='0' && devpending!='1') ||
      (devopen!='0' && devopen!='1'))
      return 1;                        // Bad data, ignore notification
    devonline-='0';
    devbusy-='0';
    devpending-='0';
    devopen-='0';
    // Assure that we have at least one free slot for the new device. I add 8
    // new slots at once.
    if (ndevice>=maxdevice) {
      pdev=(t_device *)malloc((maxdevice+8)*sizeof(t_device));
      if (pdev!=NULL) {
        memcpy(pdev,device,maxdevice*sizeof(t_device));
        free(device);
        device=pdev;
        maxdevice=maxdevice+8;
      };
    };
    // Check whether device is here. Note that several first devices are
    // reserved (bookshelf, Hercules console, CPU).
    slot=NDEVRESERVED;
    for (pdev=device+slot; slot<ndevice; slot++,pdev++) {
      if (pdev->cuu==cuu) break; };
    if (slot>=maxdevice) return 1;     // Low memory, ignore notification
    // Convert device class from string to integer.
    if (stricmp(devclassname,"RDR")==0)
      devclass=DEV_RDR;
    else if (stricmp(devclassname,"PCH")==0)
      devclass=DEV_PCH;
    else if (stricmp(devclassname,"PRT")==0)
      devclass=DEV_PRT;
    else if (stricmp(devclassname,"CON")==0)
      devclass=DEV_CON;
    else if (stricmp(devclassname,"DSP")==0)
      devclass=DEV_DSP;
    else if (stricmp(devclassname,"DASD")==0)
      devclass=DEV_DASD;
    else if (stricmp(devclassname,"TAPE")==0)
      devclass=DEV_TAPE;
    else
      devclass=DEV_UNKNOWN;
    // Get path and strip parameters.
    ps=s+24;
    while (*ps==' ') ps++;
    pa=getargs(ps);
    if (pa!=NULL && strnicmp(pa,"ro ",3)==0)
      pdev->protect=1;
    else
      pdev->protect=0;
    removeargs(ps);
    // Some configurations may contain slashes instead of backslashes.
    for (i=0; ps[i]!=0 && i<MAXPATH; i++) {
      if (ps[i]=='/') ps[i]='\\'; };
    if (pdev->devclass==DEV_NONE) {
      // Device is not yet in the list (may be placeholder), add it.
      memset(pdev,0,sizeof(t_device));
      pdev->devclass=devclass;
      pdev->cuu=cuu;
      pdev->devtype=devtype;
      pdev->state=0;
      if (devonline!=0) pdev->state|=DS_ONLINE;
      if (devbusy!=0) pdev->state|=DS_BUSY;
      if (devpending!=0) pdev->state|=DS_PENDING;
      if (devopen!=0) pdev->state|=DS_OPEN;
      strncpy(pdev->file,ps,MAXPATH);
      pdev->changed=DCH_NEW|DCH_STATE|DCH_FILE;
      if (slot==ndevice) ndevice++; }
    else {
      // Device already exists, check for changes.
      if (devclass!=pdev->devclass || devtype!=pdev->devtype) {
        // Type of device connected on cuu has changed.
        pdev->devclass=devclass;
        pdev->devtype=devtype;
        sprintf(pdev->devname,"%04i at %03X",devtype,cuu);
        pdev->changed|=DCH_REPLACED; };
      state=0;
      if (devonline!=0) state|=DS_ONLINE;
      if (devbusy!=0) state|=DS_BUSY;
      if (devpending!=0) state|=DS_PENDING;
      if (devopen!=0) state|=DS_OPEN;
      if (pdev->state!=state) {
        // State of the device has changed.
        pdev->state=state;
        pdev->changed|=DCH_STATE; };
      if (stricmp(pdev->file,ps)!=0) {
        // Attached file has changed.
        strncpy(pdev->file,ps,MAXPATH);
        pdev->changed|=DCH_FILE;
      };
    };
    devchanged|=pdev->changed;
    // Hercules seems to mix sockets if several terminals attempt to attach at
    // once, therefore I try to initialize them sequentially.
    if ((pdev->changed & DCH_NEW)!=0 && pdev->devclass==DEV_DSP) {
      Updatedevices();
      for (i=0; i<10; i++) {
        Devicestep();
        Sleep(1);
      };
    };
    pdev->changed|=DCH_CONFIRMED; }

  //  else if (memcmp(s,"DEVD=",5)==0) {   // Device is removed
  //    DCH_DELETED
  //  }

  else                                 // No match
    return 0;
  return 1;
};

// Checks whether string s, at least TEXTLEN bytes long, is a CPU status
// notification from the Hercules (dyngui). Returns 1 if message is processed
// and 0 otherwise. Note that the information about registers is not important
// to Jason, it blindly copies transmitted strings to the selected positions
// in the display buffer.
int Processcpustatus(char *s) {
  int k,x,y,n,split;
  if (s==NULL)                         // Error in input parameters
    return 0;
  split=-1;
  // Variable k is the start position, x is the column and y is the line in CPU
  // list, and n is the length of the name.
  if (memcmp(s,"STATUS=",7)==0) {
    k=7; x=0; y=0; n=0; split=63;      // CPU, PSW, status & instruction count
    sscanf(s+19,"%X %X",&cpuip,&cpupsw);
    Redrawcpupanel(); }
  else if (memcmp(s,"GR0-3=",6)==0) {
    k=0; x=0; y=3; n=6;                // General registers 0..3
    sscanf(s+n,"%X %X %X %X",cpuregs+0,cpuregs+1,cpuregs+2,cpuregs+3);
    Redrawcpupanel(); }
  else if (memcmp(s,"GR4-7=",6)==0) {
    k=0; x=0; y=4; n=6; }              // General registers 4..7
  else if (memcmp(s,"GR8-B=",6)==0) {
    k=0; x=0; y=5; n=6; }              // General registers 8..11
  else if (memcmp(s,"GRC-F=",6)==0) {
    k=0; x=0; y=6; n=6; }              // General registers 12..15
  else if (memcmp(s,"FR0-2=",6)==0) {
    k=0; x=0; y=8; n=6; }              // Floating-point registers 0 and 2
  else if (memcmp(s,"FR4-6=",6)==0) {
    k=0; x=0; y=9; n=6; }              // Floating-point registers 4 and 6
  else if (memcmp(s,"CR0-3=",6)==0) {
    k=0; x=0; y=11; n=6; }             // Control registers 0..3
  else if (memcmp(s,"CR4-7=",6)==0) {
    k=0; x=0; y=12; n=6; }             // Control registers 4..7
  else if (memcmp(s,"CR8-B=",6)==0) {
    k=0; x=0; y=13; n=6; }             // Control registers 8..11
  else if (memcmp(s,"CRC-F=",6)==0) {
    k=0; x=0; y=14; n=6; }             // Control registers 12..15
  else if (memcmp(s,"AR0-3=",6)==0) {
    k=0; x=0; y=16; n=6; }             // A registers 0..3
  else if (memcmp(s,"AR4-7=",6)==0) {
    k=0; x=0; y=17; n=6; }             // A registers 4..7
  else if (memcmp(s,"AR8-B=",6)==0) {
    k=0; x=0; y=18; n=6; }             // A registers 8..11
  else if (memcmp(s,"ARC-F=",6)==0) {
    k=0; x=0; y=19; n=6; }             // A registers 12..15
  else if (memcmp(s,"MIPS=",5)==0) {
    k=0; x=0; y=21; n=5; }             // CPU speed
  else if (memcmp(s,"SIOS=",5)==0) {
    k=0; x=16; y=21; n=5; }            // Peripheral speed
  else if (memcmp(s,"SYS=",4)==0) {
    k=0; x=32; y=21; n=4; }            // SYS light
  else if (memcmp(s,"CPUPCT=",7)==0) {
    k=0; x=48; y=21; n=7; }            // CPU utilization

  //  MAINSTOR=
  //  MAINSIZE=
  //  MAN=     // Stopped (1) or running (0)
  //  LOAD=    // CPU load state

  else
    // No match, probably notification is for devices.
    return 0;
  Setlistposition(cpulist,x,y);
  if (n>0) Addtolist(cpulist,s+k,n,1,0,0);
  if (split<=k+n)
    Addtolist(cpulist,s+k+n,0,2,0,0);
  else {
    Addtolist(cpulist,s+k+n,split-k-n,2,0,0);
    Setlistposition(cpulist,x,y+1);
    Addtolist(cpulist,s+split,0,2,0,0); };
  return 1;
};

// Processes messages and status notifications from Hercules's stdout and
// stderr. Returns 0 if there were no messages or notifications and 1 otherwise.
int Interface(void) {
  int received,processed;
  char s[TEXTLEN];
  received=0;
  // Get and display all Hercules console messages.
  while (Readpipe(&outpipe,s,TEXTLEN)>=0) {
    Addtolist(herclist,s,0,2,0,0);
    received=1; };
  // Get at most one status message, for the case of the old-style interface
  // that constantly sends too much data.
  if (Readpipe(&errpipe,s,TEXTLEN)<=0)
    return received;                   // No or empty status message
  // Process message.
  processed=Processcpustatus(s);
  if (processed==0)
    processed=Processdevicestatus(s);
  // Unrecognized status message from Hercules, report.
  if (processed==0)
    Message("JASN0001W Unknown message '%s'",s);
  return 1;
};

