////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//      JASON - FULLY GRAPHICAL INTERFACE TO HERCULES IBM 360+ EMULATOR       //
//                                                                            //
//                       LIST TEXT BUFFERS AND WINDOWS                        //
//                                                                            //
//                             Start: 11.02.2010                              //
//                        Current version: 19.03.2010                         //
//                                                                            //
// Jason is free software; you can redistribute it and/or modify it under     //
// the terms of the GNU General Public License version 3, as published by the //
// Free Software Foundation.                                                  //
//                                                                            //
// Jason is distributed in the hope that it will be useful, but WITHOUT ANY   //
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS  //
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more      //
// details.                                                                   //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program. If not, see <http://www.gnu.org/licenses/>.             //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#pragma hdrstop
#include <stdio.h>
#include <dir.h>

#include "Jason.h"


////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// LIST /////////////////////////////////////

// List is a scrollable or fixed text buffer that keeps its data in a memory.
// Its contents can be displayed simultaneously in up to NLISTWIN windows. A
// single row of text in the list is limited to TEXTLEN-1 (but not more than
// 255) characters. Each symbol in the fixed list may have its own attributes.
// Attributes in the scrollable list can be assigned only to the row as a whole.
//
// Data in the scrollable list consists of the following unaligned sequences:
//
//   t_lrechdr                         // Record header
//   data[t_lrechdr.size]              // Data itself
//   lrechdr.size                      // Record size for bidirectional walk
//
// Data in the fixed lists consists of t_lsymbol's. In fact, these two types
// are quite distinct and could be easily separated in two.

#define TIMER_CURSOR   2               // ID of blinking cursor timer
#define CURSOR_TIME    750             // Cursor blinking period, ms

// Record headers are used only for the scrollable lists.
typedef struct t_lrechdr {             // Header of list record in memory
  uchar          size;                 // Size of record data
  uchar          fgcolor;              // Index of foreground color
  uchar          bgcolor;              // Background color
  uchar          features;             // Line features, set of TF_xxx
} t_lrechdr;

// Symbol descriptors are used only for the fixed lists.
typedef struct t_lsymbol {
  uchar          fgcolor;              // Index of foreground color
  uchar          bgcolor;              // Background color
  uchar          features;             // Symbol features, set of TF_xxx
  char           symbol;               // Symbol to display
} t_lsymbol;

// The following 16-color scheme will be used as default if no other specified.
static COLORREF  defcolors[16] = {     // List of default colors (EGA)
  0x00000000,                          // 0  - Black
  0x00AA0000,                          // 1  - Blue
  0x0000AA00,                          // 2  - Green
  0x00AAAA00,                          // 3  - Cyan
  0x000000AA,                          // 4  - Red
  0x00AA00AA,                          // 5  - Magenta
  0x000055AA,                          // 6  - Brown
  0x00AAAAAA,                          // 7  - Light gray
  0x00555555,                          // 8  - Dark gray
  0x00FF5555,                          // 9  - Light blue
  0x0055FF55,                          // 10 - Light green
  0x00FFFF55,                          // 11 - Light cyan
  0x005555FF,                          // 12 - Light red
  0x00FF55FF,                          // 13 - Light magenta
  0x0055FFFF,                          // 14 - Light yellow
  0x00FFFFFF };                        // 15 - White


////////////////////////////////////////////////////////////////////////////////
//////////////////////////// INTERNAL LIST ROUTINES ////////////////////////////

// Service function, gets n-th line from the list, where 0 means line added
// last to the buffer. Does not work for LST_FIXED. Returns length of the line
// on success and -1 on error. To speed up the search, caches last found offset
// in the fields lastline and lastoffset.
static int Getline(t_list *pl,int n,t_lrechdr *phdr,char *s) {
  int i,k,lastoffset,datasize;
  uchar *data;
  if (pl==NULL || (pl->mode & LST_FIXED)!=0 || n<0 || n>=pl->nlines ||
    phdr==NULL || s==NULL)
    return -1;                 // Error in input parameters
  lastoffset=pl->lastoffset;
  datasize=pl->datasize;
  // Find record.
  if (n!=pl->lastline) {
    if (pl->lastline>=0 && n==pl->lastline+1) {
      // Frequent case of sequential access.
      lastoffset--;
      if (lastoffset<0) lastoffset+=datasize;
      lastoffset-=pl->data[lastoffset]+sizeof(t_lrechdr);
      if (lastoffset<0) lastoffset+=datasize; }
    else {                             // General case of linear search
      if (pl->next==0)                 // Find end of last message, correct
        lastoffset=datasize-1;         // possible wraparound.
      else
        lastoffset=pl->next-1;
      for (i=0; i<=n; i++) {
        lastoffset-=sizeof(t_lrechdr)+pl->data[lastoffset]+sizeof(uchar);
        if (lastoffset<0) lastoffset+=datasize; };
      lastoffset++;
      if (lastoffset>=datasize) lastoffset-=datasize; };
    pl->lastline=n;
    pl->lastoffset=lastoffset; };
  // Get header.
  data=pl->data+lastoffset;
  for (i=0; i<sigsize(t_lrechdr); i++) {
    ((uchar *)phdr)[i]=*data++;
    lastoffset++;
    if (lastoffset>=datasize) {
      lastoffset=0; data=pl->data;
    };
  };
  k=phdr->size;
  for (i=0; i<k; i++) {
    *s++=*data++;
    lastoffset++;
    if (lastoffset>=datasize) {
      lastoffset=0; data=pl->data;
    };
  };
  *s='\0';
  return k;
};

// Service function, intensifies brightness of color rgb.
static COLORREF Hilitergb(COLORREF rgb) {
  int r,g,b;
  // Get components.
  r=GetRValue(rgb);
  g=GetGValue(rgb);
  b=GetBValue(rgb);
  // Intensify colors.
  r=min(255,r*3/2+10);
  g=min(255,g*3/2+10);
  b=min(255,b*3/2+10);
  // Gather intensified color.
  return RGB(r,g,b);
};

// Service function, sets scrollbars in the list window.
static void Setlistscrollbars(t_list *pl,int slot,HWND hw) {
  int dx,dy,nx,ny,fw,fh,needx,needy;
  ulong options;
  RECT rc;
  SCROLLINFO sinfo;
  if (pl==NULL || slot<0 || slot>=NLISTWIN || hw==NULL)
    return;                            // Error in input parameters
  fw=pl->w[slot].fontdx;
  fh=pl->w[slot].fontdy;
  if (fw<1 || fh<1)
    return;                            // Unitialized font size
  // It may happen that window is large enough to fit data without scrollbars,
  // but some bar is already present, therefore data no longer fits and another
  // bar is necessary, too. Windows have no ready solution, let's check it by
  // ourselves.
  options=pl->w[slot].options;
  needx=needy=0;
  GetWindowRect(hw,&rc);
  dx=rc.right-rc.left;
  if (options & LWO_SUNKEN)
    dx-=2*GetSystemMetrics(SM_CXEDGE);
  if (options & LWO_PERFO)
    dx-=8*fw;
  else if (options & LWO_LBORDER)
    dx-=2*(fw/2);
  if ((pl->mode & LST_FIXED)==0) {
    dx-=GetSystemMetrics(SM_CXVSCROLL);
    needy=1; };
  dy=rc.bottom-rc.top;
  if (options & LWO_SUNKEN)
    dy-=2*GetSystemMetrics(SM_CYEDGE);
  if (options & LWO_LBORDER)
    dy-=2*(fh/2);
  while (1) {
    // Do as many iterations as necessary (at most 2).
    if (needx==0 && dx<pl->linesize*fw) {
      dy-=GetSystemMetrics(SM_CYHSCROLL);
      needx=1;
      continue; };
    if ((pl->mode & LST_FIXED)!=0 && needy==0 && dy<pl->linecount*fh) {
      dx-=GetSystemMetrics(SM_CXVSCROLL);
      needy=1;
      continue; };
    if ((pl->mode & LST_FIXED)==0 && needy==0 && dy<pl->nlines*fh) {
      dx-=GetSystemMetrics(SM_CXVSCROLL);
      needy=1;
      continue; };
    break;
  };
  nx=dx/fw;
  ny=dy/fh;
  // Set vertical scroll. Scrollable list windows always have it.
  sinfo.cbSize=sizeof(sinfo);
  sinfo.nPage=max(1,ny);
  sinfo.nMin=0;
  if (pl->mode & LST_FIXED) {
    // Fixed list.
    sinfo.fMask=SIF_PAGE|SIF_POS|SIF_RANGE;
    sinfo.nMax=pl->linecount-1;
    sinfo.nPos=pl->w[slot].offsety; }
  else {
    // Scrollable list.
    sinfo.fMask=SIF_PAGE|SIF_POS|SIF_RANGE|SIF_DISABLENOSCROLL;
    sinfo.nMax=pl->nlines-1;
    sinfo.nPos=pl->nlines-ny-pl->w[slot].offsety; };
  SetScrollInfo(hw,SB_VERT,&sinfo,TRUE);
  // Set horizontal scroll bar.
  sinfo.fMask=SIF_PAGE|SIF_POS|SIF_RANGE;
  sinfo.nMax=pl->linesize-1;
  sinfo.nPage=max(1,nx);
  sinfo.nPos=pl->w[slot].offsetx;
  SetScrollInfo(hw,SB_HORZ,&sinfo,TRUE);
};

// Window procedure of list window.
LRESULT CALLBACK Listwindowclassproc(HWND hw,UINT msg,WPARAM wp,LPARAM lp) {
  int i,j,k,n,r,slot,x,dx,dy,nx,ny,px,py,fw,fh;
  int linesize,fgcolor,bgcolor,features,dir;
  char s[TEXTLEN];
  t_list *pl;
  t_lsymbol *psym;
  t_lrechdr lhdr;
  RECT rc,rl,rt;
  SCROLLINFO sinfo;
  PAINTSTRUCT ps;
  TEXTMETRIC tm;
  XFORM xform;
  HDC dc,wc;
  HBITMAP hbmp;
  HPEN hpen;
  HBRUSH hbrush;
  pl=(t_list *)GetWindowLong(hw,0);
  if (pl==NULL)                        // Alarm! List window without list!
    return DefWindowProc(hw,msg,wp,lp);
  // Find slot for the current window.
  for (slot=0; slot<NLISTWIN; slot++) {
    if (pl->w[slot].hw==hw) break; };
  if (slot>=NLISTWIN)
    return DefWindowProc(hw,msg,wp,lp);
  switch (msg) {
    case WM_DESTROY:                   // About to destroy list window
      pl->w[slot].hw=NULL;
      SetWindowLong(hw,0,NULL);
      break;
    case WM_LBUTTONDOWN:
    case WM_LBUTTONDBLCLK:
    case WM_LBUTTONUP:
    case WM_RBUTTONDOWN:
    case WM_RBUTTONDBLCLK:
    case WM_RBUTTONUP:
    case WM_MBUTTONDOWN:
    case WM_MBUTTONDBLCLK:
    case WM_MBUTTONUP:
      if (pl->w[slot].options & LWO_PASSMOUSE)
        Passmousetoparent(hw,msg,wp,lp);
      else
        SetFocus(hw);
      break;
    case WM_HSCROLL:
      if (pl->w[slot].options & LWO_NOSCROLL)
        break;
      // Recalculate width of the client area into characters.
      GetClientRect(hw,&rc);
      fw=max(1,pl->w[slot].fontdx);
      dx=rc.right;
      if (pl->w[slot].options & LWO_LBORDER)
        dx-=2*(fw/2);
      nx=dx/fw;
      // Process message.
      j=pl->w[slot].offsetx;
      switch (LOWORD(wp)) {
        case SB_LINELEFT: j--; break;
        case SB_LINERIGHT: j++; break;
        case SB_PAGELEFT: j-=max(1,nx-1); break;
        case SB_PAGERIGHT: j+=max(1,nx-1); break;
        case SB_THUMBTRACK:
          sinfo.cbSize=sizeof(sinfo);
          sinfo.fMask=SIF_TRACKPOS;
          GetScrollInfo(hw,SB_HORZ,&sinfo);
          j=sinfo.nTrackPos;
          break;
        default: break; };
      linesize=pl->linesize;
      if (pl->w[slot].options & LWO_PERFO)
        linesize+=8;
      else if (pl->w[slot].options & LWO_LBORDER)
        linesize++;
      if (j+nx>linesize) j=linesize-nx;
      if (j<0) j=0;
      if (j!=pl->w[slot].offsetx) {
        pl->w[slot].offsetx=j;
        pl->w[slot].invalidated=1;
        InvalidateRect(hw,NULL,FALSE); };
      break;
    case WM_VSCROLL:
      if (pl->w[slot].options & LWO_NOSCROLL)
        break;
      // Recalculate height of the client area into characters.
      GetClientRect(hw,&rc);
      fh=max(1,pl->w[slot].fontdy);
      dy=rc.bottom;
      if (pl->w[slot].options & LWO_LBORDER)
        dy-=2*(fh/2);
      ny=dy/fh;
      // Process message.
      j=pl->w[slot].offsety;
      dir=(pl->mode & LST_FIXED?1:-1);
      switch (LOWORD(wp)) {
        case SB_LINEUP: j-=dir; break;
        case SB_LINEDOWN: j+=dir; break;
        case SB_PAGEUP: j-=max(1,ny-1)*dir; break;
        case SB_PAGEDOWN: j+=max(1,ny-1)*dir; break;
        case SB_THUMBTRACK:
          sinfo.cbSize=sizeof(sinfo);
          sinfo.fMask=SIF_TRACKPOS;
          GetScrollInfo(hw,SB_VERT,&sinfo);
          if (pl->mode & LST_FIXED)
            j=sinfo.nTrackPos;
          else
            j=pl->nlines-ny-sinfo.nTrackPos;
          break;
        default: break; };
      if (pl->mode & LST_FIXED) {
        if (j+ny>pl->linecount) j=pl->linecount-ny; }
      else {
        if (j+ny>pl->nlines) j=pl->nlines-ny; };
      if (j<0) j=0;
      if (j!=pl->w[slot].offsety) {
        pl->w[slot].offsety=j;
        pl->w[slot].invalidated=1;
        InvalidateRect(hw,NULL,FALSE); };
      break;
    case WM_USER_WHEEL:
      // Wheel scrolling is translated into the sequence of WM_VSCROLLs.
      if (pl->w[slot].options & LWO_NOSCROLL)
        break;
      i=((signed short)HIWORD(wp))/WHEEL_DELTA;
      for (j=0; j<abs(i) && j<8; j++)
        Listwindowclassproc(hw,WM_VSCROLL,(i>0?SB_LINEUP:SB_LINEDOWN),0);
      break;
    case WM_KEYDOWN:
      if (pl->w[slot].listproc!=NULL) {
        j=pl->w[slot].listproc(hw,msg,wp,lp);
        if (j!=0) break; };
      if ((pl->w[slot].options & LWO_KEYSCROLL)==0)
        break;                         // No keyboard scrolling
      switch (wp) {
        case VK_LEFT:
          Listwindowclassproc(hw,WM_HSCROLL,SB_LINELEFT,0); break;
        case VK_RIGHT:
          Listwindowclassproc(hw,WM_HSCROLL,SB_LINERIGHT,0); break;
        case VK_UP:
          Listwindowclassproc(hw,WM_VSCROLL,SB_LINEUP,0); break;
        case VK_DOWN:
          Listwindowclassproc(hw,WM_VSCROLL,SB_LINEDOWN,0); break;
        case VK_PRIOR:
          Listwindowclassproc(hw,WM_VSCROLL,SB_PAGEUP,0); break;
        case VK_NEXT:
          Listwindowclassproc(hw,WM_VSCROLL,SB_PAGEDOWN,0); break;
        default: break;
      };
      break;
    case WM_SYSKEYDOWN:
    case WM_CHAR:
      if (pl->w[slot].listproc!=NULL)
        pl->w[slot].listproc(hw,msg,wp,lp);
      break;
    case WM_TIMER:
      if (wp==TIMER_CURSOR) {
        pl->w[slot].cursoron=!pl->w[slot].cursoron;

        // VERY INEFFECTIVE IMPLEMENTATION!
        InvalidateRect(hw,NULL,FALSE);

      };
      break;
    case WM_PAINT:
      wc=BeginPaint(hw,&ps);
      // Get character size.
      SelectObject(wc,pl->w[slot].hfont);
      if (pl->w[slot].fontdx==0 || pl->w[slot].fontdy==0) {
        GetTextMetrics(wc,&tm);
        if (tm.tmAveCharWidth<1 || tm.tmHeight<1) {
          pl->w[slot].fontdx=6;
          pl->w[slot].fontdy=10; }
        else {
          pl->w[slot].fontdx=tm.tmAveCharWidth;
          pl->w[slot].fontdy=tm.tmHeight;
        };
      };
      fw=max(1,pl->w[slot].fontdx);
      fh=max(1,pl->w[slot].fontdy);
      // Set scroll bars.
      if ((pl->w[slot].options & (LWO_NOSCROLL|LWO_SCALE))==0)
        Setlistscrollbars(pl,slot,hw);
      // If scaled mode is requested, draw into the memory DC.
      GetClientRect(hw,&rc);
      if ((pl->w[slot].options & (LWO_SCALE|LWO_SNOWFREE))==0 ||
        rc.right<1 || rc.bottom<1) {
        // Draw directly on the screen.
        dc=wc; dx=rc.right; dy=rc.bottom; }
      else {
        // Draw to memory.
        dc=CreateCompatibleDC(wc);
        if (dc==NULL) {                // Emergency solution
          dc=wc; dx=rc.right; dy=rc.bottom; }
        else {
          if ((pl->w[slot].options & LWO_SCALE)==0) {
            dx=rc.right;
            dy=rc.bottom; }
          else {
            dx=pl->linesize*fw;
            if (pl->w[slot].options & LWO_PERFO)
              dx+=8*fw;
            else if (pl->w[slot].options & LWO_LBORDER)
              dx++;
            if (pl->mode & LST_FIXED)  // Scale whole screen
              dy=pl->linecount*fh;
            else                       // Fit width
              dy=(rc.bottom*dx)/rc.right;
            ;
          };
          hbmp=CreateCompatibleBitmap(wc,dx,dy);
          if (hbmp==NULL) {
            DeleteDC(dc);
            dc=wc; dx=rc.right; dy=rc.bottom; }
          else {
            SelectObject(dc,hbmp);
          };
        };
      };
      // Draw contents of the window.
      SelectObject(dc,pl->w[slot].hfont);
      SetTextAlign(dc,TA_LEFT|TA_BOTTOM);
      if (pl->w[slot].options & LWO_ROTATE) {
        xform.eM11=-1.0; xform.eM21=0.0; xform.eDx=dx;
        xform.eM12=0.0; xform.eM22=-1.0; xform.eDy=dy;
        SetGraphicsMode(dc,GM_ADVANCED);
        SetWorldTransform(dc,&xform); };
      if (pl->mode & LST_FIXED) {
        // Fixed list, drawn from top to bottom.
        rl.top=0;
        // Add top border.
        if (pl->w[slot].options & LWO_LBORDER) {
          SetBkColor(dc,pl->colors[0]);
          rl.bottom=rl.top+fh/2;
          rl.left=0;
          rl.right=dx;
          ExtTextOut(dc,rl.left,rl.bottom,ETO_CLIPPED|ETO_OPAQUE,
            &rl,NULL,0,NULL);
          rl.top=rl.bottom;
        };
        for (j=pl->w[slot].offsety; j<pl->linecount; j++) {
          // Draw next visible row.
          if (rl.top>=dy)
            break;                     // Window is filled
          rl.bottom=rl.top+fh;
          rl.left=rl.right=-pl->w[slot].offsetx*fw;
          // Add border to the left.
          if (pl->w[slot].options & LWO_LBORDER) {
            rl.right=rl.left+fw/2;
            if (rl.right>0) {
              SetBkColor(dc,pl->colors[0]);
              ExtTextOut(dc,rl.left,rl.bottom,ETO_CLIPPED|ETO_OPAQUE,
                &rl,NULL,0,NULL);
              ;
            };
            rl.left=rl.right;
          };
          psym=(t_lsymbol *)pl->data+j*pl->linesize;
          // Get and display as many characters as possible with the same
          // attributes.
          for (k=0; k<pl->linesize && rl.left<dx; ) {
            if (pl->w[slot].cursoron && k==pl->cursorx && j==pl->cursory) {
              fgcolor=psym->bgcolor;
              bgcolor=psym->fgcolor; }
            else {
              fgcolor=psym->fgcolor;
              bgcolor=psym->bgcolor; };
            features=psym->features;
            s[0]=psym->symbol;
            n=1; k++; psym++; rl.right=rl.left+fw;
            while (n<TEXTLEN && k<pl->linesize) {
              if (psym->fgcolor!=fgcolor || psym->bgcolor!=bgcolor ||
                psym->features!=features) break;
              if (pl->w[slot].cursoron && k==pl->cursorx && j==pl->cursory)
                break;
              s[n++]=psym->symbol;
              k++; psym++; rl.right+=fw; };
            if (rl.right>0) {
              if (pl->w[slot].options & LWO_HILITE) {
                // Intensified colors.
                SetBkColor(dc,Hilitergb(pl->colors[bgcolor]));
                SetTextColor(dc,Hilitergb(pl->colors[fgcolor])); }
              else {
                // Original colors.
                SetBkColor(dc,pl->colors[bgcolor]);
                SetTextColor(dc,pl->colors[fgcolor]); };
              ExtTextOut(dc,rl.left,rl.bottom,ETO_CLIPPED|ETO_OPAQUE,
                &rl,s,n,NULL);
              if (features & TF_UNDERLINE) {
                hpen=CreatePen(PS_SOLID,0,pl->colors[fgcolor]);
                SelectObject(dc,hpen);
                MoveToEx(dc,rl.left,rl.bottom-1,NULL);
                LineTo(dc,rl.right,rl.bottom-1);
                DeleteObject(hpen);
              };
            };
            rl.left=rl.right; };
          // If necessary, fill rest of the row with the default color.
          if (rl.right<dx) {
            SetBkColor(dc,pl->colors[0]);
            rl.left=rl.right;
            rl.right=dx;
            ExtTextOut(dc,rl.left,rl.bottom,ETO_CLIPPED|ETO_OPAQUE,
              &rl,NULL,0,NULL);
            ;
          };
          // Row completed.
          rl.top=rl.bottom;
        };
        // If necessary, fill rest of the window with the default color.
        if (rl.bottom<dy) {
          SetBkColor(dc,pl->colors[0]);
          rl.top=rl.bottom;
          rl.bottom=dy;
          rl.left=0;
          rl.right=dx;
          ExtTextOut(dc,rl.left,rl.bottom,ETO_CLIPPED|ETO_OPAQUE,
            &rl,NULL,0,NULL);
          ;
        }; }
      else {
        // Scrollable list, draw from bottom to top.
        k=pl->w[slot].offsetx;
        rl.left=0; rl.top=rl.bottom=dy;
        if ((pl->w[slot].options & LWO_PERFO)==0)
          rl.right=dx;
        else
          rl.right=(pl->linesize-k+8)*fw;
        if (pl->w[slot].options & LWO_PERFO)
          x=-k*fw+fw*4;                // Add perforation to the left
        else if (pl->w[slot].options & LWO_LBORDER)
          x=-k*fw+fw/2;                // Add border to the left
        else
          x=-k*fw;
        r=min(fh/2,(fw*3)/4);
        hpen=CreatePen(PS_SOLID,0,pl->colors[0]);
        hbrush=CreateSolidBrush(pl->colors[0]);
        SelectObject(dc,hpen);
        SelectObject(dc,hbrush);
        for (j=pl->w[slot].offsety; j<pl->nlines; j++) {
          // Draw next visible row.
          if (rl.bottom<=0)
            break;                     // Window is filled
          n=Getline(pl,j,&lhdr,s);
          if (n<0)
            break;                     // Internal error
          SetBkColor(dc,pl->colors[lhdr.bgcolor]);
          SetTextColor(dc,pl->colors[lhdr.fgcolor]);
          rl.top=rl.bottom-fh;
          rl.left=0;
          ExtTextOut(dc,x,rl.bottom,ETO_CLIPPED|ETO_OPAQUE,&rl,s,n,NULL);
          // If asked, add perforation.
          if ((pl->w[slot].options & LWO_PERFO)!=0) {
            if ((lhdr.features & LWO_PERFO)!=0) {
              py=rl.bottom-fh/2;
              px=(1-k)*fw+r;
              Ellipse(dc,px-r,py-r,px+r,py+r);
              px=(7+pl->linesize-k)*fw-r;
              Ellipse(dc,px-r,py-r,px+r,py+r); };
            if (rl.right<dx) {
              SetBkColor(dc,pl->colors[0]);
              rt.left=rl.right; rt.right=dx;
              rt.top==rl.top; rt.bottom==rl.bottom;
              ExtTextOut(dc,rt.left,rt.bottom,ETO_CLIPPED|ETO_OPAQUE,
                &rt,NULL,0,NULL);
              ;
            };
          };
          rl.bottom=rl.top;
        };
        // If necessary, fill rest of the window with the default color.
        if (rl.top>0) {
          SetBkColor(dc,pl->colors[0]);
          rl.bottom=rl.top;
          rl.top=0;
          ExtTextOut(dc,rl.left,rl.bottom,ETO_CLIPPED|ETO_OPAQUE,
            &rl,NULL,0,NULL);
          ;
        };
        DeleteObject(hbrush);
        DeleteObject(hpen);
      };
      // If drawing to memory DC, flush contents to the window. Note that
      // STRETCH_HALFTONE mode is slow!
      if (dc!=wc) {
        if (rc.right==dx && rc.bottom==dy &&
          (pl->w[slot].options & LWO_SCALE)==0)
          BitBlt(wc,0,0,rc.right,rc.bottom,dc,0,0,SRCCOPY);
        else {
          if (rc.right>=dx && rc.bottom>=dy)
            SetStretchBltMode(wc,STRETCH_DELETESCANS);
          else
            SetStretchBltMode(wc,STRETCH_HALFTONE);
          if (pl->w[slot].options & LWO_ROTATE)
            StretchBlt(wc,rc.right-1,rc.bottom-1,-rc.right,-rc.bottom,
            dc,0,0,dx,dy,SRCCOPY);
          else
            StretchBlt(wc,0,0,rc.right,rc.bottom,dc,0,0,dx,dy,SRCCOPY);
          ;
        };
        DeleteObject(hbmp);
        DeleteDC(dc); };
      EndPaint(hw,&ps);
      pl->w[slot].invalidated=0;
      break;
    default:
      return DefWindowProc(hw,msg,wp,lp);
    ;
  };
  return 0;
};


////////////////////////////////////////////////////////////////////////////////
//////////////////////////////// USER INTERFACE ////////////////////////////////

// Creates new list. Returns pointer to the list on success and NULL on error.
t_list *Createlist(ulong mode,int linesize,int linecount,
  COLORREF *colors,int ncolors) {
  ulong datasize;
  t_list *pl;
  if (linesize<=0 || linecount<=0)
    return NULL;                       // Error in input parameters
  if (linesize>TEXTLEN-1)
    linesize=TEXTLEN-1;                // Correct obvious error
  pl=(t_list *)malloc(sizeof(t_list));
  if (pl==NULL)
    return NULL;                       // Low memory
  memset(pl,0,sizeof(t_list));
  if (mode & LST_FIXED)
    // Fixed list.
    datasize=sizeof(t_lsymbol)*linesize*linecount;
  else {
    // Scrollable list.
    if (linesize<16)
      linesize=16;                     // Correct to more sound values
    else if (linesize>255)
      linesize=255;                    // Maximum enforced by data structures
    if (linecount<256)
      linecount=256;
    datasize=(sizeof(t_lrechdr)+linesize+sizeof(uchar))*linecount; };
  // Allocate data buffer.
  pl->data=(uchar *)malloc(datasize);
  if (pl->data==NULL) {
    free(pl);                          // Low memory
    return NULL; };
  if (mode & LST_FIXED)
    memset(pl->data,0,datasize);
  // Set descriptor and report success.
  pl->mode=mode;
  pl->linesize=linesize;
  pl->linecount=linecount;
  pl->datasize=datasize;
  pl->lastline=-1;
  pl->cursorx=pl->cursory=0;
  if (colors==NULL || ncolors<2) {
    pl->colors=defcolors;
    pl->ncolors=sizeof(defcolors)/sizeof(defcolors[0]); }
  else {
    pl->colors=colors;
    pl->ncolors=min(ncolors,256); };
  return pl;
};

// Changes size of fixed list and clears buffer. Returns 0 on success and -1
// on error.
int Changelistsize(t_list *pl,int linesize,int linecount) {
  int i;
  ulong datasize;
  uchar *data;
  if (pl==NULL || (pl->mode & LST_FIXED)==0 || linesize<=0 || linecount<=0)
    return -1;                         // Error in input parameters
  if (linesize>TEXTLEN-1)
    linesize=TEXTLEN-1;                // Correct obvious error
  if (pl->linesize!=linesize || pl->linecount!=linecount) {
    datasize=sizeof(t_lsymbol)*linesize*linecount;
    data=(uchar *)malloc(datasize);
    if (data==NULL)
      return -1;                       // Low memory
    free(pl->data);
    pl->linesize=linesize;
    pl->linecount=linecount;
    pl->data=data;
    pl->datasize=datasize; };
  memset(pl->data,0,pl->datasize);
  pl->lastline=-1;
  pl->bufferx=pl->buffery=0;
  // Update attached windows.
  for (i=0; i<NLISTWIN; i++) {
    if (pl->w[i].hw==NULL) continue;
    pl->w[i].invalidated=1;
    InvalidateRect(pl->w[i].hw,NULL,FALSE); };
  // Report success.
  return 0;
};

// Destroys list and attached windows.
void Destroylist(t_list *pl) {
  int i;
  if (pl==NULL)
    return;                            // Error in input parameters
  // If there are still open windows, close them now.
  for (i=0; i<NLISTWIN; i++) {
    if (pl->w[i].hw!=NULL) DestroyWindow(pl->w[i].hw); };
  // Free allocated memory.
  if (pl->data!=NULL)
    free(pl->data);
  free(pl);
};

// Calculates optimal list window size. On success, sets width and height and
// returns 0. On error, returns -1. Note that height for scrollable lists can
// be freely changed.
int Getlistwindowsize(t_list *pl,HFONT hfont,int options,
  int *width,int *height) {
  int w,h,fw,fh;
  TEXTMETRIC tm;
  HDC dc;
  if (pl==NULL || hfont==NULL)
    return -1;                         // Error in input parameters
  // Get font size.
  dc=GetDC(hwmain);
  if (dc==NULL)
    return -1;
  SelectObject(dc,hfont);
  memset(&tm,0,sizeof(tm));
  GetTextMetrics(dc,&tm);
  ReleaseDC(hwmain,dc);
  if (tm.tmAveCharWidth<1 || tm.tmHeight<1) {
    fw=6; fh=10; }
  else {
    fw=tm.tmAveCharWidth;
    fh=tm.tmHeight; };
  // Calculate width.
  w=fw*pl->linesize;
  if (options & LWO_PERFO)
    w+=8*fw;
  else if (options & LWO_LBORDER)
    w+=2*(fw/2);
  if ((pl->mode & LST_FIXED)==0 && (options & LWO_NOSCROLL)==0)
    w+=GetSystemMetrics(SM_CXVSCROLL);
  if (options & LWO_SUNKEN)
    w+=2*GetSystemMetrics(SM_CXEDGE);
  // For fixed window, calculate height. For scrollable window, height is 3/4
  // of the width, rounded to the integer line height.
  if (pl->mode & LST_FIXED) {
    h=fh*pl->linecount;
    if (options & LWO_LBORDER)
      h+=2*(fh/2);
    if (options & LWO_SUNKEN)
      h+=2*GetSystemMetrics(SM_CYEDGE);
    ; }
  else {
    h=(w*3)/4;
    h=((h+fh-1)/fh)*fh; };
  // Report success.
  if (width!=NULL) *width=w;
  if (height!=NULL) *height=h;
  return 0;
};

// Attaches new window to the list. If hparent is NULL, window is standalone.
// If hfont is NULL, selects some default font. Returns handle of the window on
// success and NULL on error.
HWND Createlistwindow(t_list *pl,char *name,int x,int y,int width,int height,
  HWND hparent,int id,HINSTANCE hinst,HFONT hfont,LISTPROC listproc,
  int options) {
  int slot;
  ulong style;
  WNDCLASS wc;
  TEXTMETRIC tm;
  HWND hw;
  HDC dc;
  if (pl==NULL)
    return NULL;                       // Error in input parameters
  // Assure that list window class is registered.
  if (GetClassInfo(hinst,LISTCLASS,&wc)==0) {
    wc.style=CS_OWNDC|CS_HREDRAW|CS_VREDRAW;
    wc.lpfnWndProc=Listwindowclassproc;
    wc.cbClsExtra=0;
    wc.cbWndExtra=sizeof(ulong);
    wc.hInstance=hinst;
    wc.hIcon=NULL;
    wc.hCursor=LoadCursor(NULL,IDC_ARROW);
    wc.hbrBackground=NULL;
    wc.lpszMenuName=NULL;
    wc.lpszClassName=LISTCLASS;
    if (RegisterClass(&wc)==0) return NULL; };
  // Find free slot.
  for (slot=0; slot<NLISTWIN; slot++) {
    if (pl->w[slot].hw==NULL) break; };
  if (slot>=NLISTWIN)
    return NULL;                       // No free slot for the new window
  // Create window.
  if (hparent==NULL)
    style=WS_OVERLAPPEDWINDOW|WS_CLIPCHILDREN;
  else
    style=WS_CHILD;
  if ((options & (LWO_NOSCROLL|LWO_SCALE))==0)
    style|=WS_HSCROLL|WS_VSCROLL;
  hw=CreateWindowEx(
    (hparent!=NULL && (options & LWO_SUNKEN)!=0?WS_EX_CLIENTEDGE:0),
    LISTCLASS,name,style,
    x,y,width,height,hparent,(hparent==NULL?NULL:(HMENU)id),hinst,pl);
  if (hw==NULL)
    return NULL;                       // Unable to create window
  // Set default font.
  if (hfont==NULL)
    hfont=GetStockObject(ANSI_FIXED_FONT);
  // If necessary, set cursor blinking timer.
  if ((pl->mode & LST_FIXED)!=0 &&
    (options & LWO_SCALE)==0 && (options & LWO_CURSOR)!=0)
    SetTimer(hw,TIMER_CURSOR,CURSOR_TIME/2,NULL);
  // Save parameters.
  pl->w[slot].hw=hw;
  pl->w[slot].options=options;
  pl->w[slot].hfont=hfont;
  pl->w[slot].offsetx=0;
  pl->w[slot].offsety=0;
  pl->w[slot].listproc=listproc;
  pl->w[slot].invalidated=1;
  // Get character size.
  dc=GetDC(hw);
  SelectObject(dc,pl->w[slot].hfont);
  GetTextMetrics(dc,&tm);
  if (tm.tmAveCharWidth<1 || tm.tmHeight<1) {
    pl->w[slot].fontdx=6;
    pl->w[slot].fontdy=10; }
  else {
    pl->w[slot].fontdx=tm.tmAveCharWidth;
    pl->w[slot].fontdy=tm.tmHeight; };
  ReleaseDC(hw,dc);
  SetWindowLong(hw,0,(ulong)pl);
  if ((options & (LWO_NOSCROLL|LWO_SCALE))==0)
    Setlistscrollbars(pl,slot,hw);
  // Show list and report success.
  ShowWindow(hw,SW_SHOW);
  return hw;
};

// Adds text of length n to the list (n<0: text is null-terminated). In the
// fixed mode, text starts at the current position and wraps to the next line
// or to the top of the list. In the scrollable mode, either adds new line
// (LST_CRLF is not set) or attaches text to existing, honoring CR/LF (LST_CRLF
// is set, typewriter mode). If there is not enough free space, removes several
// oldest messages.
void Addtolist(t_list *pl,char *text,int n,int fgcolor,int bgcolor,
  int features) {
  int i,ebcdic,offset,length;
  ulong u,v,datasize,available,nrec;
  uchar *data,buf[sizeof(t_lrechdr)+TEXTLEN+sizeof(uchar)];
  t_lsymbol *psym;
  t_lrechdr hdr;
  if (pl==NULL || pl->data==NULL || text==NULL)
    return;                            // Error in input parameters
  if (fgcolor<0 || fgcolor>=pl->ncolors)
    fgcolor=pl->ncolors-1;             // Correct obvious errors
  if (bgcolor<0 || bgcolor>=pl->ncolors)
    bgcolor=0;
  ebcdic=pl->mode & LST_EBCDIC;
  data=pl->data;
  if (n<=0)                            // Zero-terminated string
    n=strlen(text);
  if (pl->mode & LST_FIXED) {
    // Fixed list. Due to the automatical wrapping, text string may be longer
    // than TEXTLEN-1.
    pl->bufferx=max(0,min(pl->bufferx,pl->linesize-1));
    pl->buffery=max(0,min(pl->buffery,pl->linecount-1));
    psym=((t_lsymbol *)data)+pl->buffery*pl->linesize+pl->bufferx;
    for (i=0; i<n; i++) {
      psym->fgcolor=(uchar)fgcolor;
      psym->bgcolor=(uchar)bgcolor;
      psym->features=(uchar)features;
      if (ebcdic)
        psym->symbol=ebcdic_to_ascii[text[i] & 0xFF];
      else
        psym->symbol=text[i];
      psym++;
      if (++pl->bufferx>=pl->linesize) {
        pl->bufferx=0;
        if (++pl->buffery>=pl->linecount) {
          pl->buffery=0;
          psym=(t_lsymbol *)data;
        };
      };
    }; }
  else {
    // Scrollable list. In the CR/LF mode several lines may be added at once.
    offset=0;                          // Offset in text[]
    if ((pl->mode & LST_CRLF)==0) {
      // Standard mode.
      if (n>pl->linesize) n=pl->linesize;
      length=0; }
    else {
      // Typewriter mode, get and remove last added record.
      length=Getline(pl,0,&hdr,(char *)buf+sizeof(t_lrechdr));
      if (length<0)                    // Empty buffer? Anyway, ignore!
        length=0;
      else {                           // Remove last added record
        pl->next-=sizeof(t_lrechdr)+length+sizeof(uchar);
        if (pl->next>pl->datasize) pl->next+=pl->datasize;
        pl->nlines--;
      };
    };
    // Add lines, one after another.
    while (1) {
      // Get text.
      while (offset<n && length<pl->linesize) {
        if ((pl->mode & LST_CRLF)!=0 &&
          (text[offset]=='\n' || text[offset]=='\r')) break;
        if (ebcdic)
          buf[length+sizeof(t_lrechdr)]=ebcdic_to_ascii[text[offset] & 0xFF];
        else
          buf[length+sizeof(t_lrechdr)]=text[offset];
        length++;
        offset++;
      };
      // Prepare record.
      ((t_lrechdr *)buf)->size=(uchar)length;
      ((t_lrechdr *)buf)->fgcolor=(uchar)fgcolor;
      ((t_lrechdr *)buf)->bgcolor=(uchar)bgcolor;
      ((t_lrechdr *)buf)->features=(uchar)features;
      buf[sizeof(t_lrechdr)+length]=(uchar)length;
      nrec=sizeof(t_lrechdr)+length+sizeof(uchar);
      // Calculate free space in the list data buffer.
      datasize=pl->datasize;
      available=                       // Available free space
        datasize-pl->next+pl->oldest;
      if (available>datasize)          // Correct for possible wraparound
        available-=datasize;
      else if (available==datasize && pl->nlines!=0)
        available=0;                   // Completely full buffer!
      // Discard one or several old records if free space is insufficient.
      while (available<=nrec) {
        u=sizeof(t_lrechdr)+data[pl->oldest]+sizeof(uchar);
        pl->oldest+=u;
        if (pl->oldest>=datasize) pl->oldest-=datasize;
        pl->nlines--;
        available+=u; };
      // Insert new string. If wraparound will not occur, use fast memcpy(),
      // otherwise place characters one by one and check each time for
      // wraparound.
      u=pl->next;
      if (u+nrec<datasize) {
        // Wraparound will not occur.
        memcpy(data+u,buf,nrec);
        u+=nrec; }
      else {
        // Wraparound will occur.
        for (v=0; v<nrec; v++) {
          data[u++]=buf[v];
          if (u>=datasize) u-=datasize;
        }; };
      pl->next=u;
      pl->nlines++;
      pl->lastline=-1;                 // Offset of last found line is invalid
      // Check if data is complete.
      if ((pl->mode & LST_CRLF)==0 || offset>=n)
        break;
      if (text[offset]=='\n' || text[offset]=='\r') {
        offset++;
        if (offset<n && (text[offset]=='\n' || text[offset]=='\r') &&
          text[offset-1]!=text[offset]) offset++;
        ;
      };
      length=0;
    };
  };
  // Update attached windows.
  for (i=0; i<NLISTWIN; i++) {
    if (pl->w[i].hw==NULL) continue;
    if (pl->w[i].invalidated==0) {
      pl->w[i].invalidated=1;
      InvalidateRect(pl->w[i].hw,NULL,FALSE);
    };
  };
};

// Sets zero-based buffer position in the fixed list. Ignored by scrollable
// list.
void Setlistposition(t_list *pl,int bufferx,int buffery) {
  if (pl==NULL || pl->data==NULL)
    return;                            // Error in input parameters
  if ((pl->mode & LST_FIXED)==0)
    return;                            // Does not apply
  pl->bufferx=max(0,min(bufferx,pl->linesize-1));
  pl->buffery=max(0,min(buffery,pl->linecount-1));
};

// Sets zero-based cursor position in the fixed list. Ignored by scrollable
// list and scalable windows.
void Setcursorposition(t_list *pl,int cursorx,int cursory) {
  int i;
  if (pl==NULL || pl->data==NULL)
    return;                            // Error in input parameters
  if ((pl->mode & LST_FIXED)==0)
    return;                            // Does not apply
  if (pl->cursorx==cursorx && pl->cursory==cursory)
    return;                            // Cursor already set
  pl->cursorx=cursorx;
  pl->cursory=cursory;
  // Update attached windows.
  for (i=0; i<NLISTWIN; i++) {
    if (pl->w[i].hw==NULL)
      continue;
    if ((pl->w[i].options & LWO_SCALE)!=0)
      continue;
    if ((pl->w[i].options & LWO_CURSOR)==0)
      continue;
    pl->w[i].cursoron=1;               // Cursor is vislible after moving
    KillTimer(pl->w[i].hw,TIMER_CURSOR);
    SetTimer(pl->w[i].hw,TIMER_CURSOR,CURSOR_TIME/2,NULL);
    if (pl->w[i].invalidated==0) {
      pl->w[i].invalidated=1;
      InvalidateRect(pl->w[i].hw,NULL,FALSE);
    };
  };
};

// Clears list. Fixed list is filled with the spaces (background color 0).
void Clearlist(t_list *pl) {
  int i,n;
  t_lsymbol lsym,*psym;
  if (pl==NULL || pl->data==NULL)
    return;                            // Error in input parameters
  if (pl->mode & LST_FIXED) {
    // Fixed list.
    lsym.fgcolor=(uchar)(pl->ncolors-1);
    lsym.bgcolor=0;
    lsym.features=0;
    lsym.symbol=' ';
    n=pl->linesize*pl->linecount;
    for (i=0,psym=(t_lsymbol *)(pl->data); i<n; i++,psym++)
      *psym=lsym;
    pl->bufferx=pl->buffery=0; }
  else {
    // Scrollable list.
    pl->lastline=0;
    pl->lastoffset=0;
    pl->oldest=pl->next=0;
    pl->nlines=0;
  };
  // Update attached windows.
  for (i=0; i<NLISTWIN; i++) {
    if (pl->w[i].hw==NULL) continue;
    pl->w[i].invalidated=1;
    InvalidateRect(pl->w[i].hw,NULL,FALSE);
  };
};

// Asks user to save contents of the list to ASCII file. Returns 0 on success,
// 1 when operator cancelled this action and -1 on error.
int Savelisttofile(t_list *pl,int type) {
  int i,j,result;
  char s[TEXTLEN];
  t_lsymbol *psym;
  t_lrechdr phdr;
  FILE *f;
  if (pl==NULL)
    return -1;                         // Error in input parameters
  result=Browsefilename("Save data",pl->filename,type,NULL,NULL,1);
  if (result!=0)
    return result;
  f=fopen(pl->filename,"wt");
  if (f==NULL) {
    Message("JASN0024E  Unable to create file '%s'",pl->filename);
    return -1; };
  if (pl->mode & LST_FIXED) {
    // Fixed list. Trailing spaces are truncated, but empty lines on the bottom
    // are saved to the data.
    psym=(t_lsymbol *)pl->data;
    for (j=0; j<pl->linecount; j++) {
      for (i=0; i<pl->linesize; i++,psym++)
        s[i]=psym->symbol;
      while (i>0 && s[i-1]==' ')
        i--;
      if (i==0)
        fprintf(f,"\n");
      else
        fprintf(f,"%*.*s\n",i,i,s);
      ;
    }; }
  else {
    // Scrollable list. The implementation could be better: Getline() doesn't
    // use cache if line index counts down.
    for (j=pl->nlines-1; j>=0; j--) {
      i=Getline(pl,j,&phdr,s);
      if (i<0) continue;
      if (i==0)
        fprintf(f,"\n");
      else
        fprintf(f,"%*.*s\n",i,i,s);
      ;
    };
  };
  fclose(f);
  return 0;
};

