////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//      JASON - FULLY GRAPHICAL INTERFACE TO HERCULES IBM 360+ EMULATOR       //
//                                                                            //
//                              MAINFRAME WINDOW                              //
//                                                                            //
//                             Start: 10.02.2010                              //
//                        Current version: 16.08.2010                         //
//                                                                            //
// Jason is free software; you can redistribute it and/or modify it under     //
// the terms of the GNU General Public License version 3, as published by the //
// Free Software Foundation.                                                  //
//                                                                            //
// Jason is distributed in the hope that it will be useful, but WITHOUT ANY   //
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS  //
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more      //
// details.                                                                   //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program. If not, see <http://www.gnu.org/licenses/>.             //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#pragma hdrstop
#include <stdio.h>
#include <dir.h>

#include "Jason.h"

#define MFCAPT_L       0x01            // Mouse captured by left click
  #define MFCAPT_TL    0x02            // Timer on left border
  #define MFCAPT_TR    0x04            // Timer on right border
#define MFCAPT_R       0x08            // Mouse captured by right click

#define TIMER_DMOVE    3               // ID of device moving timer
#define DMOVE_TIME     100             // Device moving period, ms
#define DMOVE_DIST     30              // Device moving distance, pixels

static int       mfoffset;             // X offset of mainframe window
static int       mfcapture;            // Set of MFCAPT_xxx
static int       mfcaptxl;             // Mouse X when captured by left button
static int       mfcaptxlast;          // Last mouse X when captured by left
static int       mfcaptmoved;          // Mouse moved when captured by left
static int       mfcaptdev;            // Device captured by mouse left button
static int       mfcaptxr;             // Mouse X + offset when captured right

// Given x coordinate in the mainframe window, checks for device at this
// location. If device is available, optionally sets device X origin in client
// pixels (may be negative!) and returns index of the device. In all other
// cases, returns -1.
int Devicefromclient(int x,int *xorigin) {
  int i,x0;
  t_device *pdev;
  x0=0;
  x+=mfoffset;
  for (i=0,pdev=device; i<ndevice; i++,pdev++) {
    if (pdev->devclass==DEV_NONE)
      continue;                        // Placeholder
    if (pdev->bitmapsize.right<=0)
      continue;                        // Must not happen
    if (x>=x0 && x<x0+pdev->bitmapsize.right) {
      if (xorigin!=NULL) *xorigin=x0-mfoffset;
      return i; };
    x0+=pdev->bitmapsize.right;
  };
  // Device not found.
  return -1;
};

// Procedure that draws mainframe window.
static void Paintmainframe(HDC dc,RECT *rc) {
  int i,x0,x1;
  char *loadmap;
  t_device *pdev;
  RECT ra,rb,rr,rt;
  COLORREF bkcolor;
  ra=*rc; rb=*rc;
  ra.bottom=ra.top+mediumfont.dy+6;
  rb.top=ra.bottom;
  x1=rc->left-mfoffset;
  for (i=0,pdev=device; i<ndevice; i++,pdev++) {
    if (pdev->devclass==DEV_NONE)
      continue;                        // Placeholder
    if (x1>=rc->right)
      break;                           // End of mainframe window reached
    if (pdev->bitmapsize.right<=0)
      continue;                        // Must not happen
    x0=x1;
    x1+=pdev->bitmapsize.right;
    ra.left=rb.left=x0;
    ra.right=rb.right=x1;
    if (x1>rc->left) {
      // Draw title.
      SelectObject(dc,mediumfont.hfont);
      SetBkColor(dc,pdev->selected?RGB(96,232,96):GetSysColor(COLOR_BTNFACE));
      SetTextColor(dc,GetSysColor(COLOR_BTNTEXT));
      SetTextAlign(dc,TA_BOTTOM|TA_LEFT);
      rr=ra;
      FrameRect(dc,&rr,btnbrush);
      rr.left++; rr.top++; rr.right--; rr.bottom--;
      DrawEdge(dc,&rr,EDGE_RAISED,BF_RECT);
      rr.left+=2; rr.top+=2; rr.right-=2; rr.bottom-=2;
      ExtTextOut(dc,rr.left+3,rr.bottom,ETO_CLIPPED|ETO_OPAQUE,
        &rr,pdev->devname,strlen(pdev->devname),NULL);
      // Draw device.
      if (compactview) {
        if (rb.top<rb.bottom) FillRect(dc,&rb,btnbrush); }
      else {
        if (mfcaptmoved!=0 && mfcaptdev==i)
          bkcolor=RGB(96,232,96);
        else if (pdev->selected!=0 && hilitesel!=0)
          bkcolor=RGB(96,232,96);
        else
          bkcolor=fillcolor;
        Drawbitmap(dc,pdev->bitmap,&rb,bkcolor,0);
        if (pdev->file[0]!='\0' && pdev->file[0]!='*') {
          // Draw media.
          if (pdev->protect)
            loadmap=pdev->protmap;
          else
            loadmap=pdev->unprmap;
          if (loadmap[0]!='\0') {
            rt.left=rb.left+pdev->listrc.left;
            rt.top=rb.bottom-pdev->bitmapsize.bottom+pdev->listrc.top;
            rt.right=rb.left+pdev->listrc.right;
            rt.bottom=rb.bottom-pdev->bitmapsize.bottom+pdev->listrc.bottom;
            // listrc.left==0 means bitmap with background, like DASD disc
            // cover.
            Drawbitmap(dc,loadmap,&rt,
              (pdev->listrc.left==0?bkcolor:FILL_USEBG),0);
            ;
          };
        };
        // Draw custom text.
        if (pdev->text[0]!='\0' && pdev->textrc.right-pdev->textrc.left>0 &&
          pdev->textrc.bottom-pdev->textrc.top>0
        ) {
          if (pdev->textrc.bottom-pdev->textrc.top>15)
            SelectObject(dc,mediumfont.hfont);
          else
            SelectObject(dc,lucidafont.hfont);
          SetTextColor(dc,RGB(64,64,64));
          SetTextAlign(dc,TA_BASELINE|TA_RIGHT);
          SetBkMode(dc,TRANSPARENT);
          rt.left=ra.left+pdev->textrc.left;
          rt.top=rc->bottom-pdev->bitmapsize.bottom+pdev->textrc.top;
          rt.right=ra.left+pdev->textrc.right;
          rt.bottom=rc->bottom-pdev->bitmapsize.bottom+pdev->textrc.bottom;
          ExtTextOut(dc,rt.right-3,rt.bottom-2,ETO_CLIPPED,
            &rt,pdev->text,strlen(pdev->text),NULL);
          ;
        };
      };
    };
  };
  // Fill rest of the window.
  if (x1<rc->right) {
    rb=*rc; rb.left=x1;
    Drawbitmap(dc,"EMPTY",&rb,fillcolor,0);
  };
};

// Service function, moves list and state windows into correct positions.
static void Repositionsubwindows(void) {
  int i,x,y,x0,y0,x1,y1;
  t_device *pdev;
  RECT rc,rw;
  HDWP hdefer;
  x=-mfoffset;
  GetClientRect(hwframe,&rc);
  hdefer=BeginDeferWindowPos(16);
  for (i=0,pdev=device; i<ndevice; i++,pdev++) {
    if (pdev->devclass==DEV_NONE)
      continue;                        // Placeholder
    if (pdev->bitmapsize.right<=0)
      continue;                        // Must not happen
    if (pdev->listhw!=NULL) {
      // Calculate coordinates of list window in mainframe.
      y=rc.bottom-pdev->bitmapsize.bottom;
      x0=x+pdev->listrc.left; x1=x+pdev->listrc.right;
      if (compactview) {
        y0=pdev->listrc.top-pdev->listrc.bottom; y1=0; }
      else {
        y0=y+pdev->listrc.top; y1=y+pdev->listrc.bottom; };
      // Get actual coordinates of the window.
      GetWindowRect(pdev->listhw,&rw);
      ScreenToClient(hwframe,((POINT *)&rw)+0);
      ScreenToClient(hwframe,((POINT *)&rw)+1);
      if (rw.left!=x0 || rw.top!=y0 || rw.right!=x1 || rw.bottom!=y1) {
        DeferWindowPos(hdefer,pdev->listhw,NULL,x0,y0,x1-x0,y1-y0,
        SWP_NOZORDER|SWP_SHOWWINDOW);
      };
    };
    if (pdev->statehw!=NULL) {
      // Calculate coordinates of device state panel in mainframe.
      y=rc.bottom-pdev->bitmapsize.bottom;
      x0=x+pdev->staterc.left; x1=x+pdev->staterc.right;
      if (compactview) {
        y0=pdev->staterc.top-pdev->staterc.bottom; y1=0; }
      else {
        y0=y+pdev->staterc.top; y1=y+pdev->staterc.bottom; };
      // Get actual coordinates of the window.
      GetWindowRect(pdev->statehw,&rw);
      ScreenToClient(hwframe,((POINT *)&rw)+0);
      ScreenToClient(hwframe,((POINT *)&rw)+1);
      if (rw.left!=x0 || rw.top!=y0 || rw.right!=x1 || rw.bottom!=y1) {
        DeferWindowPos(hdefer,pdev->statehw,NULL,x0,y0,x1-x0,y1-y0,
        SWP_NOZORDER|SWP_SHOWWINDOW);
      };
    };
    x+=pdev->bitmapsize.right;
  };
  EndDeferWindowPos(hdefer);
};

// Windows function of the mainframe child window.
LRESULT CALLBACK Mainframewp(HWND hw,UINT msg,WPARAM wp,LPARAM lp) {
  int i,x,width,dev,xorigin,newoffset;
  t_device *pdev,temp;
  RECT rc;
  SCROLLINFO sinfo;
  PAINTSTRUCT ps;
  HDC dc;
  switch (msg) {
    case WM_DESTROY:
      hwframe=NULL;
      break;
    case WM_HSCROLL:
      // Get width of the window.
      GetClientRect(hw,&rc);
      // Calculate total width of the devices.
      width=0;
      for (i=0,pdev=device; i<ndevice; i++,pdev++) {
        if (pdev->devclass==DEV_NONE)
          continue;                    // Placeholder
        if (pdev->bitmapsize.right<=0)
          continue;                    // Must not happen
        width+=pdev->bitmapsize.right; };
      // Calculate new offset. Scroll bar itself is set by the WM_PAINT.
      newoffset=mfoffset;
      switch (LOWORD(wp)) {
        case SB_LINELEFT:
          newoffset-=rc.right/10; break;
        case SB_LINERIGHT:
          newoffset+=rc.right/10; break;
        case SB_PAGELEFT:
          newoffset-=rc.right*4/5; break;
        case SB_PAGERIGHT:
          newoffset+=rc.right*4/5; break;
        case SB_THUMBTRACK:
          // Horizontal scroll range may exceed 16-bit limit, so we must get
          // the full 32-bit tracking position.
          sinfo.cbSize=sizeof(sinfo);
          sinfo.fMask=SIF_TRACKPOS;
          GetScrollInfo(hw,SB_HORZ,&sinfo);
          newoffset=sinfo.nTrackPos;
        break;
      };
      // Correct offset.
      if (newoffset+rc.right>width)
        newoffset=width-rc.right;
      if (newoffset<0)
        newoffset=0;
      if (mfoffset!=newoffset) {
        mfoffset=newoffset;
        InvalidateRect(hw,NULL,FALSE); };
      break;
    case WM_LBUTTONDOWN:
      // Selects device.
      mfcaptdev=Devicefromclient(LOWORD(lp),NULL);
      if (mfcaptdev>=0 && device[mfcaptdev].selected==0)
        Selectdevice(mfcaptdev);
      if (mfcaptdev>=NDEVRESERVED) {   // Reserved devices are not movable
        mfcaptxl=LOINT(lp);
        mfcaptmoved=0;
        if (mfcapture==0)
          SetCapture(hw);
        mfcapture|=MFCAPT_L; };
      break;
    case WM_RBUTTONDOWN:
      // Scrolls mainframe window.
      mfcaptxr=LOINT(lp)+mfoffset;
      if (mfcapture==0)
        SetCapture(hw);
      mfcapture|=MFCAPT_R;
      break;
    case WM_LBUTTONUP:
      if (mfcapture & (MFCAPT_TL|MFCAPT_TR))
        KillTimer(hw,TIMER_DMOVE);
      mfcapture&=~(MFCAPT_L|MFCAPT_TL|MFCAPT_TR);
      if (mfcapture==0)
        ReleaseCapture();
      mfcaptmoved=0;
      InvalidateRect(hw,NULL,FALSE);
      break;
    case WM_RBUTTONUP:
      mfcapture&=~MFCAPT_R;
      if (mfcapture==0)
        ReleaseCapture();
      break;
    case WM_MOUSEMOVE:
      x=LOINT(lp);
      if (mfcapture & MFCAPT_L) {
        // Change device order by mouse.
        mfcaptxlast=x;
        if (!mfcaptmoved && abs(mfcaptxl-x)>GetSystemMetrics(SM_CXDOUBLECLK)) {
          mfcaptmoved=1;
          InvalidateRect(hw,NULL,FALSE); };
        dev=Devicefromclient(LOWORD(lp),&xorigin);
        if (dev>=NDEVRESERVED && dev!=mfcaptdev && mfcaptmoved!=0) {
          // Assure that when devices are swapped, cursor is still in the
          // selected device.
          if (x-xorigin<device[mfcaptdev].bitmapsize.right) {
            temp=device[mfcaptdev];
            device[mfcaptdev]=device[dev];
            device[dev]=temp;
            mfcaptdev=dev;
            InvalidateRect(hw,NULL,FALSE);
          };
        };
        GetClientRect(hw,&rc);
        if (x<0) {
          if ((mfcapture & (MFCAPT_TL|MFCAPT_TR))==0)
            SetTimer(hw,TIMER_DMOVE,DMOVE_TIME,NULL);
          mfcapture=(mfcapture & ~MFCAPT_TR)|MFCAPT_TL; }
        else if (x>=rc.right) {
          if ((mfcapture & (MFCAPT_TL|MFCAPT_TR))==0)
            SetTimer(hw,TIMER_DMOVE,DMOVE_TIME,NULL);
          mfcapture=(mfcapture & ~MFCAPT_TL)|MFCAPT_TR; }
        else if ((mfcapture & (MFCAPT_TL|MFCAPT_TR))!=0) {
          KillTimer(hw,TIMER_DMOVE);
          mfcapture&=~(MFCAPT_TL|MFCAPT_TR);
        };
      };
      if (mfcapture & MFCAPT_R) {
        // Scroll window by mouse.
        mfoffset=mfcaptxr-x;
        InvalidateRect(hw,NULL,FALSE); };
      if ((mfcapture & MFCAPT_L)!=0 && mfcaptmoved!=0)
        SetCursor(LoadCursor(NULL,IDC_SIZEALL));
      else
        SetCursor(LoadCursor(NULL,IDC_ARROW));
      break;
    case WM_TIMER:
      if (wp==TIMER_DMOVE) {
        // Here I repeat parts of WM_PAINT and WM_MOUSEMOVE.
        if (mfcapture & MFCAPT_TL) {
          mfoffset-=DMOVE_DIST;
          if (mfoffset<0) mfoffset=0; }
        else if (mfcapture & MFCAPT_TR) {
          mfoffset+=DMOVE_DIST;
          GetClientRect(hw,&rc);
          width=0;
          for (i=0,pdev=device; i<ndevice; i++,pdev++) {
            if (pdev->devclass==DEV_NONE)
              continue;                // Placeholder
            if (pdev->bitmapsize.right<=0)
              continue;                // Must not happen
            width+=pdev->bitmapsize.right; };
          if (mfoffset+rc.right>width)
            mfoffset=width-rc.right;
          ;
        };
        dev=Devicefromclient(mfcaptxlast,&xorigin);
        if (dev>=NDEVRESERVED && dev!=mfcaptdev) {
          if (mfcaptxlast-xorigin<device[mfcaptdev].bitmapsize.right) {
            temp=device[mfcaptdev];
            device[mfcaptdev]=device[dev];
            device[dev]=temp;
            mfcaptdev=dev;
          };
        };
        InvalidateRect(hw,NULL,FALSE);
      };
      break;
    case WM_ERASEBKGND:
      return 1;
    case WM_PAINT:
      dc=BeginPaint(hw,&ps);
      GetClientRect(hw,&rc);
      // Calculate total width of the devices and correct offset.
      width=0;
      for (i=0,pdev=device; i<ndevice; i++,pdev++) {
        if (pdev->devclass==DEV_NONE)
          continue;                    // Placeholder
        if (pdev->bitmapsize.right<=0)
          continue;                    // Must not happen
        width+=pdev->bitmapsize.right; };
      if (mfoffset+rc.right>width)
        mfoffset=width-rc.right;
      if (mfoffset<0)
        mfoffset=0;
      // Move children.
      Repositionsubwindows();
      // Set up sctroll bar.
      sinfo.cbSize=sizeof(sinfo);
      sinfo.fMask=SIF_DISABLENOSCROLL|SIF_PAGE|SIF_POS|SIF_RANGE;
      sinfo.nMin=0;
      sinfo.nMax=width;
      sinfo.nPage=rc.right;
      sinfo.nPos=mfoffset;
      SetScrollInfo(hw,SB_HORZ,&sinfo,TRUE);
      // Paint window.
      Paintmainframe(dc,&rc);
      EndPaint(hw,&ps);
      break;
    default: return DefWindowProc(hw,msg,wp,lp);
  };
  return 0L;
};

// Creates mainframe child in the top part of the main window which displays
// computer. This function is called once during startup.
HWND Createmainframe(void) {
  RECT rc;
  WNDCLASS wc;
  if (hwframe!=NULL)                   // Window already exists
    return hwframe;
  // Register class of mainframe child.
  if (GetClassInfo(hinst,FRAMECLASS,&wc)==0) {
    wc.style=CS_OWNDC|CS_HREDRAW|CS_VREDRAW;
    wc.lpfnWndProc=Mainframewp;
    wc.cbClsExtra=wc.cbWndExtra=0;
    wc.hInstance=hinst;
    wc.hIcon=NULL;
    wc.hCursor=NULL;
    wc.hbrBackground=NULL;
    wc.lpszMenuName=NULL;
    wc.lpszClassName=FRAMECLASS;
    if (!RegisterClass(&wc)) return NULL; };
  // Create mainframe child.
  mfcapture=0;
  GetClientRect(hwmain,&rc);
  hwframe=CreateWindowEx(
    WS_EX_CLIENTEDGE,FRAMECLASS,"",
    WS_CHILD|WS_HSCROLL|WS_VISIBLE|WS_CLIPCHILDREN,
    BORDER,BORDER,rc.right-2*BORDER,FRAMEDY,
    hwmain,(HMENU)0,hinst,NULL);
  if (hwframe==NULL)
    return NULL;

  return hwframe;
};

// Fits mainframe to the new size of the main Jason window.
void Resizemainframe(void) {
  RECT rc;
  GetClientRect(hwmain,&rc);
  if (compactview==0)
    MoveWindow(hwframe,BORDER,BORDER,rc.right-2*BORDER,FRAMEDY,1);
  else
    MoveWindow(hwframe,BORDER,BORDER,rc.right-2*BORDER,
    GetSystemMetrics(SM_CYHSCROLL)+2*GetSystemMetrics(SM_CYBORDER)+
    mediumfont.dy+8,1);
  ;
};

// Sets or resets compact mainframe view (only titles or full animation).
void Compactmainframe(int compact) {
  compactview=compact;
  Resizemainframe();
  Resizedevicewindow();
};

