////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//      JASON - FULLY GRAPHICAL INTERFACE TO HERCULES IBM 360+ EMULATOR       //
//                                                                            //
//                             SERVICE FUNCTIONS                              //
//                                                                            //
//                             Start: 10.02.2010                              //
//                        Current version: 26.09.2010                         //
//                                                                            //
// Jason is free software; you can redistribute it and/or modify it under     //
// the terms of the GNU General Public License version 3, as published by the //
// Free Software Foundation.                                                  //
//                                                                            //
// Jason is distributed in the hope that it will be useful, but WITHOUT ANY   //
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS  //
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more      //
// details.                                                                   //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program. If not, see <http://www.gnu.org/licenses/>.             //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#pragma hdrstop
#include <stdio.h>
#include <dir.h>

#include "Jason.h"
#include "resdata.h"


////////////////////////////////////////////////////////////////////////////////
/////////////////////////////// DATA CONVERSION ////////////////////////////////

// Table that converts EBCDIC characters to ASCII.
char           ebcdic_to_ascii[256] = {
  0x00, 0x01, 0x02, 0x03, 0x9C, 0x09, 0x86, 0x7F,  // 0x00
  0x97, 0x8D, 0x8E, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,  // 0x08
  0x10, 0x11, 0x12, 0x13, 0x9D, 0x85, 0x08, 0x87,  // 0x10
  0x18, 0x19, 0x92, 0x8F, 0x1C, 0x1D, 0x1E, 0x1F,  // 0x18
  0x80, 0x81, 0x82, 0x83, 0x84, 0x0A, 0x17, 0x1B,  // 0x20
  0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x05, 0x06, 0x07,  // 0x28
  0x90, 0x91, 0x16, 0x93, 0x94, 0x95, 0x96, 0x04,  // 0x30
  0x98, 0x99, 0x9A, 0x9B, 0x14, 0x15, 0x9E, 0x1A,  // 0x38
  0x20, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6,  // 0x40
  0xA7, 0xA8, 0xD5, 0x2E, 0x3C, 0x28, 0x2B, 0x7C,  // 0x48
  0x26, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF,  // 0x50
  0xB0, 0xB1, 0x21, 0x24, 0x2A, 0x29, 0x3B, 0x5E,  // 0x58
  0x2D, 0x2F, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7,  // 0x60
  0xB8, 0xB9, 0xE5, 0x2C, 0x25, 0x5F, 0x3E, 0x3F,  // 0x68
  0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xC0, 0xC1,  // 0x70
  0xC2, 0x60, 0x3A, 0x23, 0x40, 0x27, 0x3D, 0x22,  // 0x78
  0xC3, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,  // 0x80
  0x68, 0x69, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9,  // 0x88
  0xCA, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70,  // 0x90
  0x71, 0x72, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0,  // 0x98
  0xD1, 0x7E, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,  // 0xA0
  0x79, 0x7A, 0xD2, 0xD3, 0xD4, 0x5B, 0xD6, 0xD7,  // 0xA8
  0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF,  // 0xB0
  0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0x5D, 0xE6, 0xE7,  // 0xB8
  0x7B, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,  // 0xC0
  0x48, 0x49, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED,  // 0xC8
  0x7D, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50,  // 0xD0
  0x51, 0x52, 0xEE, 0xEF, 0xF0, 0xF1, 0xF2, 0xF3,  // 0xD8
  0x5C, 0x9F, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,  // 0xE0
  0x59, 0x5A, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9,  // 0xE8
  0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,  // 0xF0
  0x38, 0x39, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF   // 0xF8
};

// Table that converts ASCII characters to EBCDIC.
char           ascii_to_ebcdic[256] = {
  0x00, 0x01, 0x02, 0x03, 0x37, 0x2D, 0x2E, 0x2F,  // 0x00
  0x16, 0x05, 0x25, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,  // 0x08
  0x10, 0x11, 0x12, 0x13, 0x3C, 0x3D, 0x32, 0x26,  // 0x10
  0x18, 0x19, 0x3F, 0x27, 0x1C, 0x1D, 0x1E, 0x1F,  // 0x18
  0x40, 0x5A, 0x7F, 0x7B, 0x5B, 0x6C, 0x50, 0x7D,  // 0x20
  0x4D, 0x5D, 0x5C, 0x4E, 0x6B, 0x60, 0x4B, 0x61,  // 0x28
  0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,  // 0x30
  0xF8, 0xF9, 0x7A, 0x5E, 0x4C, 0x7E, 0x6E, 0x6F,  // 0x38
  0x7C, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,  // 0x40
  0xC8, 0xC9, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6,  // 0x48
  0xD7, 0xD8, 0xD9, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6,  // 0x50
  0xE7, 0xE8, 0xE9, 0xAD, 0xE0, 0xBD, 0x5F, 0x6D,  // 0x58
  0x79, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,  // 0x60
  0x88, 0x89, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96,  // 0x68
  0x97, 0x98, 0x99, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6,  // 0x70
  0xA7, 0xA8, 0xA9, 0xC0, 0x4F, 0xD0, 0xA1, 0x07,  // 0x78
  0x20, 0x21, 0x22, 0x23, 0x24, 0x15, 0x06, 0x17,  // 0x80
  0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x09, 0x0A, 0x1B,  // 0x88
  0x30, 0x31, 0x1A, 0x33, 0x34, 0x35, 0x36, 0x08,  // 0x90
  0x38, 0x39, 0x3A, 0x3B, 0x04, 0x14, 0x3E, 0xE1,  // 0x98
  0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,  // 0xA0
  0x49, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,  // 0xA8
  0x58, 0x59, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,  // 0xB0
  0x68, 0x69, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75,  // 0xB8
  0x76, 0x77, 0x78, 0x80, 0x8A, 0x8B, 0x8C, 0x8D,  // 0xC0
  0x8E, 0x8F, 0x90, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E,  // 0xC8
  0x9F, 0xA0, 0xAA, 0xAB, 0xAC, 0x4A, 0xAE, 0xAF,  // 0xD0
  0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7,  // 0xD8
  0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0x6A, 0xBE, 0xBF,  // 0xE0
  0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xDA, 0xDB,  // 0xE8
  0xDC, 0xDD, 0xDE, 0xDF, 0xEA, 0xEB, 0xEC, 0xED,  // 0xF0
  0xEE, 0xEF, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF   // 0xF8
};

// Contains 1 if byte is US ASCII symbol and 0 otherwise.
static uchar isusascii[256] = {
  0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0,  // 0x00
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  // 0x10
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,  // 0x20
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,  // 0x30
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,  // 0x40
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,  // 0x50
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,  // 0x60
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,  // 0x70
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  // 0x80
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  // 0x90
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  // 0xA0
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  // 0xB0
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  // 0xC0
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  // 0xD0
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  // 0xE0
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0   // 0xF0
};

// Contains 1 if byte is US EBCDIC symbol and 0 otherwise.
static uchar isusebcdic[256] = {
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,  // 0x00
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  // 0x10
  0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  // 0x20
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  // 0x30
  1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1,  // 0x40
  1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1,  // 0x50
  1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1,  // 0x60
  0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1,  // 0x70
  0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,  // 0x80
  0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,  // 0x90
  0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,  // 0xA0
  0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,  // 0xB0
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,  // 0xC0
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,  // 0xD0
  1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,  // 0xE0
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0   // 0xF0
};


////////////////////////////////////////////////////////////////////////////////
////////////////////////////// STRING OPERATIONS ///////////////////////////////

// Safe strlen(), checks at most n characters from src.
int Strlength(char *src,int n) {
  int length;
  if (src==NULL || n<=0)
    return 0;                          // Error in input parameters
  length=0;
  while (n>0 && src[length]!='\0') {
    n--; length++; };
  return length;
};

// Safe strcpy(), copies at most n-1 chars from src to dest and adds null
// character. Returns number of copied characters, or 0 on error.
int Strcopy(char *dest,int n,char *src) {
  int ncopy;
  if (dest==NULL || n<=0 || src==NULL)
    return 0;                          // Error in input parameters
  ncopy=0;
  while (n>1 && src[ncopy]!='\0') {
    dest[ncopy]=src[ncopy]; n--; ncopy++; };
  dest[ncopy]='\0';
  return ncopy;
};

// Attempts to determine type of the specified file, first by extention and
// then by data. Returns type as FF_xxx.
int Verifyfileformat(char *path) {
  int i,length,isascii,isebcdic;
  char buf[TEXTLEN],ext[MAXEXT];
  FILE *f;
  if (path==NULL)
    return FF_UNKNOWN;
  // First check extention.
  fnsplit(path,NULL,NULL,NULL,ext);
  if (stricmp(ext,".aws")==0)
    return FF_AWSTAPE;
  if (stricmp(ext,".ckd")==0 || stricmp(ext,".cckd")==0)
    return FF_CKD;
  if (stricmp(ext,".ebc")==0 || stricmp(ext,".ebcdic")==0)
    return FF_EBCDIC;
  if (stricmp(ext,".het")==0)
    return FF_HETTAPE;
  if (stricmp(ext,".jcl")==0)
    return FF_ASCII;
  if (stricmp(ext,".txt")==0)
    return FF_ASCII;
  // Unknown extention, check by contents.
  f=fopen(path,"rb");
  if (f==NULL)
    return FF_UNKNOWN;
  length=fread(buf,1,TEXTLEN,f);
  fclose(f);
  if (length<4)
    return FF_UNKNOWN;
  if (memcmp(buf,"CKD_",4)==0)
    return FF_CKD;
  if (memcmp(buf,"FBA_",4)==0)
    return FF_FBA;
  if (memcmp(buf,"P\0\0\0",4)==0)
    return FF_AWSTAPE;
  isascii=isebcdic=1;
  for (i=0; i<length-1; i++) {
    if (isusascii[buf[i]]==0)
      isascii=0;
    if (isusebcdic[buf[i]]==0)
      isebcdic=0;
    ;
  };
  if (isascii)
    return FF_ASCII;
  if (isebcdic)
    return FF_EBCDIC;
  return FF_UNKNOWN;
};


////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////// BITMAPS ////////////////////////////////////

// Determines size of the resource bitmap in pixels. Returns 0 on success and
// -1 on error.
int Getbitmapsize(char *resource,RECT *rc) {
  BITMAP bmp;
  HBITMAP hbmp;
  if (resource==NULL || rc==NULL)
    return -1;                         // Error in input parameters
  hbmp=LoadBitmap(hinst,resource);
  if (hbmp==NULL)
    return -1;                         // No such bitmap?
  GetObject(hbmp,sizeof(bmp),&bmp);
  DeleteObject(hbmp);
  rc->left=rc->top=0;
  rc->right=bmp.bmWidth;
  rc->bottom=bmp.bmHeight;
  return 0;
};

// Draws resource bitmap in the specified rectangle aligned with its left
// bottom corner (aligntop=0) or left top corner (aligntop=1). If fillcolor is
// FILL_USEBG, rest of the rectangle is filled with the background color of
// the bitmap (pixel in the bottom left corner). If fillcolor is FILL_TRANSP,
// background color is considered transparent. Otherwise, background of the
// bitmap and rest of the rectangle are filled with the fillcolor. Returns 0 on
// success and -1 on error.
int Drawbitmap(HDC dc,char *rsrc,RECT *rc,COLORREF fillcolor,int aligntop) {
  int dx,dy,result,ysrc;
  COLORREF background,rest;
  RECT rdraw,rfill;
  BITMAP bmp;
  HBITMAP hbmp,hmask;
  HDC dccolor,dcmono;
  HBRUSH hbrush;
  if (dc==NULL || rsrc==NULL || rc==NULL)
    return -1;                         // Error in input parameters
  // Prepare for error recovery.
  result=-1;
  hmask=NULL; dccolor=NULL; dcmono=NULL; hbrush=NULL;
  // Load bitmap from resource.
  hbmp=LoadBitmap(hinst,rsrc);
  if (hbmp==NULL)
    goto finish;                       // No such bitmap?
  // Get bitmap dimensions.
  GetObject(hbmp,sizeof(bmp),&bmp);
  dx=bmp.bmWidth;
  dy=bmp.bmHeight;
  // Create color DC, select bitmap and get its background color.
  dccolor=CreateCompatibleDC(dc);
  if (dccolor==NULL)
    goto finish;
  SelectObject(dccolor,hbmp);
  background=GetPixel(dccolor,0,dy-1);
  // Determine color of the filling.
  if (fillcolor==FILL_USEBG)
    rest=background;
  else
    rest=fillcolor;
  rdraw=*rc;
  if (fillcolor!=FILL_TRANSP &&
    (rdraw.bottom-rdraw.top>dy || rdraw.right-rdraw.left>dx))
    hbrush=CreateSolidBrush(rest);
  // If necessary, fill top of the rectangle.
  if (aligntop==0 && rdraw.bottom-rdraw.top>dy) {
    if (hbrush!=NULL) {
      rfill=rdraw; rfill.bottom-=dy;
      FillRect(dc,&rfill,hbrush); };
    rdraw.top=rfill.bottom; };
  // If necessary, fill right part of the rectangle.
  if (rdraw.right-rdraw.left>dx) {
    if (hbrush!=NULL) {
      rfill=rdraw; rfill.left+=dx;
      FillRect(dc,&rfill,hbrush); };
    rdraw.right=rfill.left; };
  // If necessary, fill bottom of the rectangle.
  if (aligntop!=0 && rdraw.bottom-rdraw.top>dy) {
    if (hbrush!=NULL) {
      rfill=rdraw; rfill.top+=dy;
      FillRect(dc,&rfill,hbrush); };
    rdraw.bottom=rfill.top; };
  ysrc=(aligntop?0:dy-(rdraw.bottom-rdraw.top));
  if (fillcolor==FILL_USEBG) {
    // Request to leave bitmap's background color, use fast BitBlt().
    BitBlt(dc,rdraw.left,rdraw.top,
      rdraw.right-rdraw.left,rdraw.bottom-rdraw.top,
      dccolor,0,ysrc,SRCCOPY);
    ; }
  else {
    // Background color is changed, create mask.
    dcmono=CreateCompatibleDC(dc);
    hmask=CreateBitmap(dx,dy,1,1,NULL);
    if (dcmono==NULL || hmask==NULL)
      goto finish;
    SelectObject(dcmono,hmask);
    SetBkColor(dccolor,background);
    BitBlt(dcmono,0,0,dx,dy,dccolor,0,0,NOTSRCCOPY);
    // Draw non-background parts of the bitmap.
    MaskBlt(dc,rdraw.left,rdraw.top,
      rdraw.right-rdraw.left,rdraw.bottom-rdraw.top,
      dccolor,0,ysrc,hmask,0,ysrc,
      MAKEROP4(SRCCOPY,0x00AA0029));
    // Draw background. Two separate passes may be slow, but image is not
    // blinking while drawing.
    SetTextColor(dc,rest);
    MaskBlt(dc,rdraw.left,rdraw.top,
      rdraw.right-rdraw.left,rdraw.bottom-rdraw.top,
      dcmono,0,ysrc,hmask,0,ysrc,MAKEROP4(0x00AA0029,SRCCOPY));
    ; };
  result=0;
finish:
  if (hmask!=NULL) DeleteObject(hmask);
  if (dcmono!=NULL) DeleteDC(dcmono);
  if (dccolor!=NULL) DeleteDC(dccolor);
  if (hbmp!=NULL) DeleteObject(hbmp);
  if (hbrush!=NULL) DeleteObject(hbrush);
  return result;
};


////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////// WINDOWS ////////////////////////////////////

// Passes mouse message to the parent, recalculating coordinates from child hw
// to its parent.
void Passmousetoparent(HWND hw,UINT msg,WPARAM wp,LPARAM lp) {
  POINT pt;
  HWND hparent;
  hparent=GetParent(hw);
  if (hparent==NULL)
    return;                            // Strange indeed...
  if (msg!=WM_LBUTTONDOWN && msg!=WM_LBUTTONDBLCLK && msg!=WM_LBUTTONUP &&
    msg!=WM_RBUTTONDOWN && msg!=WM_RBUTTONDBLCLK && msg!=WM_RBUTTONUP &&
    msg!=WM_MBUTTONDOWN && msg!=WM_MBUTTONDBLCLK && msg!=WM_MBUTTONUP)
    return;                            // Unsupported mouse message
  pt.x=LOINT(lp);
  pt.y=HIINT(lp);
  ClientToScreen(hw,&pt);
  ScreenToClient(hparent,&pt);
  SendMessage(hparent,msg,wp,MAKELONG(pt.x,pt.y));
};


////////////////////////////////////////////////////////////////////////////////
///////////////////////////// INITIALIZATION FILE //////////////////////////////

// Gets unformatted string from the main initialization file into buffer s of
// specified length (may be significantly larger than TEXTLEN). Returns string
// length in characters, or 0 on error.
int Stringfromini(char *section,char *key,char *s,int length) {
  if (section==NULL || key==NULL || s==NULL)
    return 0;                          // Error in parameters
  return GetPrivateProfileString(section,key,"",s,length,definifile);
};

// Gets formatted information from the main initialization file. Returns number
// of scanned variables. Set all variables to defaults before call to
// Getfromini(), in this case returned value can be ignored.
int Getfromini(char *section,char *key,char *format,...) {
  int n;
  char s[TEXTLEN];
  va_list ap;
  if (section==NULL || key==NULL || format==NULL)
    return 0;                          // Error in parameters
  n=GetPrivateProfileString(section,key,"",s,TEXTLEN,definifile);
  if (n==0)
    return 0;                          // String absent
  va_start(ap,format);
  n=vsscanf(s,format,ap);
  if (n<0) n=0;                        // vsscanf() may return EOF
  va_end(ap);
  return n;
};

// Removes all .ini file entries from the specified section.
void Clearinisection(char *section) {
  WritePrivateProfileString(section,NULL,NULL,definifile);
};

// Writes formatted information to the main initialization file. Returns length
// of string, or 0 on error. In most cases, returned value can be ignored.
// Attention, vsprintf() used here doesn't check for possible buffer overflow!
int Writetoini(char *section,char *key,char *format,...) {
  int n;
  char s[TEXTLEN*2];
  va_list ap;
  if (section==NULL || key==NULL || format==NULL)
    return 0;                          // Error in parameters
  va_start(ap,format);
  n=vsprintf(s+1,format,ap);
  if (s[1]=='"' || s[1]=='\'') {
    // Preserve quotes by quoting string once again.
    s[0]='"'; s[n+1]='"'; s[n+2]='\0';
    WritePrivateProfileString(section,key,s,definifile); }
  else
    WritePrivateProfileString(section,key,s+1,definifile);
  va_end(ap);
  return n;
};


////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////// FONTS /////////////////////////////////////

// Creates font according to description and calculates its width and height.
// Returns 0 on success (even partial, when function was unable to create font
// and substituted system fixed font instead) and -1 if input data is invalid.
int Createfont(t_font *pf) {
  TEXTMETRIC tm;
  HDC dc;
  HGDIOBJ oldfont;
  if (pf==NULL)                        // Error in input parameters
    return -1;
  if (pf->hfont!=NULL) {               // Font was not deleted
    if (pf->deleteonexit) DeleteObject(pf->hfont);
    pf->hfont=NULL; };
  if (pf->stockindex>0) {
    pf->hfont=(HFONT)GetStockObject(pf->stockindex);
    pf->deleteonexit=0; }
  else {
    pf->hfont=CreateFontIndirect(&(pf->logfont));
    if (pf->hfont!=NULL)
      pf->deleteonexit=1;
    else {
      pf->hfont=(HFONT)GetStockObject(SYSTEM_FIXED_FONT);
      pf->deleteonexit=0;
    };
  };
  dc=GetDC(hwmain);
  oldfont=SelectObject(dc,pf->hfont);
  GetTextMetricsA(dc,&tm);
  pf->dx=max(tm.tmAveCharWidth,1L);
  pf->dy=max(tm.tmHeight,1L);
  SelectObject(dc,oldfont);
  ReleaseDC(hwmain,dc);
  return 0;                            // Report success
};

// Service function, reads font description from the .ini file. On success,
// returns 0. If description is absent, fills logfont with defaults and returns
// 1. On error, returns -1.
static int Readfontdescription(char *key,t_font *pf) {
  int a[9];
  char facename[TEXTLEN];
  if (pf==NULL)
    return -1;                         // Error in input parameters
  memset(pf,0,sizeof(t_font));
  if (key==NULL || key[0]=='\0' || Getfromini("Fonts",
    key,"%i,%i,%i,%i,%i,%i,%i,%i,%i %s",
    a+0,a+1,a+2,a+3,a+4,a+5,a+6,a+7,a+8,facename)<10
  ) {                                  // Bad or missing initialization record
    pf->logfont.lfHeight=12;
    pf->logfont.lfWidth=8;
    pf->logfont.lfWeight=FW_NORMAL;
    pf->logfont.lfCharSet=255;
    pf->logfont.lfOutPrecision=OUT_DEFAULT_PRECIS;
    pf->logfont.lfClipPrecision=CLIP_DEFAULT_PRECIS;
    pf->logfont.lfQuality=PROOF_QUALITY;
    pf->logfont.lfPitchAndFamily=FF_MODERN;
    strcpy(pf->logfont.lfFaceName,"Terminal");
    return 1; }
  pf->logfont.lfHeight=min(32,max(-32,a[0]));
  pf->logfont.lfWidth=min(24,max(-24,a[1]));
  pf->logfont.lfWeight=min(FW_BLACK,max(0,a[2]));
  pf->logfont.lfItalic=(a[3]!=0);
  pf->logfont.lfUnderline=(a[4]!=0);
  pf->logfont.lfStrikeOut=(a[5]!=0);
  pf->logfont.lfCharSet=(uchar)a[6];
  pf->logfont.lfQuality=(uchar)a[7];
  pf->logfont.lfPitchAndFamily=(uchar)a[8];
  strcpy(pf->logfont.lfFaceName,facename);
  return 0;
};

// Service function, saves font description to the .ini file. Returns 0 on
// success and -1 on any error.
static int Savefontdescription(char *key,t_font *pf) {
  if (key==NULL || key[0]=='\0' || pf==NULL)
    return -1;                         // Error in input parameters
  if (Writetoini("Fonts",key,"%i,%i,%i,%i,%i,%i,%i,%i,%i %s",
    pf->logfont.lfHeight,pf->logfont.lfWidth,
    pf->logfont.lfWeight,pf->logfont.lfItalic,
    pf->logfont.lfUnderline,pf->logfont.lfStrikeOut,
    pf->logfont.lfCharSet,pf->logfont.lfQuality,
    pf->logfont.lfPitchAndFamily,
    pf->logfont.lfFaceName)<=0) return -1;
  return 0;
};

// Creates fonts used by Jason. Call it once during startup.
void Createjasonfonts(void) {
  // System-alike proportional font.
  memset(&sysfont,0,sizeof(t_font));
  sysfont.logfont.lfHeight=13;
  sysfont.logfont.lfWidth=5;
  sysfont.logfont.lfWeight=FW_NORMAL;
  sysfont.logfont.lfCharSet=DEFAULT_CHARSET;
  sysfont.logfont.lfOutPrecision=OUT_RASTER_PRECIS;
  sysfont.logfont.lfClipPrecision=CLIP_DEFAULT_PRECIS;
  sysfont.logfont.lfQuality=PROOF_QUALITY;
  sysfont.logfont.lfPitchAndFamily=DEFAULT_PITCH|FF_DONTCARE;
  strcpy(sysfont.logfont.lfFaceName,"MS Sans Serif");
  Createfont(&sysfont);
  // Medium-size proportional font.
  memset(&mediumfont,0,sizeof(t_font));
  mediumfont.logfont.lfHeight=18;
  mediumfont.logfont.lfWidth=0;
  mediumfont.logfont.lfWeight=FW_BOLD;
  mediumfont.logfont.lfCharSet=ANSI_CHARSET;
  mediumfont.logfont.lfOutPrecision=OUT_TT_PRECIS;
  mediumfont.logfont.lfClipPrecision=CLIP_DEFAULT_PRECIS;
  mediumfont.logfont.lfQuality=PROOF_QUALITY;
  mediumfont.logfont.lfPitchAndFamily=DEFAULT_PITCH|FF_SWISS|0x04;
  strcpy(mediumfont.logfont.lfFaceName,"");
  Createfont(&mediumfont);
  // Large proportional font.
  memset(&largefont,0,sizeof(t_font));
  largefont.logfont.lfHeight=28;
  largefont.logfont.lfWidth=0;
  largefont.logfont.lfWeight=FW_BOLD;
  largefont.logfont.lfCharSet=ANSI_CHARSET;
  largefont.logfont.lfOutPrecision=OUT_TT_PRECIS;
  largefont.logfont.lfClipPrecision=CLIP_DEFAULT_PRECIS;
  largefont.logfont.lfQuality=PROOF_QUALITY;
  largefont.logfont.lfPitchAndFamily=DEFAULT_PITCH|FF_SWISS|0x04;
  strcpy(largefont.logfont.lfFaceName,"");
  Createfont(&largefont);
  // Small fixed font.
  memset(&smallfixfont,0,sizeof(t_font));
  smallfixfont.logfont.lfHeight=9;
  smallfixfont.logfont.lfWidth=6;
  smallfixfont.logfont.lfWeight=FW_BOLD;
  smallfixfont.logfont.lfCharSet=255;
  smallfixfont.logfont.lfOutPrecision=OUT_DEFAULT_PRECIS;
  smallfixfont.logfont.lfClipPrecision=CLIP_DEFAULT_PRECIS;
  smallfixfont.logfont.lfQuality=PROOF_QUALITY;
  smallfixfont.logfont.lfPitchAndFamily=FF_MODERN;
  strcpy(smallfixfont.logfont.lfFaceName,"Terminal");
  Createfont(&smallfixfont);
  // Lucida Console font.
  memset(&lucidafont,0,sizeof(t_font));
  lucidafont.logfont.lfHeight=10;
  lucidafont.logfont.lfWidth=6;
  lucidafont.logfont.lfWeight=FW_NORMAL;
  lucidafont.logfont.lfCharSet=DEFAULT_CHARSET;
  lucidafont.logfont.lfOutPrecision=OUT_DEFAULT_PRECIS;
  lucidafont.logfont.lfClipPrecision=CLIP_DEFAULT_PRECIS;
  lucidafont.logfont.lfQuality=PROOF_QUALITY;
  lucidafont.logfont.lfPitchAndFamily=FIXED_PITCH|FF_DONTCARE|0x04;
  strcpy(lucidafont.logfont.lfFaceName,"Lucida Console");
  Createfont(&lucidafont);
  // Small font for punched cards.
  memset(&smallcardfont,0,sizeof(t_font));
  smallcardfont.logfont.lfHeight=8;
  smallcardfont.logfont.lfWidth=4;
  smallcardfont.logfont.lfWeight=FW_LIGHT;
  smallcardfont.logfont.lfCharSet=ANSI_CHARSET;
  smallcardfont.logfont.lfOutPrecision=OUT_RASTER_PRECIS;
  smallcardfont.logfont.lfClipPrecision=CLIP_DEFAULT_PRECIS;
  smallcardfont.logfont.lfQuality=PROOF_QUALITY;
  smallcardfont.logfont.lfPitchAndFamily=VARIABLE_PITCH;
  strcpy(smallcardfont.logfont.lfFaceName,"Small fonts");
  Createfont(&smallcardfont);
  // Read descriptions and create fixed list fonts.
  Readfontdescription("Display font",&dspfont);
  Createfont(&dspfont);
  Readfontdescription("Console font",&consfont);
  Createfont(&consfont);
  Readfontdescription("Editor font",&editfont);
  Createfont(&editfont);
  Readfontdescription("Printer font",&prtfont);
  Createfont(&prtfont);
  Readfontdescription("Hercules font",&hercfont);
  Createfont(&hercfont);
};

// Frees font resources.
void Deletejasonfonts(void) {
  // Delete general fonts.
  if (sysfont.deleteonexit)
    DeleteObject(sysfont.hfont);
  if (mediumfont.deleteonexit)
    DeleteObject(mediumfont.hfont);
  if (largefont.deleteonexit)
    DeleteObject(largefont.hfont);
  if (smallfixfont.deleteonexit)
    DeleteObject(smallfixfont.hfont);
  if (lucidafont.deleteonexit)
    DeleteObject(lucidafont.hfont);
  if (smallcardfont.deleteonexit)
    DeleteObject(smallcardfont.hfont);
  // Save descriptions and delete list fonts.
  Savefontdescription("Display font",&dspfont);
  if (dspfont.deleteonexit)
    DeleteObject(dspfont.hfont);
  Savefontdescription("Console font",&consfont);
  if (consfont.deleteonexit)
    DeleteObject(consfont.hfont);
  Savefontdescription("Editor font",&editfont);
  if (editfont.deleteonexit)
    DeleteObject(editfont.hfont);
  Savefontdescription("Printer font",&prtfont);
  if (prtfont.deleteonexit)
    DeleteObject(prtfont.hfont);
  Savefontdescription("Hercules font",&hercfont);
  if (hercfont.deleteonexit)
    DeleteObject(hercfont.hfont);
  ;
};


////////////////////////////////////////////////////////////////////////////////
//////////////////////////////// FILE BROWSING /////////////////////////////////

// Browses for file. Returns 0 on success, 1 if user cancelled input and -1 on
// error.
int Browsefilename(char *title,char *name,int type,char *defdir,
  HWND hwnd,int issave) {
  char defext[MAXEXT],filter[TEXTLEN*2+2];
  int n,result;
  OPENFILENAME ofn;
  if (name==NULL)
    return -1;                         // Error in input parameters
  // Create list of extentions (filter).
  defext[0]='\0';
  n=0;
  if (type & FT_BOOK) {
    n+=Strcopy(filter+n,TEXTLEN*2-n,"Readable file"); n++;
    n+=Strcopy(filter+n,TEXTLEN*2-n,
      "*.pdf;*.chm;*.htm;*.html;*.txt;*.doc;*.docx;*.rtf;*.hlp");
    n++;
    strcpy(defext,"pdf"); };
  if (type & FT_EXE) {
    n+=Strcopy(filter+n,TEXTLEN*2-n,"Executable file"); n++;
    n+=Strcopy(filter+n,TEXTLEN*2-n,"*.exe"); n++;
    strcpy(defext,"exe"); };
  if (type & FT_CONFIG) {
    n+=Strcopy(filter+n,TEXTLEN*2-n,"Hercules configuration file"); n++;
    n+=Strcopy(filter+n,TEXTLEN*2-n,"*.cnf;*.conf"); n++;
    strcpy(defext,"cnf"); };
  if (type & FT_AWS) {
    n+=Strcopy(filter+n,TEXTLEN*2-n,"AWS tape"); n++;
    n+=Strcopy(filter+n,TEXTLEN*2-n,"*.aws"); n++;
    strcpy(defext,"aws"); };
  if (type & FT_TAPE) {
    n+=Strcopy(filter+n,TEXTLEN*2-n,"Tape"); n++;
    n+=Strcopy(filter+n,TEXTLEN*2-n,"*.aws;*.het;*.xmi;*.tape;*."); n++;
    strcpy(defext,"aws"); };
  if (type & FT_DASD) {
    n+=Strcopy(filter+n,TEXTLEN*2-n,"DASD volume"); n++;
    n+=Strcopy(filter+n,TEXTLEN*2-n,"*.ckd;*.cckd;*.fba;*.dasd;*."); n++;
    strcpy(defext,"cckd"); };
  if (type & FT_PCH) {
    n+=Strcopy(filter+n,TEXTLEN*2-n,"Deck of cards"); n++;
    n+=Strcopy(filter+n,TEXTLEN*2-n,"*.jcl;*.pch;*."); n++;
    strcpy(defext,"jcl"); };
  if (type & FT_TXT) {
    n+=Strcopy(filter+n,TEXTLEN*2-n,"Text file"); n++;
    n+=Strcopy(filter+n,TEXTLEN*2-n,"*.txt;*.lst"); n++;
    strcpy(defext,"txt"); };
  n+=Strcopy(filter+n,TEXTLEN*2-n,"Any file (*.*)"); n++;
  n+=Strcopy(filter+n,TEXTLEN*2-n,"*.*"); n++;
  filter[n]='\0';                      // End of filter
  // Assure non-empty file name.
  if (name[0]=='\0')
    sprintf(name,"unnamed.%s",defext);
  // Fill in the structure and call standard dialog. To avoid compatibility
  // problems when running on different versions of Windows, I use version 4.00
  // of OPENFILENAME.
  ofn.lStructSize=OPENFILENAME_SIZE_VERSION_400;
  ofn.hwndOwner=(hwnd!=NULL?hwnd:hwmain);
  ofn.hInstance=hinst;
  ofn.lpstrFilter=filter;
  ofn.lpstrCustomFilter=NULL;
  ofn.nMaxCustFilter=0;
  ofn.nFilterIndex=1;
  ofn.lpstrFile=name;
  ofn.nMaxFile=MAXPATH;
  ofn.lpstrFileTitle=NULL;
  ofn.nMaxFileTitle=0;
  ofn.lpstrInitialDir=defdir;
  ofn.lpstrTitle=title;
  ofn.Flags=OFN_EXPLORER|OFN_HIDEREADONLY|OFN_LONGNAMES|OFN_NOCHANGEDIR|
    OFN_NOTESTFILECREATE|OFN_PATHMUSTEXIST;
  ofn.nFileOffset=0;
  ofn.nFileExtension=0;
  ofn.lpstrDefExt=defext;
  ofn.lCustData=0;
  ofn.lpfnHook=NULL;
  ofn.lpTemplateName=NULL;
  if (issave)
    result=GetSaveFileName(&ofn);
  else
    result=GetOpenFileName(&ofn);
  return (result==0?1:0);
};


////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////// MESSAGES ///////////////////////////////////

// Decodes error with code errcode (or last Windows error if errcode is 0)
// to string s of length n characters. Returns number of characters in the
// resulting string.
int Decodeerror(char *s,int n,ulong errcode) {
  int length;
  if (s==NULL)
    return 0;                          // Error in input parameters
  if (errcode==0)
    errcode=GetLastError();
  length=FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,NULL,errcode,
    MAKELANGID(LANG_NEUTRAL,SUBLANG_DEFAULT),s,n,NULL);
  if (length==0 && n>24)
    length=sprintf(s,"Windows error %i",errcode);
  return length;
};

// Adds message to the Hercules console list.
void Message(char *format,...) {
  int n;
  char s[2*TEXTLEN+MAXPATH];
  va_list ap;
  if (format==NULL) return;
  va_start(ap,format);
  n=vsprintf(s,format,ap);
  Addtolist(herclist,s,n,1,0,0);
  va_end(ap);
};

// JASN0001W  Unknown message '%s'
// JASN0002E  Unable to create pipe '%s'
// JASN0003E  Unable to create file '%s'
// JASN0004E  Hercules executable absent: '%s'
// JASN0005E  Hercules configuration file absent: '%s'
// JASN0006E  Unable to create standard pipes
// JASN0007E  Unable to start Hercules
// JASN0008W  (As yet unsupported 3270 function)
// JASN0009E  (Unable to create 3270 terminal)
// JASN0010E  Invalid drag-and-drop target
// JASN0011E  Invalid drag-and-drop data
// JASN0012E  Low memory, unable to process drag-and-drop
// JASN0013E  Invalid device at drag-and-drop target
// JASN0014E  Device does not support drag-and-drop
// JASN0016E  Unable to open file '%s'
// JASN0017E  Low memory, unable to edit file '%s'
// JASN0018E  Unable to read file '%s'
// JASN0019E  Low memory, file '%s' incomplete
// JASN0020W  Low memory, undo is not possible
// JASN0021E  Socket error %i while sending data
// JASN0022E  Socket timeout while sending data
// JASN0023E  Unable to create file '%s'
// JASN0024E  Unable to create file '%s'
// JASN0025E  (Unable to create console)

