////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//      JASON - FULLY GRAPHICAL INTERFACE TO HERCULES IBM 360+ EMULATOR       //
//                                                                            //
//                          DIFFERENT SMALL WINDOWS                           //
//                                                                            //
//                             Start: 27.02.2010                              //
//                        Current version: 01.03.2010                         //
//                                                                            //
// Jason is free software; you can redistribute it and/or modify it under     //
// the terms of the GNU General Public License version 3, as published by the //
// Free Software Foundation.                                                  //
//                                                                            //
// Jason is distributed in the hope that it will be useful, but WITHOUT ANY   //
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS  //
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more      //
// details.                                                                   //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program. If not, see <http://www.gnu.org/licenses/>.             //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#pragma hdrstop
#include <stdio.h>
#include <dir.h>
#include <math.h>

#include "Jason.h"

#define PANELCLASS     "JASONCPUPNL"   // Class name of CPU panel window
#define IPLCLASS       "JASONIPLPNL"   // Class name of CPU IPL switches
#define STATECLASS     "JASONDSTATE"   // Class name of device state panel

#define SUB_IPL        1001            // ID of IPL button


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////// CPU PANEL ///////////////////////////////////

// Windows function of the CPU panel window with blinking lamps, everything for
// fun!
LRESULT CALLBACK Cpupanelwp(HWND hw,UINT msg,WPARAM wp,LPARAM lp) {
  int i,j;
  ulong u;
  RECT rc,lc;
  PAINTSTRUCT ps;
  HDC dc,memdc;
  HBITMAP hbmp;
  HBRUSH hbrush;
  switch (msg) {
    case WM_LBUTTONDOWN:
    case WM_LBUTTONDBLCLK:
    case WM_LBUTTONUP:
    case WM_RBUTTONDOWN:
    case WM_RBUTTONDBLCLK:
    case WM_RBUTTONUP:
    case WM_MBUTTONDOWN:
    case WM_MBUTTONDBLCLK:
    case WM_MBUTTONUP:
      Passmousetoparent(hw,msg,wp,lp);
      break;
    case WM_ERASEBKGND:
      return 1;
    case WM_PAINT:
      dc=BeginPaint(hw,&ps);
      GetClientRect(hw,&rc);
      memdc=CreateCompatibleDC(dc);
      hbmp=LoadBitmap(hinst,"CPUPANEL");
      SelectObject(memdc,hbmp);
      hbrush=CreateSolidBrush(RGB(255,64,64));
      for (i=0; i<6; i++) {
        if (i<4) u=cpuregs[i];
        else if (i==4) {
          DeleteObject(hbrush);
          hbrush=CreateSolidBrush(RGB(64,255,64));
          u=cpupsw; }
        else u=cpuip;
        lc.top=35+i*8; lc.bottom=lc.top+2;
        for (j=0; j<32 && u!=0; j++,u>>=1) {
          if ((u & 1)==0) continue;
          lc.left=140-3*j; if (j>=16) lc.left-=10;
          lc.right=lc.left+2;
          FillRect(memdc,&lc,hbrush);
        };
      };
      DeleteObject(hbrush);
      BitBlt(dc,0,0,rc.right,rc.bottom,
        memdc,0,0,SRCCOPY);
      DeleteObject(hbmp);
      DeleteDC(memdc);
      EndPaint(hw,&ps);
      break;
    default: return DefWindowProc(hw,msg,wp,lp);
  };
  return 0L;
};

// Creates CPU front panel window.
HWND Createcpufrontpanel(HWND hparent,int x0,int y0,int dx,int dy) {
  WNDCLASS wc;
  HWND hw;
  if (GetClassInfo(hinst,PANELCLASS,&wc)==0) {
    wc.style=CS_OWNDC;
    wc.lpfnWndProc=Cpupanelwp;
    wc.cbClsExtra=wc.cbWndExtra=0;
    wc.hInstance=hinst;
    wc.hIcon=NULL;
    wc.hCursor=LoadCursor(NULL,IDC_ARROW);
    wc.hbrBackground=NULL;
    wc.lpszMenuName=NULL;
    wc.lpszClassName=PANELCLASS;
    if (!RegisterClass(&wc)) return NULL; };
  hw=CreateWindow(PANELCLASS,NULL,
    WS_CHILD|WS_VISIBLE,
    x0,y0,dx,dy,hparent,0,hinst,NULL);
  return hw;
};


////////////////////////////////////////////////////////////////////////////////
///////////////////////////////// IPL SWITCHES /////////////////////////////////


static int       iplcaptdigit;         // Hex digit with capture (0,1,2)

static float     iplcaptangle;         // Angle at capture
static int       iplcaptcuu;           // Original iplcuu at capture

// Windows function of the IPL panel window, everything for fun!
LRESULT CALLBACK Iplpanelwp(HWND hw,UINT msg,WPARAM wp,LPARAM lp) {
  int i,j,k,x,y,r2,x0,y0,delta;
  ulong mask;
  char c;
  float angle;
  RECT rc;
  PAINTSTRUCT ps;
  HDC dc,memdc;
  HBITMAP hbmp;
  HFONT hfont;
  switch (msg) {
    case WM_LBUTTONDOWN:
      SetFocus(GetParent(hw));         // Note: no break!
    case WM_LBUTTONDBLCLK:
    case WM_MOUSEMOVE:
      x=LOINT(lp);
      y=HIINT(lp);
      // Check whether cursor is over some switch.
      y0=58;
      if (GetCapture()==hw) {
        j=iplcaptdigit;
        x0=296-j*119; }
      else {
        for (j=0; j<3; j++) {
          x0=296-j*119;
          r2=(x-x0)*(x-x0)+(y-y0)*(y-y0);
          if (r2<54*54) break;
        };
      };
      if (j<3) {
        if (x==x0 && y==y0)
          angle=0.0;
        else
          angle=atan2(y-y0,x-x0);
        SetCursor(LoadCursor(NULL,IDC_HAND)); }
      else
        SetCursor(LoadCursor(NULL,IDC_ARROW));
      if (msg==WM_LBUTTONDOWN || msg==WM_LBUTTONDBLCLK) {
        iplcaptdigit=j;
        iplcaptangle=angle;
        iplcaptcuu=iplcuu;
        SetCapture(hw); }
      else if (GetCapture()==hw) {
        delta=(int)floor((iplcaptangle-angle)*8.0/M_PI+0.5) & 0x0F;
        mask=0x00F<<iplcaptdigit*4;
        iplcuu=(iplcaptcuu & ~mask)+
          ((iplcaptcuu+(delta<<iplcaptdigit*4)) & mask);
        InvalidateRect(hw,NULL,FALSE); };
      break;
    case WM_LBUTTONUP:
      if (GetCapture()==hw)
        ReleaseCapture();
      break;
    case WM_RBUTTONDOWN:
    case WM_RBUTTONDBLCLK:
    case WM_RBUTTONUP:
    case WM_MBUTTONDOWN:
    case WM_MBUTTONDBLCLK:
    case WM_MBUTTONUP:
      Passmousetoparent(hw,msg,wp,lp);
      break;
    case WM_ERASEBKGND:
      return 1;
    case WM_PAINT:
      dc=BeginPaint(hw,&ps);
      GetClientRect(hw,&rc);
      memdc=CreateCompatibleDC(dc);
      hbmp=CreateCompatibleBitmap(dc,rc.right,rc.bottom);
      SelectObject(memdc,hbmp);
      Drawbitmap(memdc,"IPL",&rc,FILL_USEBG,1);
      SetTextAlign(memdc,TA_BOTTOM|TA_CENTER);
      SetBkMode(memdc,TRANSPARENT);
      // To draw rotated characters, I need fonts with different escapements.
      // I optimize the process by drawing hex numbers with the same escapement
      // at different wheels in one pass.
      for (i=0; i<16; i++) {
        // Create font with escapement.
        hfont=CreateFont(18,0,-i*225,-i*225,
          FW_BOLD,0,0,0,
          DEFAULT_CHARSET,
          OUT_TT_ONLY_PRECIS,
          CLIP_DEFAULT_PRECIS,
          DEFAULT_QUALITY,
          DEFAULT_PITCH|FF_SWISS,
          NULL);
        SelectObject(memdc,hfont);
        SetTextColor(memdc,i==0?RGB(0,0,0):RGB(100,100,140));
        // Calculate character coordinates.
        x0=31.0*cos(M_PI/2.0-(M_PI/8.0)*i);
        y0=58-31.0*sin(M_PI/2.0-(M_PI/8.0)*i);
        // Fine correct character positions.
        if (i==0) y0-=2;
        else if (i==4) x0+=2;
        else if (i==8) y0+=4;
        else if (i==12) x0-=2;
        // Draw characters on all three wheels.
        for (j=0; j<3; j++) {
          k=((iplcuu>>(j*4))+i) & 0x0F;
          if (k<=9) k+='0';
          else k+='A'-10;
          c=(char)k;
          TextOut(memdc,x0+296-j*119,y0,&c,1);
        };
        DeleteObject(hfont);
      };
      BitBlt(dc,0,0,rc.right,rc.bottom,
        memdc,0,0,SRCCOPY);
      DeleteObject(hbmp);
      DeleteDC(memdc);
      EndPaint(hw,&ps);
      break;
    case WM_COMMAND:
      switch (LOWORD(wp)) {
        case SUB_IPL:
          Sendstdin("IPL %03X",iplcuu);
          break;
        default: break;
      };
      break;
    default: return DefWindowProc(hw,msg,wp,lp);
  };
  return 0L;
};

// Determines size of the IPL panel window. On success, fills height and width
// and returns 0. On error, returns -1.
int Getiplpanelsize(int *width,int *height) {
  int result;
  RECT rc;
  result=Getbitmapsize("IPL",&rc);
  if (result!=0)
    return result;
  if (width!=NULL) {
    if (rc.right<IPLBTNDX+2*BORDER)
      rc.right=IPLBTNDX+2*BORDER;
    *width=rc.right+2*GetSystemMetrics(SM_CXEDGE); };
  if (height!=NULL)
    *height=rc.bottom+IPLBTNDY+2*BORDER+2*GetSystemMetrics(SM_CYEDGE);
  return 0;
};

// Creates IPL panel window. Its size is predefined, use Getiplpanelsize() to
// determine.
HWND Createiplpanel(HWND hparent,int x0,int y0) {
  int width,height;
  RECT rc;
  WNDCLASS wc;
  HWND hw,hipl;
  if (GetClassInfo(hinst,IPLCLASS,&wc)==0) {
    wc.style=CS_OWNDC;
    wc.lpfnWndProc=Iplpanelwp;
    wc.cbClsExtra=wc.cbWndExtra=0;
    wc.hInstance=hinst;
    wc.hIcon=NULL;
    wc.hCursor=NULL;
    wc.hbrBackground=NULL;
    wc.lpszMenuName=NULL;
    wc.lpszClassName=IPLCLASS;
    if (!RegisterClass(&wc)) return NULL; };
  width=358; height=168;
  Getiplpanelsize(&width,&height);
  hw=CreateWindowEx(WS_EX_STATICEDGE,IPLCLASS,NULL,
    WS_CHILD|WS_VISIBLE|WS_CLIPCHILDREN,
    x0,y0,width,height,hparent,0,hinst,NULL);
  if (hw==NULL)
    return NULL;
  GetClientRect(hw,&rc);
  hipl=CreateWindow("BUTTON","IPL",
    WS_CHILD|WS_VISIBLE|BS_PUSHBUTTON,
    (rc.right-IPLBTNDX)/2,rc.bottom-BORDER-IPLBTNDY,IPLBTNDX,IPLBTNDY,
    hw,(HMENU)SUB_IPL,hinst,NULL);
  SendMessage(hipl,WM_SETFONT,(WPARAM)largefont.hfont,1);
  return hw;
};


////////////////////////////////////////////////////////////////////////////////
////////////////////////////// DEVICE STATE PANEL //////////////////////////////

// Windows function of the device state panel window with blinking lamps,
// everything for fun!
LRESULT CALLBACK Statepanelwp(HWND hw,UINT msg,WPARAM wp,LPARAM lp) {
  int i,dx,x,r,g,b;
  ulong state;
  PAINTSTRUCT ps;
  HDC dc;
  HPEN hpen;
  HBRUSH hbrush;
  switch (msg) {
    case WM_LBUTTONDOWN:
    case WM_LBUTTONDBLCLK:
    case WM_LBUTTONUP:
    case WM_RBUTTONDOWN:
    case WM_RBUTTONDBLCLK:
    case WM_RBUTTONUP:
    case WM_MBUTTONDOWN:
    case WM_MBUTTONDBLCLK:
    case WM_MBUTTONUP:
      Passmousetoparent(hw,msg,wp,lp);
      break;
    case WM_ERASEBKGND:
      return 1;
    case WM_PAINT:
      dc=BeginPaint(hw,&ps);
      state=GetWindowLong(hw,0);
      hpen=CreatePen(PS_SOLID,0,RGB(0,0,0));
      SelectObject(dc,hpen);
      dx=(STATEDX-1)/4;
      for (i=0; i<4; i++) {
        x=i*dx;
        if (i==0) {                    // Red lamp
          r=255; g=176; b=176; }
        else if (i==1 || i==2) {       // Green lamp
          r=176; g=255; b=176; }
        else {                         // Yellow lamp
          r=224; g=224; b=144; };
        if ((state & (1<<i))==0)       // Dark lamp
          hbrush=CreateSolidBrush(RGB(r/4+16,g/4+16,b/4+16));
        else                           // Full brightness
          hbrush=CreateSolidBrush(RGB(r,g,b));
        SelectObject(dc,hbrush);
        Rectangle(dc,x,0,x+dx+1,STATEDY);
        DeleteObject(hbrush);
      };
      DeleteObject(hpen);
      EndPaint(hw,&ps);
      break;
    default: return DefWindowProc(hw,msg,wp,lp);
  };
  return 0L;
};

// Creates device state panel at the specified coordinates with given initial
// state. State panel is always STATEDX*STATEDY pixel large.
HWND Createstatepanel(HWND hparent,int x0,int y0,ulong state) {
  WNDCLASS wc;
  HWND hw;
  if (GetClassInfo(hinst,STATECLASS,&wc)==0) {
    wc.style=CS_OWNDC;
    wc.lpfnWndProc=Statepanelwp;
    wc.cbClsExtra=0;
    wc.cbWndExtra=sizeof(ulong);
    wc.hInstance=hinst;
    wc.hIcon=NULL;
    wc.hCursor=NULL;
    wc.hbrBackground=NULL;
    wc.lpszMenuName=NULL;
    wc.lpszClassName=STATECLASS;
    if (!RegisterClass(&wc)) return NULL; };
  hw=CreateWindow(STATECLASS,NULL,
    WS_CHILD|WS_VISIBLE,
    x0,y0,STATEDX,STATEDY,hparent,0,hinst,NULL);
  if (hw!=NULL)
    SetWindowLong(hw,0,state);
  return hw;
};

// Changes state of the device state panel. Returns 0 on success and -1 on
// error, but who cares?
int Setstatepanelstate(HWND hw,ulong state) {
  if (hw==NULL)
    return -1;
  SetWindowLong(hw,0,state);
  InvalidateRect(hw,NULL,FALSE);
  return 0;
};

