////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//      JASON - FULLY GRAPHICAL INTERFACE TO HERCULES IBM 360+ EMULATOR       //
//                                                                            //
//                               TELNET SUPPORT                               //
//                                                                            //
//                             Start: 10.02.2010                              //
//                        Current version: 23.02.2010                         //
//                                                                            //
// Jason is free software; you can redistribute it and/or modify it under     //
// the terms of the GNU General Public License version 3, as published by the //
// Free Software Foundation.                                                  //
//                                                                            //
// Jason is distributed in the hope that it will be useful, but WITHOUT ANY   //
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS  //
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more      //
// details.                                                                   //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program. If not, see <http://www.gnu.org/licenses/>.             //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#pragma hdrstop
#include <stdio.h>
#include <dir.h>

#include "Jason.h"

// Telnet commands.
#define TC_IAC         0xFF            // Interprete as command
#define TC_DONT        0xFE            // Don't use option
#define TC_DO          0xFD            // Use option
#define TC_WONT        0xFC            // Refuse option
#define TC_WILL        0xFB            // Accept option
#define TC_SUBNEG      0xFA            // Subnegotiation will follow
#define TC_GA          0xF9            // Go ahead
#define TC_EL          0xF8            // Erase line
#define TC_EC          0xF7            // Erase character
#define TC_AYT         0xF6            // Are you there?
#define TC_AO          0xF5            // Abort output
#define TC_IP          0xF4            // Interrupt process
#define TC_BREAK       0xF3            // Break
#define TC_DATAMARK    0xF2            // Data mark
#define TC_NOP         0xF1            // NOP
#define TC_SBEND       0xF0            // End of subnegotiation
#define TC_EOR         0xEF            // End of record

// Selected Telnet options.
#define TO_BINARY      0               // Binary transmission
#define TO_ECHO        1               // Echo
#define	TO_SGA         3               // Suppress go ahead
#define	TO_TMARK       6               // Timing mark
#define	TO_TTYPE       24              // Terminal type
#define	TO_EOR         25              // End of record

// Telnet suboptions.
#define TS_IS          0               // Option is
#define TS_SEND        1               // Send option
#define TS_INFO        2               // ENVIRON version of TS_IS
#define TS_REPLY       2               // AUTHENTICATION version of TS_IS
#define TS_NAME        3               // AUTHENTICATION version of TS_IS


////////////////////////////////////////////////////////////////////////////////
/////////////////////////////// SERVICE ROUTINES ///////////////////////////////

// Sends raw data to host. Returns 0 on success and -1 on any error.
int Sendhost(t_telnet *ptnt,uchar *data,int ndata) {
  int n,err;
  ulong t;
  if (ptnt==NULL || data==NULL)
    return -1;                         // Error in input parameters
  if (ptnt->socketvalid==0 || ptnt->socket==INVALID_SOCKET)
    return -1;                         // Connection absent or lost
  t=GetTickCount();
  while (ndata>0) {
    // Note that we have non-blocking socket!
    n=send(ptnt->socket,(char *)data,ndata,0);
    if (n>=0) {
      data+=n; ndata-=n; }             // Some data was sent
    else {
      err=WSAGetLastError();
      if (err!=WSAEWOULDBLOCK) {
        // Recover from the error.
        //-----========-==-=-==========----------===------====-----======-==-===
        return -1;
      };
      if (GetTickCount()-t>100LU+ndata) {
        // Timeout, recover!
        //-----========-==-=-==========----------===------====-----======-==-===
        return -1;
      };
    };
  };
  // All data successfully sent.
  return 0;
};

// Formats data according to Telnet requirements and sends it to host. Returns
// 0 on success and -1 on any error.
int Sendtelnet(t_telnet *ptnt,uchar *data,int ndata) {
  int i,j;
  uchar buf[2*DATALEN+2];

  // ATTENTION, TN3270E mode requires 5-byte TN3270E header (RFC 1647)!

  if (ptnt==NULL || data==NULL || ndata<0 || ndata>DATALEN)
    return -1;                         // Error in input parameters
  for (i=j=0; j<ndata; j++) {
    buf[i++]=data[j];
    if (data[j]==TC_IAC) buf[i++]=TC_IAC; };
  buf[i++]=TC_IAC;
  buf[i++]=TC_EOR;
  return Sendhost(ptnt,buf,i);
};

// Connects telnet session to remote host. On success, returns 0. On error,
// sets errmsg and returns -1.
int Connecttohost(t_telnet *ptnt,char *hostname,int port,char *errmsg) {
  int n,r;
  ulong mode;
  struct hostent *he;
  struct sockaddr_in addr;
  if (ptnt==NULL) {
    if (errmsg!=NULL) strcpy(errmsg,"Connect to host: error in parameters");
    return -1; };
  if (ptnt->socketvalid!=0 && (ptnt->socket>=0 || ptnt->reconnecting))
    return 0;                          // Already connected
  if (hostname==NULL || hostname[0]=='\0') {
    // Host is not defined, assume local connection.
    r=gethostname(ptnt->hostname,TEXTLEN);
    if (r!=0) {
      if (errmsg!=NULL) strcpy(errmsg,"Connect to host: unknown host");
      return -1;
    }; }
  else {
    // General case of full host name.
    while (*hostname==' ')
      hostname++;
    strncpy(ptnt->hostname,hostname,TEXTLEN);
    ptnt->hostname[TEXTLEN-1]='\0';
    n=strlen(ptnt->hostname);
    while (n>0 && ptnt->hostname[n-1]==' ')
      n--;
    ptnt->hostname[n]='\0';
    if (n==0) {
      if (errmsg!=NULL) strcpy(errmsg,"Connect to host: error in parameters");
      return -1;
    };
  };
  if (port<=0 || port>=65536)
    port=23;                           // Standard TELNET port
  ptnt->port=port;
  // Prepare address.
  he=gethostbyname(ptnt->hostname);
  if (he==NULL || *(he->h_addr_list)==NULL) {
    if (errmsg!=NULL) strcpy(errmsg,"Connect to host: can't get host address");
    return -1; };
  memcpy(&(addr.sin_addr.s_addr),*(he->h_addr_list),he->h_length);
  addr.sin_family=AF_INET;
  addr.sin_port=htons(port);
  // Create and connect socket.
  ptnt->socket=socket(PF_INET,SOCK_STREAM,0);
  if (ptnt->socket==INVALID_SOCKET) {
    if (errmsg!=NULL) strcpy(errmsg,"Connect to host: unable to connect");
    return -1; };
  ptnt->socketvalid=1; 
  mode=1;
  r=setsockopt(ptnt->socket,
    SOL_SOCKET,SO_OOBINLINE,(char *)&mode,sizeof(mode));
  if (r==0) r=setsockopt(ptnt->socket,
    SOL_SOCKET,SO_KEEPALIVE,(char *)&mode,sizeof(mode));
  if (r!=0) {
    if (errmsg!=NULL) strcpy(errmsg,"Connect to host: unable to set options");
    closesocket(ptnt->socket);
    ptnt->socket=INVALID_SOCKET;
    ptnt->socketvalid=0; 
    return -1; };
  if (connect(ptnt->socket,
    (struct sockaddr *)&addr,sizeof(addr))==SOCKET_ERROR) {
    if (errmsg!=NULL) strcpy(errmsg,"Connect to host: connection refused");
    closesocket(ptnt->socket);
    ptnt->socket=INVALID_SOCKET;
    ptnt->socketvalid=0; 
    return -1; };
  // Set socket mode.
  mode=1;
  ioctlsocket(ptnt->socket,FIONBIO,&mode);
  return 0;
};

// Processes IAC Telnet commands. On success, returns number of bytes to skip.
// On error, or if transaction is as yet incomplete, returns 0. If IAC command
// is not known, assumes 2-byte sequence.
static int Telnet(t_telnet *ptnt,uchar *pb,int n) {
  int nsend;
  uchar buf[TEXTLEN];
  if (ptnt==NULL || pb==NULL || n<=0 || pb[0]!=TC_IAC)
    return 0;                          // Error in input parameters
  if (n<2)
    return 0;                          // Incomplete data
  if (pb[1]==TC_DO) {
    // DO-phrases, answer WILL to known and WONT to all other.
    if (n<3) return 0;                 // Incomplete data
    buf[0]=TC_IAC;
    if (pb[2]==TO_BINARY || pb[2]==TO_TTYPE || pb[2]==TO_EOR)
      buf[1]=TC_WILL;
    else
      buf[1]=TC_WONT;
    buf[2]=pb[2];
    Sendhost(ptnt,buf,3);
    return 3; };
  if (pb[1]==TC_WILL) {
    // WILL-phrases, answer DO to known and DONT to all other.
    if (n<3) return 0;                 // Incomplete data
    buf[0]=TC_IAC;
    if (pb[2]==TO_BINARY || pb[2]==TO_TTYPE || pb[2]==TO_EOR)
      buf[1]=TC_DO;
    else
      buf[1]=TC_DONT;
    buf[2]=pb[2];
    Sendhost(ptnt,buf,3);
    return 3; }
  if (pb[1]==TC_SUBNEG && pb[2]==TO_TTYPE) {
    if (n<4) return 0;
    if (pb[3]==TS_SEND) {
      if (n<6) return 0;
      if (pb[4]==TC_IAC && pb[5]==TC_SBEND) {
        // Subnegotiation: terminal type.
        buf[0]=TC_IAC;
        buf[1]=TC_SUBNEG;
        buf[2]=TO_TTYPE;
        buf[3]=TS_IS;
        nsend=4+Strcopy((char *)buf+4,DATALEN-6,ptnt->name);
        buf[nsend++]=TC_IAC;
        buf[nsend++]=TC_SBEND;
        Sendhost(ptnt,buf,nsend);
        return 6; }
      else return 2; }                 // Unknown sequence, recover
    else return 2; };                  // Unknown sequence, recover
  if (pb[1]==TC_EOR)
    // End of record, most probably remainder of the just processed command.
    return 2;
  // Unknown sequence, recover.
  return 2;
};

// Gets next portion of data from telnet connection. If there is no complete
// command, returns 0. On error, or if buffer is too short to fit the command,
// returns -1 (command is discarded). Otherwise, fills buf with the command
// and returns its length.
int Gettelnetdata(t_telnet *ptnt,uchar *buf,int maxbuf) {
  int i,n,nbuf,err;
  uchar *pb;
  if (ptnt==NULL || buf==NULL || maxbuf<=0)
    return -1;                         // Error in input parameters
  // Check for socket state and for data from the host.
  if (ptnt->reconnecting) {
    return 0; };
  if (ptnt->socketvalid==0 || ptnt->socket==INVALID_SOCKET)
    return -1;
  // Get next piece of data.
  if (ptnt->nsockbuf<DATALEN) {
    n=recv(ptnt->socket,(char *)ptnt->sockbuf+ptnt->nsockbuf,
      DATALEN-ptnt->nsockbuf,0);
    if (n>0)
      // Some data received.
      ptnt->nsockbuf+=n;
    else if (n==0)
      // Host disconnected.
      //-----========-==-=-==========----------===------====-----======-==-===
      ptnt->reconnecting=1;
    else {
      // Some error, most probably WSAEWOULDBLOCK.
      err=WSAGetLastError();
      if (err!=WSAEWOULDBLOCK) {
        // Recover from the error.
        //-----========-==-=-==========----------===------====-----======-==-=
        ptnt->reconnecting=1;
      };
    };
  };
  // Check if complete portion of data is available.
  n=ptnt->nsockbuf;
  if (n<2) return 0;                   // Data portion is at least 2 bytes long
  pb=ptnt->sockbuf;
  nbuf=0;
  if (pb[0]==TC_IAC && pb[1]!=TC_IAC)
    // Telnet negotiation, process.
    i=Telnet(ptnt,pb,n);
  else {
    // Data, check whether IAC+nonIAC combination is available. Combinations
    // IAC IAC should be translated to a single IAC data byte.
    i=0;
    while (i<n) {
      if (pb[i]!=TC_IAC) {
        if (nbuf>=maxbuf) {            // Data is longer than buffer
          nbuf=-1; break; };
        buf[nbuf++]=pb[i++]; }         // Ordinary data byte
      else {
        if (i==n-1)
          return 0;                    // Still incomplete data
        if (pb[i+1]==TC_IAC) {
          if (nbuf>=maxbuf) {          // Data is longer than buffer
            nbuf=-1; break; };
          buf[nbuf++]=TC_IAC; i+=2; }
        else break;                    // End of data reached
      };
    };
  };
  // Skip processed data.
  if (i<n) memmove(ptnt->sockbuf,ptnt->sockbuf+i,n-i);
  ptnt->nsockbuf-=i;
  // Report result.
  return nbuf;
};

// Closes telnet connection.
void Closetelnet(t_telnet *ptnt) {
  if (ptnt==NULL)
    return;                            // Error in input parameters
  if (ptnt->socketvalid!=0 && ptnt->socket!=INVALID_SOCKET) {
    shutdown(ptnt->socket,SD_BOTH);
    closesocket(ptnt->socket); };
  memset(ptnt,0,sizeof(t_telnet));
};

