////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//      JASON - FULLY GRAPHICAL INTERFACE TO HERCULES IBM 360+ EMULATOR       //
//                                                                            //
//                               3270 EMULATOR                                //
//                                                                            //
//                             Start: 10.02.2010                              //
//                        Current version: 24.09.2010                         //
//                                                                            //
// Jason is free software; you can redistribute it and/or modify it under     //
// the terms of the GNU General Public License version 3, as published by the //
// Free Software Foundation.                                                  //
//                                                                            //
// Jason is distributed in the hope that it will be useful, but WITHOUT ANY   //
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS  //
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more      //
// details.                                                                   //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program. If not, see <http://www.gnu.org/licenses/>.             //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include <windows.h>
#pragma hdrstop
#include <stdio.h>
#include <dir.h>

#include "Jason.h"

// 3270 EBCDIC commands.
#define CMD_NONE       0x00            // Internal: no pending commans
#define CMD_RMA        0x6E            // Read modified all
#define CMD_EAU        0x6F            // Erase all unprotected
#define CMD_EWA        0x7E            // Erase/write alternate
#define CMD_W          0xF1            // Write
#define CMD_RB         0xF2            // Read buffer
#define CMD_WSF        0xF3            // Write structured field
#define CMD_EW         0xF5            // Erase/write
#define CMD_RM         0xF6            // Read modified

// 3270 EBCDIC orders.
#define ORD_PT         0x05            // Program tab
#define ORD_GE         0x08            // Graphic escape
#define ORD_SBA        0x11            // Set buffer address
#define ORD_EUA        0x12            // Erase unprotected to address
#define ORD_IC         0x13            // Insert cursor
#define ORD_SF         0x1D            // Start field
#define ORD_SA         0x28            // Set attributes
#define ORD_SFE        0x29            // Start field extended
#define ORD_MF         0x2C            // Modify field
#define ORD_RA         0x3C            // Repeat to address

// Attention Identification keys.
#define AID_NO         0x60            // No AID
#define AID_QREPLY     0x61
#define AID_ENTER      0x7D
#define AID_PF1        0xF1
#define AID_PF2        0xF2
#define AID_PF3        0xF3
#define AID_PF4        0xF4
#define AID_PF5        0xF5
#define AID_PF6        0xF6
#define AID_PF7        0xF7
#define AID_PF8        0xF8
#define AID_PF9        0xF9
#define AID_PF10       0x7A
#define AID_PF11       0x7B
#define AID_PF12       0x7C
#define AID_PF13       0xC1
#define AID_PF14       0xC2
#define AID_PF15       0xC3
#define AID_PF16       0xC4
#define AID_PF17       0xC5
#define AID_PF18       0xC6
#define AID_PF19       0xC7
#define AID_PF20       0xC8
#define AID_PF21       0xC9
#define AID_PF22       0x4A
#define AID_PF23       0x4B
#define AID_PF24       0x4C
#define AID_OICR       0xE6
#define AID_MSR_MHS    0xE7

#define AID_SELECT     0x7E            // Selection key attention
#define AID_PA1        0x6C            // PA1 key
#define AID_PA2        0x6E            // PA2 key (Cncl)
#define AID_PA3        0x6B            // PA3 key
#define AID_CLEAR      0x6D            // Clear key
#define AID_SYSREQ     0xF0            // Test request

#define AID_SF         0x88            // Structured fields follow

// 3270 attribute types.
#define ATR_ALL        0x00            // All attributes
#define ATR_FIELD      0xC0            // Filed attribute
#define ATR_VALIDATE   0xC1            // Field validation
#define ATR_OUTLINE    0xC2            // Field outlining
#define ATR_HILITE     0x41            // Extended highlighting
#define ATR_FGCOLOR    0x42            // Foreground color
#define ATR_CHARSET    0x43            // Character set
#define ATR_BGCOLOR    0x45            // Background color
#define ATR_TRANSP     0x46            // Transparency

// Reply modes.
#define REPLY_FIELD    0               // Field reply mode
#define REPLY_EXTFIELD 1               // Extended field reply mode
#define REPLY_CHAR     2               // Character reply mode

// Some EBCDIC characters and control codes.
#define EBCDIC_NULL    0x00            // Null character
#define EBCDIC_SUB     0x3F            // Substitute solid circle
#define EBCDIC_DUP     0x1C            // Duplicate, overscore asterisk
#define EBCDIC_FM      0x1E            // Field mark, overscore semicolon
#define EBCDIC_FF      0x0C            // Form feed
#define EBCDIC_CR      0x0D            // Carriage return
#define EBCDIC_NL      0x15            // New line
#define EBCDIC_EM      0x19            // End of medium
#define EBCDIC_SPACE   0x40            // Space

// 6-to-8 bit EBCDIC encoding table.
static uchar     encodeebcdic[64] = {  // EBCDIC encoding
  0x40, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
  0xC8, 0xC9, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F,
  0x50, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7,
  0xD8, 0xD9, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F,
  0x60, 0x61, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7,
  0xE8, 0xE9, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F,
  0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
  0xF8, 0xF9, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F };

static int aidpfmap[24] = {
  AID_PF1,  AID_PF2,  AID_PF3,  AID_PF4,  AID_PF5,  AID_PF6,
  AID_PF7,  AID_PF8,  AID_PF9,  AID_PF10, AID_PF11, AID_PF12,
  AID_PF13, AID_PF14, AID_PF15, AID_PF16, AID_PF17, AID_PF18,
  AID_PF19, AID_PF20, AID_PF21, AID_PF22, AID_PF23, AID_PF24
};

static int aidpamap[3] = {
  AID_PA1,  AID_PA2,  AID_PA3
};

static COLORREF  colors3270[18] = {    // List of 3270 colors
  // 3270 colors.
  0x00000000,                          // 0  - Black
  0x00DD0000,                          // 1  - Blue
  0x000000DD,                          // 2  - Red
  0x00AA00AA,                          // 3  - Pink
  0x0000DD00,                          // 4  - Green
  0x00AAAA00,                          // 5  - Turquoise
  0x0000AAAA,                          // 6  - Yellow
  0x00AAAAAA,                          // 7  - White
  0x00555555,                          // 8  - Highlighted Black
  0x00FF5555,                          // 9  - Highlighted Blue
  0x005555FF,                          // 10 - Highlighted Red
  0x00FF55FF,                          // 11 - Highlighted Pink
  0x0055FF55,                          // 12 - Highlighted Green
  0x00FFFF55,                          // 13 - Highlighted Turquoise
  0x0055FFFF,                          // 14 - Highlighted Yellow
  0x00FFFFFF,                          // 15 - Highlighted White
  // Colours in information line.
  0x00227722,                          // 16 - Dark Green
  0x00000000                           // 17 - Black
};

// Special keys:
// Insert     - toggles insert mode on and off;
// Enter      - Enter
// Esc        - Clear

// Fxx        - PF01..PF12

// Shift+Fxx  - PF13..PF24

// Ctrl+F1    - PA1

// Ctrl+F2    - PA2

// Ctrl+F3    - PA3


// Not yet implemented:
// Ctrl+Del   - Erase EOF
//            - Cancel


// Deviations from standard:

// - If field ends with spaces, Insert is still allowed. Trailing spaces are

// discarded.

// - If cursor is in the protected field, keyboard input is ignored instead of

// sending event to host.



////////////////////////////////////////////////////////////////////////////////
/////////////////////////////// SERVICE ROUTINES ///////////////////////////////

// Service function, reports unimplemented feature to Hercules terminal.
static void Unimplemented(char *text) {
  Message("JASN0008W  Unimplemented 3270 feature: %s",text);
};

// Service function, increments address with wraparound. On wraparound,
// optionally sets wrapped to 1. Otherwise, leaves wrapped unchanged.
static void Incrementaddr(t_3270 *pt,int *paddr,int *wrapped) {
  if (pt==NULL || paddr==NULL) return;
  (*paddr)++;
  if (*paddr>=pt->columns*pt->rows) {
    (*paddr)=0;
    if (wrapped!=NULL) *wrapped=1;
  };
};

// Service function, returns address of field attributes for the given address,
// or -1 if the whole screen buffer is unformatted.
static int Getfaaddress(t_3270 *pt,int bufaddr) {
  int addr;
  if (pt==NULL || bufaddr<0 || bufaddr>=pt->columns*pt->rows)
    return -1;
  addr=bufaddr;
  while (1) {
    if (pt->buf[addr].fa!=0)
      return addr;
    addr--;
    if (addr<0) addr=pt->columns*pt->rows-1;
    if (addr==bufaddr) return -1;
  };
};

// Service function, returns 1 if screen location is protected, 2 if it contains
// field attribute, and 0 otherwise.
static int Isprotected(t_3270 *pt,int bufaddr) {
  int addr;
  addr=Getfaaddress(pt,bufaddr);
  if (addr<0)
    return 0;                          // Unformatted screen
  if (addr==bufaddr)
    return 2;                          // Field attribute
  if (pt->buf[addr].c & FA_PROTECTED)
    return 1;                          // Protected field
  return 0;
};

// Service function, sets cursor and turns PROT/FA screen indicator on or off
// according to the cursor position.
static void Setcursor(t_3270 *pt,int cursor) {
  int isprotected;
  if (pt==NULL)
    return;                            // Error in input parameters
  if (cursor!=pt->cursor) {
    pt->cursor=cursor;
    pt->redraw=1; };
  isprotected=Isprotected(pt,cursor);
  if (isprotected!=pt->isprotected) {
    pt->isprotected=isprotected;
    pt->redraw=1;
  };
};

// Service function, encodes address to pb[2]. Assumes that address does not
// exceed 4095.
static void Setaddress(uchar *pb,int addr) {
  pb[0]=(uchar)encodeebcdic[(addr>>6) & 0x3F];
  pb[1]=(uchar)encodeebcdic[addr & 0x3F];
};

// Service function, processes Read Modified and Read Modified All commands.
// Also called when opeartor presses Enter key. Returns 0 on success and -1 on
// error.
static int Readmodified(t_3270 *pt,int readall) {
  int ndata,senddata,bufaddr,nfield,wrapped,ea;
  uchar data[DATALEN];
  if (pt==NULL)
    return -1;                         // Error in input parameters
  if (pt->aid==AID_SF) {
    Unimplemented("AID_SF"); return -1; };
  ndata=0;                             // Length of data to send to host
  senddata=1;                          // Whether to attach screen data
  // Create header.
  switch (pt->aid) {
    case AID_SYSREQ:                   // Test request
      data[ndata++]=0x01;
      data[ndata++]=0x5B;
      data[ndata++]=0x61;
      data[ndata++]=0x02;
      break;
    case AID_PA1:                      // PA1 key
    case AID_PA2:                      // PA2 key (Cncl)
    case AID_PA3:                      // PA3 key
    case AID_CLEAR:                    // Clear key
      if (readall==0) {
        data[ndata++]=(uchar)pt->aid;
        Sendtelnet(&(pt->telnet),data,ndata);
        return 0;
      };                               // Note: no break!
    case AID_SELECT:                   // Selection key attention
      if (readall==0) senddata=0;
    default:
      data[ndata++]=(uchar)pt->aid;
      Setaddress(data+ndata,pt->cursor); ndata+=2;
    break;
  };
  // Add data. Note that in the SNA mode reading starts at buffer position 0.
  nfield=0;
  wrapped=0;
  bufaddr=0;
  ea=0;
  while (bufaddr<pt->rows*pt->columns && wrapped==0) {
    // Skip till next field attribute.
    if (pt->buf[bufaddr].fa==0) {
      bufaddr++; continue; };
    nfield++;                          // Screen is formatted
    if ((pt->buf[bufaddr].c & FA_MDT)==0) {
      bufaddr++; continue; };          // Field is not modified
    Incrementaddr(pt,&bufaddr,&wrapped);
    data[ndata++]=ORD_SBA;
    Setaddress(data+ndata,bufaddr); ndata+=2;
    while (pt->buf[bufaddr].fa==0) {
      if (senddata && pt->buf[bufaddr].c!=EBCDIC_NULL) {
        // If necessary, send attributes.
        if (pt->replymode==REPLY_CHAR && pt->buf[bufaddr].ea!=ea) {
          Unimplemented("REPLY_CHAR");
          ea=pt->buf[bufaddr].ea; };
        // Send character.
        data[ndata++]=pt->buf[bufaddr].c; };
      Incrementaddr(pt,&bufaddr,&wrapped);
    };
  };
  if (nfield==0) {
    // Screen is unformatted and requires special treatment.
    for (bufaddr=0; bufaddr<pt->rows*pt->columns; bufaddr++) {
      if (pt->buf[bufaddr].c!=EBCDIC_NULL) data[ndata++]=pt->buf[bufaddr].c;
    };
  };
  Sendtelnet(&(pt->telnet),data,ndata);
  return 0;
};

// Creates list buffer for the specified terminal. On success, returns 0. On
// error, sets errmsg and returns -1.
static int Createterminallist(t_3270 *pt,char *errmsg) {
  if (pt==NULL || pt->rows<1 || pt->columns<1) {
    if (errmsg!=NULL) strcpy(errmsg,"Create terminal: error in parameters");
    return -1; };
  // If list is already associated, delete it.
  if (pt->list!=NULL)
    Destroylist(pt->list);
  // Create new list, adding 1 oversize row on the bottom.
  pt->list=Createlist(LST_FIXED|LST_EBCDIC,pt->columns,pt->rows+1,
    colors3270,sizeof(colors3270)/sizeof(colors3270[0]));
  if (pt->list==NULL) {
    if (errmsg!=NULL) strcpy(errmsg,"Create terminal: unable to set buffer");
    return -1; };
  // Set affected parameters and report success.
  pt->bufaddr=0;
  Setcursor(pt,0);
  pt->list->userptr=(void *)pt;
  return 0;
};

// Service function, erases display.
static void Erasescreen(t_3270 *pt) {
  memset(pt->buf,0,sizeof(pt->buf));
  pt->bufaddr=0;
  Setcursor(pt,0);
};

// Creates new terminal and connects it to the host. On success, returns
// pointer to terminal descriptor. On error, sets errmsg and returns NULL.
t_3270 *Createterminal(int model,int options,
  char *hostname,int port,char *parm,char *errmsg) {
  int n;
  t_3270 *pt;
  // Verify input parameters.
  if (model<2 || model>5)
    model=2;                           // Least common denominator
  if ((options & TO_3279)==0 && model>3)
    model=3;
  // Allocate memory for the terminal.
  pt=(t_3270 *)GlobalAlloc(GPTR,sizeof(t_3270));
  if (pt==NULL) {
    if (errmsg!=NULL)
      strcpy(errmsg,"Unable to create 3270 terminal");
    return NULL; };
  // Set model-independent and uninitialized parameters.
  pt->model=model;
  pt->options=options;
  pt->minrows=24;
  pt->mincolumns=80;
  pt->list=NULL;
  pt->programcheck=0;
  pt->replymode=REPLY_FIELD;
  pt->kbdlock=0;
  pt->insertmode=0;
  pt->aid=AID_NO;
  pt->redraw=1;
  pt->isprotected=0;
  pt->showfields=0;
  Erasescreen(pt);
  // Set model-dependent parameters.
  switch (pt->model) {
    case 2:
      pt->maxrows=24;
      pt->maxcolumns=80;
      break;
    case 3:
      pt->maxrows=32;
      pt->maxcolumns=80;
      break;
    case 4:
      pt->maxrows=43;
      pt->maxcolumns=80;
      break;
    case 5:
      pt->maxrows=27;
      pt->maxcolumns=132;
    break;
  };
  // Make initializations.
  n=sprintf(pt->telnet.name,
    "IBM-327%i-%i",(pt->options & TO_3279?9:8),pt->model);
  if (parm!=NULL && parm[0]!='\0') {
    pt->telnet.name[n++]='@';
    Strcopy(pt->telnet.name+n,SHORTNAME-n,parm); };
  pt->rows=pt->minrows;
  pt->columns=pt->mincolumns;
  if (Createterminallist(pt,errmsg)!=0) {
    GlobalFree((HGLOBAL)pt);
    return NULL; };
  // Connect to host.
  if (Connecttohost(&(pt->telnet),hostname,port,errmsg)!=0) {
    Destroylist(pt->list);
    GlobalFree((HGLOBAL)pt);
    return NULL; };
  // Report success.
  return pt;
};

// Callback function of the console window, receives keyboard messages.
ulong Termcallback(HWND hw,UINT msg,WPARAM wp,LPARAM lp) {
  int c,aid,readall,prev,next,last,shiftkey,controlkey,cursor,addr,fa;
  t_list *pl;
  t_3270 *pt;
  t_ch chprev,chnext;
  pl=(t_list *)GetWindowLong(hw,0);
  if (pl==NULL)                        // Alarm! List window without list!
    return 0;
  pt=(t_3270 *)pl->userptr;
  if (pt==NULL)
    return 0;
  switch (msg) {
    case WM_KEYDOWN:
      if (pt->kbdlock)
        break;    //-------=---=======------------------=====-=-================
      shiftkey=GetKeyState(VK_SHIFT) & 0x8000;
      controlkey=GetKeyState(VK_CONTROL) & 0x8000;
      cursor=pt->cursor;
      aid=0; readall=0;
      switch (wp) {
        case VK_INSERT:
          pt->insertmode=!pt->insertmode;
          pt->redraw=1; break;
        case VK_LEFT:
          cursor--;
          if (cursor<0) cursor=pt->rows*pt->columns-1;
          break;
        case VK_RIGHT:
          cursor++;
          if (cursor>=pt->rows*pt->columns) cursor=0;
          break;
        case VK_UP:
          cursor-=pt->columns;
          if (cursor<0) cursor+=pt->rows*pt->columns;
          break;
        case VK_DOWN:
          cursor+=pt->columns;
          if (cursor>=pt->rows*pt->columns)
            cursor-=pt->rows*pt->columns;
          break;
        case VK_ESCAPE:
          aid=AID_CLEAR; readall=0; break;
        case VK_RETURN:
          aid=AID_ENTER; readall=1; break;
        case VK_F1:
        case VK_F2:
        case VK_F3:
        case VK_F4:
        case VK_F5:
        case VK_F6:
        case VK_F7:
        case VK_F8:
        case VK_F9:
        case VK_F10:
        case VK_F11:
        case VK_F12:
          if (controlkey==0) {
            aid=aidpfmap[wp-VK_F1+(shiftkey?12:0)]; readall=0; }
          else if (wp>=VK_F1 && wp<=VK_F3) {
            aid=aidpamap[wp-VK_F1]; readall=0; };
          break;
        case VK_BACK:
        case VK_DELETE:
          if (wp==VK_BACK) {
            if (controlkey) break;
            cursor--;
            if (cursor<0) cursor=pt->rows*pt->columns-1; };
          addr=Getfaaddress(pt,cursor);
          if (addr>=0 && (addr==cursor || (pt->buf[addr].c & FA_PROTECTED)!=0))
            break;                     // Protected field -----===-====----=====
          // Move all characters till end of the field or screen to the left
          // or, if Ctrl+Del is pressed, clear rest of the field.
          next=cursor;
          while (1) {
            prev=next; Incrementaddr(pt,&next,NULL);
            if (pt->buf[next].fa!=0 || (addr<0 && next==0)) {
              pt->buf[prev].c=EBCDIC_NULL;
              pt->buf[prev].ea=0;
              break; };
            if (controlkey==0)
              pt->buf[prev]=pt->buf[next];
            else {
              pt->buf[prev].c=EBCDIC_NULL;
              pt->buf[prev].ea=0;
            };
          };
          if (addr>=0) pt->buf[addr].c|=FA_MDT;
          pt->redraw=1; break;
        default: break;
      };
      if (aid!=0) {
        // Attention key preassed, send notification to host.
        pt->aid=aid;
        Readmodified(pt,readall);
        pt->kbdlock=1;
        pt->redraw=1; }
      else if (cursor!=pt->cursor)
        Setcursor(pt,cursor);
      break;
    case WM_CHAR:
      if ((wp & 0xFF)==0x08)
        break;                         // Backspace is already processed
      if (pt->kbdlock)
        break;    //-------===-=======------------------=====-=-================
      c=ascii_to_ebcdic[wp & 0xFF];
      if (c==0)
        break;                         // Not a valid EBCDIC character
      cursor=pt->cursor;
      addr=Getfaaddress(pt,cursor);
      if (addr>=0 && (addr==cursor || (pt->buf[addr].c & FA_PROTECTED)!=0))
        break;                         // Protected field -----===-====----=====
      // If in insert mode, shift following characters. Unlike in the original
      // display, I allow spaces at the end of the field to be discarded.
      if (pt->insertmode!=0 && pt->buf[cursor].c!=EBCDIC_NULL) {
        // Find first EBCDIC_NULL.
        next=cursor;
        while (1) {
          if (pt->buf[next].c==EBCDIC_NULL) break;
          prev=next; Incrementaddr(pt,&next,NULL);
          if (next==cursor) break;
          if (pt->buf[next].fa!=0) break; };
        if (prev==cursor) {
          if (pt->buf[cursor].c!=EBCDIC_SPACE) break; }
        else {
          if (pt->buf[prev].c!=EBCDIC_NULL && pt->buf[prev].c!=EBCDIC_SPACE)
            break;
          last=prev;
          next=cursor;
          chprev=pt->buf[next];
          while (1) {
            prev=next; Incrementaddr(pt,&next,NULL);
            chnext=pt->buf[next]; pt->buf[next]=chprev; chprev=chnext;
            if (next==last) break;
          };
        };
      };
      // Insert character at cursor.
      pt->buf[cursor].c=(uchar)c;
      pt->buf[cursor].ea=0;
      // Field is modified.
      if (addr>=0)
        pt->buf[addr].c|=FA_MDT;
      // Advance cursor, skipping field attributes and protected fields.
      fa=0;
      while (1) {
        Incrementaddr(pt,&cursor,NULL);
        if (pt->buf[cursor].fa!=0)
          fa=pt->buf[cursor].c;
        else if ((fa & FA_PROTECTED)==0)
          break;
        ;
      };
      Setcursor(pt,cursor);
      pt->redraw=1; break;
    default: return 0;                 // Message is not processed
  };
  // Message processed.
  return 1;
};

// Calculates optimal terminal window size. On success, sets width and height
// and returns 0. On error, returns -1.
int Getterminalwindowsize(t_3270 *pt,HFONT hfont,int options,
  int *width,int *height) {
  if (pt==NULL) return -1;
  return Getlistwindowsize(pt->list,hfont,options,width,height);
};

// Opens terminal window associated with the specified terminal. Several (up
// to NLISTWIN) windows may be attached to the terminal simultaneously.
HWND Createterminalwindow(t_3270 *pt,int x,int y,int width,int height,
  HWND hparent,int id,HINSTANCE hinst,HFONT hfont,int options) {
  return Createlistwindow(pt->list,pt->telnet.name,x,y,width,height,
  hparent,id,hinst,hfont,Termcallback,options);
};

// Service function, interpretes pb[2] as a 12/14/16-bit address.
static int Getaddress(uchar *pb,int sixteenbit) {
  if (pb==NULL)
    return 0;                          // Better ideas?
  if (sixteenbit || (pb[0] & 0xC0)==0x00)
    return pb[0]*256+pb[1];
  return (pb[0] & 0x3F)*64+(pb[1] & 0x3F);
};

// Service function, increments buffer address with wraparound.
static void Incrementbufaddr(t_3270 *pt) {
  Incrementaddr(pt,&(pt->bufaddr),NULL);
};

// Service function, processes write, erase/write and erase/write alternative
// commands. On success, returns number of processed bytes. If data is
// incomplete, returns 0. On error, returns -1.
static int Writecmd(t_3270 *pt,uchar *pb,int n) {
  int i,j,k,wcc,unlock,ea,rows,columns,exitloop,count,nrest,addr,attr;
  t_ch *pbuf;
  if (pt==NULL || pb==NULL || n<=0)
    return -1;                         // Error in input parameters
  if (pb[0]!=CMD_W && pb[0]!=CMD_EW && pb[0]!=CMD_EWA)
    return -1;                         // Error in input parameters
  if (n<2)
    return 0;                          // Incomplete data
  unlock=0;
  wcc=pb[1];
  if (pb[0]!=CMD_W) {
    // Erase screen and set mode.
    if (pb[0]==CMD_EW) {
      rows=pt->minrows;
      columns=pt->mincolumns; }
    else {
      rows=pt->maxrows;
      columns=pt->maxcolumns; };
    if (pt->rows!=rows || pt->columns!=columns) {
      if (Changelistsize(pt->list,columns,rows+1)==0) {
        pt->rows=rows;
        pt->columns=columns;
      };
    };
    Erasescreen(pt);                   // Erase screen
    pt->bufaddr=0;
    Setcursor(pt,0);
    if (wcc & 0x40)
      pt->replymode=REPLY_FIELD;       // Set inbound reply mode to Field
    pt->programcheck=0;                // Erase program check indication
    // Provide a negative trigger reply ------========------==========-=-=---===
    pt->redraw=1;
  };
  // Process Write Control Character.
  // if (wcc & 0x08) send negative response 0801 (no printer).-------===========
  if (wcc & 0x04)
    Beep(1000,150);                    // Beep
  if (wcc & 0x02)
    unlock=1;                          // Restore keyboard at end
  if (wcc & 0x01) {
    for (i=0; i<BUFSIZE; i++) {        // Clear modified data flags
      if (pt->buf[i].fa) pt->buf[i].c&=~FA_MDT;
    };
  };
  pt->bufaddr=pt->cursor;              // By default, start writing at cursor
  ea=0;                                // By default, reset extended attributes
  count=2;
  exitloop=0;                          // -1: low data, >0: report error
  // Now process the data that follows write command.
  while (exitloop==0 && count<n) {
    nrest=n-count;
    switch (pb[count]) {
      case ORD_PT:                     // Program tab
        count++;
        while (1) {
          if (pt->buf[pt->bufaddr].fa!=0 &&
            (pt->buf[pt->bufaddr].c & FA_PROTECTED)==0
          ) {
            // Unprotected field found. Note that I do not zero the field.
            Incrementbufaddr(pt);
            break; };
          Incrementbufaddr(pt);
          if (pt->bufaddr==0)
            break;                     // Wraparound, stop search
          ;
        };
        pt->redraw=1; break;
      case ORD_GE:                     // Graphic escape, unsupported
        exitloop=0x0863; break;
      case ORD_SBA:                    // Set buffer address
        if (nrest<3) {
          exitloop=-1; break; };       // Command is too long
        addr=Getaddress(pb+count+1,0); count+=3;
        if (addr>=pt->columns*pt->rows) {
          exitloop=0x0801; break; };   // Buffer address outside the buffer
        pt->bufaddr=addr;
        pt->redraw=1; break;
      case ORD_EUA:                    // Erase unprotected to address
        if (nrest<3) {
          exitloop=-1; break; };       // Command is too long
        addr=Getaddress(pb+count+1,0); count+=3;
        if (addr>=pt->columns*pt->rows) {
          exitloop=0x1005; break; };   // Buffer address outside the buffer
        // Get attributes of the actual buffer position.
        j=Getfaaddress(pt,pt->bufaddr);
        if (j<0)
          attr=0;
        else
          attr=pt->buf[j].c;
        // Note that wraparound is allowed.
        while (1) {
          if (pt->buf[pt->bufaddr].fa)
            attr=pt->buf[pt->bufaddr].c;
          else if ((attr & FA_PROTECTED)==0) {
            pt->buf[pt->bufaddr].c=0;
            pt->buf[pt->bufaddr].ea=0; };
          Incrementbufaddr(pt);
          if (pt->bufaddr==addr) break; };
        pt->redraw=1; break;
      case ORD_IC:                     // Insert cursor
        count++;
        Setcursor(pt,pt->bufaddr);
        break;
      case ORD_SF:                     // Start field
        if (nrest<2) {
          exitloop=-1; break; };       // Command is too long
        count++;
        pt->buf[pt->bufaddr].c=pb[count++];
        pt->buf[pt->bufaddr].fa=1;
        pt->buf[pt->bufaddr].ea=0;
        Incrementbufaddr(pt);
        pt->redraw=1; break;
      case ORD_SA:                     // Set attributes
        if (nrest<3) {
          exitloop=-1; break; };       // Command is too long
        count++;
        switch (pb[count++]) {
          case ATR_ALL:                // All attributes. I don't validate arg
            ea=0; break;
          case ATR_HILITE:             // Extended highlighting
            if (pb[count]==0xF1)
              ea=(ea & ~EA_MODEMASK)|EA_BLINK;
            else if (pb[count]==0xF2)
              ea=(ea & ~EA_MODEMASK)|EA_REVERSE;
            else if (pb[count]==0xF4)
              ea=(ea & ~EA_MODEMASK)|EA_UNDERLINE;
            else
              ea=(ea & ~EA_MODEMASK)|EA_NORMAL;
            break;
          case ATR_FIELD:              // Filed attribute
          case ATR_VALIDATE:           // Field validation
          case ATR_OUTLINE:            // Field outlining
          case ATR_FGCOLOR:            // Foreground color
          case ATR_CHARSET:            // Character set
          case ATR_BGCOLOR:            // Background color
          case ATR_TRANSP:             // Transparency
          default:
            Unimplemented("Attributes");
          break; };
        count++;
        break;
      case ORD_SFE:                    // Start field extended
        if (nrest<2) {
          exitloop=-1; break; };       // Command is too long
        k=pb[count+1];                 // Note: may be zero!
        if (nrest<2*k+2) {
          exitloop=-1; break; };       // Command is too long
        count+=2;
        pbuf=pt->buf+pt->bufaddr;
        pbuf->c=0;
        pbuf->fa=1;
        pbuf->ea=0;
        for (j=0; j<k; j++) {
          switch (pb[count++]) {
            case ATR_FIELD:            // Filed attribute
              pbuf->c=pb[count]; break;
            case ATR_HILITE:           // Extended highlighting
              if (pb[count]==0xF1)
                pbuf->ea=(uchar)((pbuf->ea & ~EA_MODEMASK)|EA_BLINK);
              else if (pb[count]==0xF2)
                pbuf->ea=(uchar)((pbuf->ea & ~EA_MODEMASK)|EA_REVERSE);
              else if (pb[count]==0xF4)
                pbuf->ea=(uchar)((pbuf->ea & ~EA_MODEMASK)|EA_UNDERLINE);
              else
                pbuf->ea=(uchar)((pbuf->ea & ~EA_MODEMASK)|EA_NORMAL);
              break;
            case ATR_VALIDATE:         // Field validation
            case ATR_OUTLINE:          // Field outlining
            case ATR_FGCOLOR:          // Foreground color
            case ATR_CHARSET:          // Character set
            case ATR_BGCOLOR:          // Background color
            case ATR_TRANSP:           // Transparency
            default:
              Unimplemented("Attributes");
            break; };
          count++;
        };
        Incrementbufaddr(pt);
        pt->redraw=1; break;
      case ORD_MF:                     // Modify field
        Unimplemented("ORD_MF");
        exitloop=0xFFFE; //--------============-======--------------------------
        break;
      case ORD_RA:                     // Repeat to address
        if (nrest<4) {
          exitloop=-1; break; };       // Command is too long
        addr=Getaddress(pb+count+1,0); count+=3;
        if (addr>=pt->columns*pt->rows) {
          exitloop=0x1005; break; };   // Buffer address outside the buffer
        while (1) {
          pt->buf[pt->bufaddr].c=pb[count];
          pt->buf[pt->bufaddr].fa=0;
          pt->buf[pt->bufaddr].ea=(uchar)ea;
          Incrementbufaddr(pt);
          if (pt->bufaddr==addr) break; };
        count++;
        pt->redraw=1; break;
      default:
        pt->buf[pt->bufaddr].c=pb[count++];
        pt->buf[pt->bufaddr].fa=0;
        pt->buf[pt->bufaddr].ea=(uchar)ea;
        Incrementbufaddr(pt);
        pt->redraw=1;
      break;
    };
    if (exitloop<0) {
      // Incomplete command.
      Unimplemented("Wait");
      count=0; break; }
    else if (exitloop>0) {
      // Error detected.
      Unimplemented("exitloop");
      count=-1;
      break;
    };
  };
  // Make postponed operations.
  if (count>0 && unlock!=0) {
    pt->kbdlock=0;
    pt->insertmode=0;
    pt->aid=AID_NO; };
  Setcursor(pt,pt->cursor);
  return count;
};

// Service function, processes Read Buffer command. Returns 0 on success and -1
// on error.
static int Readbuffer(t_3270 *pt) {
  int ndata,bufaddr,ea;
  uchar data[DATALEN];
  if (pt==NULL)
    return -1;                         // Error in input parameters
  ndata=0;
  data[ndata++]=(uchar)pt->aid;
  Setaddress(data+ndata,pt->cursor); ndata+=2;
  ea=0;
  // Add data. Note that in the SNA mode reading starts at buffer position 0.
  for (bufaddr=0; bufaddr<pt->rows*pt->columns; bufaddr++) {
    if (pt->buf[bufaddr].fa) {
      if (pt->replymode==REPLY_FIELD)
        data[ndata++]=ORD_SF;
      else {
        data[ndata++]=ORD_SFE;
        data[ndata++]=1;               // AND OTHER SFE TOO!!!!!!------======--=
        data[ndata++]=ATR_FIELD; };
      data[ndata++]=encodeebcdic[pt->buf[bufaddr].c & 0x3F]; }
    else {
      // If necessary, send attributes.
      if (pt->replymode==REPLY_CHAR && pt->buf[bufaddr].ea!=ea) {
        Unimplemented("REPLY_CHAR");
        ea=pt->buf[bufaddr].ea; };
      // Send character.
      data[ndata++]=pt->buf[bufaddr].c;
    };
  };
  Sendtelnet(&(pt->telnet),data,ndata);
  return 0;
};

// This function must be called periodically in order for the terminal to
// operate. Returns 1 if some actions were necessary and 0 otherwise.
int Terminalstep(t_3270 *pt) {
  int i,j,n,fa,fea,ea,fg,bg,addr,showfields,features,ret;
  char c,s[TEXTLEN];
  uchar buf[CMDSIZE],keys[256];
  t_ch *pb;
  if (pt==NULL)
    return 0;                          // Error in input parameters
  ret=0;
  // Process data from telnet stream, as many commands at once as possible.
  while (1) {
    n=Gettelnetdata(&(pt->telnet),buf,sizeof(buf));
    if (n<=0) break;                   // No data or invalid data
    switch (buf[0]) {
      case CMD_RMA:                    // Read modified all
        Readmodified(pt,1);
        break;
      case CMD_EAU:                    // Erase all unprotected
        Unimplemented("CMD_EAU");
        break;
      case CMD_EWA:                    // Erase/write alternative
      case CMD_W:                      // Write
      case CMD_EW:                     // Erase/write
        Writecmd(pt,buf,n);
        break;
      case CMD_RB:                     // Read buffer
        Readbuffer(pt);
        break;
      case CMD_WSF:                    // Write structured field
        Unimplemented("CMD_WSF");
        break;
      case CMD_RM:                     // Read modified
        Readmodified(pt,0);
      break;
    };
    ret=1;
  };
  // Process scroll lock state.
  if (GetKeyboardState(keys)!=0) {
    showfields=keys[VK_SCROLL] & 1;
    if (showfields!=pt->showfields) {
      pt->showfields=showfields;
      pt->redraw=1;
    };
  };
  // Check state of telnet connection.
  if (pt->reconnecting!=pt->telnet.reconnecting) {
    pt->reconnecting=pt->telnet.reconnecting;
    pt->redraw=1; };
  // If necessary, update list.
  if (pt->redraw) {
    fea=0; fg=bg=0;
    Setlistposition(pt->list,0,0);
    addr=Getfaaddress(pt,0);
    if (addr>=0)
      fa=pt->buf[addr].c;
    else
      fa=0;
    for (i=0,pb=pt->buf; i<pt->rows*pt->columns; i++,pb++) {
      // Get character and attributes.
      if (pb->fa) {
        fa=pb->c; fea=pb->ea; ea=0;
        c=(char)EBCDIC_SPACE; }
      else {
        ea=pb->ea;
        c=(char)pb->c; };
      // Determine colors.
      if ((ea & EA_COLOR)!=0) {
        fg=(ea & EA_COLOR)>>3;
        if ((fa & FA_INTMASK)==FA_BRIGHT) fg+=8; }
      else if ((fea & EA_COLOR)!=0) {
        fg=(fea & EA_COLOR)>>3;
        if ((fa & FA_INTMASK)==FA_BRIGHT) fg+=8; }
      else
        fg=((fa & FA_INTMASK)==FA_BRIGHT?15:4);
      if (pt->showfields!=0 && pb->fa==0 && (fa & FA_PROTECTED)==0)
        bg=16;                         // Highlight unprotected fields
      else
        bg=0;
      if ((ea & EA_MODEMASK)==EA_REVERSE ||
        ((ea & EA_MODEMASK)==EA_NORMAL && (fea & EA_MODEMASK)==EA_REVERSE)
      ) {
        j=fg; fg=bg; bg=j; }
      if ((fa & FA_INTMASK)==FA_FHIDDEN)
        fg=bg;                         // Hidden field
      // Determine extended features.
      features=0;
      if ((ea & EA_MODEMASK)==EA_UNDERLINE ||
        ((ea & EA_MODEMASK)==EA_NORMAL && (fea & EA_MODEMASK)==EA_UNDERLINE))
        features=TF_UNDERLINE;
      // Display character.
      Addtolist(pt->list,&c,1,fg,bg,features);
    };
    // Set cursor position.
    Setcursorposition(pt->list,pt->cursor%pt->columns,pt->cursor/pt->columns);
    // Display different data in the overline.
    memset(s,' ',pt->columns);
    if (pt->insertmode)
      memcpy(s+1,"INS",3);
    else
      memcpy(s+1,"OVR",3);
    if (pt->kbdlock)
      memcpy(s+5,"LOCK",4);
    if (pt->isprotected==1)
      memcpy(s+10,"PROT",4);
    else if (pt->isprotected==2)
      memcpy(s+10,"FA",2);
    if (pt->reconnecting)
      memcpy(s+15,"OFFLINE",7);
    sprintf(s+pt->columns-8,"%02i:%02i",
      pt->cursor/pt->columns+1,pt->cursor%pt->columns+1);
    for (i=0; i<pt->columns; i++)
      s[i]=ascii_to_ebcdic[s[i] & 0xFF];
    Addtolist(pt->list,s,pt->columns,17,16,0);
    pt->redraw=0;
    ret=1;
  };
  return ret;
};

// Frees resources associated with the terminal.
void Deleteterminal(t_3270 *pt) {
  if (pt==NULL)
    return;                            // Error in input parameters
  if (pt->list!=NULL)
    Destroylist(pt->list);
  Closetelnet(&(pt->telnet));
  GlobalFree((HGLOBAL)pt);
};

